package de.brightbyte.web.rip;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.ListResourceBundle;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import de.brightbyte.util.StringUtils;
import de.brightbyte.util.SystemUtils;

public abstract class RipServlet extends HttpServlet {
	
	public static final String parameterMapAttribute = "de.brightbyte.web.rip.RipServlet.queryParameter";

	private String homePage;
	private Map<String, RipPageHandler> pages = new HashMap<String, RipPageHandler>();
	private Map<String, RipTemplateEngine> engines = new HashMap<String, RipTemplateEngine>();
	private Map<String, RipActionHandler> actions = new HashMap<String, RipActionHandler>();
	
	protected String encoding;
	protected RipPageHandler errorPage;
	protected Pattern badPathes;  
	
	private Map<String, Object> applicationInfo = new HashMap<String, Object>();
	
	protected RipActionHandler defaultActionHandler;
	private String[] pagePackages;
	private String[] pageTemplatePathes;
	private ClassLoader pageClassLoader;
	
	protected static class SkinDef {
		public final Object template;
		public final RipTemplateEngine engine;
		public final String contentName;
		
		public SkinDef(final Object template, final RipTemplateEngine engine, final String contentName) {
			super();
			this.template = template;
			this.engine = engine;
			this.contentName = contentName;
		}
	}
	
	private Map<String, SkinDef> skins = new HashMap<String, SkinDef>();
	
	private boolean usePathPrefix;
	private boolean usePathSuffix;
	
	public static class PathInfo {
		public final String prefix;
		public final String name;
		public final String postfix;
		public final String extension;
		public final String path;
		
		public PathInfo(final String path, final String prefix, final String name, final String postfix, final String extension) {
			super();
			this.path = path;
			this.prefix = prefix;
			this.name = name;
			this.postfix = postfix;
			this.extension = extension;
		}

		@Override
		public int hashCode() {
			final int PRIME = 31;
			int result = 1;
			result = PRIME * result + ((extension == null) ? 0 : extension.hashCode());
			result = PRIME * result + ((name == null) ? 0 : name.hashCode());
			result = PRIME * result + ((postfix == null) ? 0 : postfix.hashCode());
			result = PRIME * result + ((prefix == null) ? 0 : prefix.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final PathInfo other = (PathInfo) obj;
			if (extension == null) {
				if (other.extension != null)
					return false;
			} else if (!extension.equals(other.extension))
				return false;
			if (name == null) {
				if (other.name != null)
					return false;
			} else if (!name.equals(other.name))
				return false;
			if (postfix == null) {
				if (other.postfix != null)
					return false;
			} else if (!postfix.equals(other.postfix))
				return false;
			if (prefix == null) {
				if (other.prefix != null)
					return false;
			} else if (!prefix.equals(other.prefix))
				return false;
			return true;
		}
		
		public String toString() {
			String s = name;
			if (prefix!=null) s = prefix + "/" + s;
			if (postfix!=null) s = s + "/" + postfix;
			if (extension!=null) s = s + "." + extension;
			return s;
		}
	}

	private Map<Locale, ResourceBundle> messages = new HashMap<Locale, ResourceBundle>();
	private ResourceBundle NO_BUNDLE = new ListResourceBundle() {
		@Override
		protected Object[][] getContents() {
			return null;
		}
	};
	
	protected MimeInfo mime = new MimeInfo();
	protected Context jndiRoot;

	public RipServlet() {
		Class c = getClass();
		
		pageClassLoader = c.getClassLoader();
		pageTemplatePathes = new String[] { "/WEB-INF/templates" };

		List<String> pkgs = new ArrayList<String>();
		while (c!=null) {
			pkgs.add( c.getPackage().getName() );
			c = c.getSuperclass();
		}
		
		pagePackages = (String[]) pkgs.toArray(new String[pkgs.size()]);
	}
	
	protected Object jndiLookup(String name) throws NamingException {
		try {
			return jndiRoot.lookup(name);
		} catch (NameNotFoundException e) {
			return null;
		}
	}
	
	public void setSkin(String type, RipTemplateEngine engine, Object template, String contentName) {
		skins.put(type, new SkinDef(template, engine, contentName));
	}
	
	public void setSkin(String type, String skin, String contentName) throws IOException, RipException {
		URL u = getTemplateURL(skin, null, null, type); 
		if (u==null) throw new RipException("Skin file not found: "+skin);

		RipTemplateEngine engine = guessEngine(u);
		Object template = engine.loadTemplate("Skin", u);
		
		setSkin(type, engine, template, contentName);
	}

	//private int maxStaticPageSize = 256 * 1024;
	
	public Object loadTemplate(String name, URL url) throws RipException, IOException {
		RipTemplateEngine engine = guessEngine(url);
		Object template = engine.loadTemplate(name, url);
		return template;
	}

	public TemplatedPage loadTemplatedPage(String name, URL url) throws RipException, IOException {
		RipTemplateEngine engine = guessEngine(url);
		Object template = engine.loadTemplate(name, url);
		return new TemplatedPage(engine, template);
	}
	
	@Override
	public void init() throws ServletException {
		try {
			jndiRoot = new InitialContext();
			
			ServletContext ctx = getServletContext();
			
			URL u = ctx.getResource("/WEB-INF/mime.types");
			if (u!=null) mime.load(u);
			
			u = ctx.getResource("/WEB-INF/mime.properties");
			if (u!=null) mime.load(u);
			
			Class c = getClass();
			while (c!=null) {
				u = c.getResource("mime.types");
				if (u!=null) mime.load(u);
				
				u = c.getResource("mime.properties");
				if (u!=null) mime.load(u);

				c = c.getSuperclass();
			}
			
			initEngines();

			homePage = getInitParameter("wikiword.home", "Home");

			encoding = getInitParameter("wikiword.encoding", "utf-8"); //NOTE: don't use local default, files come from WAR!

			badPathes = Pattern.compile(getInitParameter("wikiword.badPathPattern", "(^|/)\\.\\.($|/)|^/?[A-Z]+-INF($|/)|^/?\\."));
			
			Enumeration names = ctx.getInitParameterNames();
			while (names.hasMoreElements()) {
				String name = (String)names.nextElement();
				if (name.startsWith("wikiword.skin-template.")) {
					String type = name.substring("wikiword.skin-template.".length());
					String skinTemplateFile = getInitParameter("wikiword.skin-template."+type, null);
					if (skinTemplateFile!=null) {
						String contentName = getInitParameter("wikiword.skin-content-name."+type, "content");
						setSkin(type, skinTemplateFile, contentName);
					}
				}
			}
			
			
			String errorTemplateFile = getInitParameter("wikiword.error-template", "error");			
			URL errorURL = getTemplateURL(errorTemplateFile, "page", null, "html");
			if (errorURL==null) throw new ServletException("Error file not found: "+errorTemplateFile);
			
			RipTemplateEngine errorEngine = guessEngine(errorURL);
			
			Object errorTemplate = errorEngine.loadTemplate("Error", errorURL);
			errorPage = new ErrorPage(errorEngine, errorTemplate);
			errorPage.attach(this, "*error*");

			names = ctx.getInitParameterNames();
			while (names.hasMoreElements()) {
				String name = (String)names.nextElement();
				applyConfig(name, ctx.getInitParameter(name));
			}
			
			ServletConfig cfg = getServletConfig();
			names = cfg.getInitParameterNames();
			while (names.hasMoreElements()) {
				String name = (String)names.nextElement();
				applyConfig(name, cfg.getInitParameter(name));
			}
			
			for (RipTemplateEngine e: engines.values()) {
				e.putSharedValue("application", applicationInfo);
				e.putSharedValue("servlet", this);
			}
			
			initApp();			
			initPages();
			initActions();
		} catch (Exception e) {
			throw new ServletException(e);
		} 		
		
		//TODO: show pretty error page on exception!
	}
	
	protected static final Pattern extensionPattern = Pattern.compile("\\.([\\-\\w\\d]+)(\\.([\\-\\w\\d]+))?$");
	
	protected RipTemplateEngine getEngine(String templateFormat, String outputFormat) throws RipException {
		RipTemplateEngine engine = null;
		if (outputFormat!=null) engine = engines.get(templateFormat+"."+outputFormat);
		if (engine==null) engine = engines.get(templateFormat);
		if (engine==null) throw new RipException("no engine for "+templateFormat+" -> "+outputFormat);
		return engine;
	}
	
	protected RipTemplateEngine guessEngine(URL u) throws RipException {
		return guessEngine(u.getPath());
	}
	
	protected RipTemplateEngine guessEngine(String path) throws RipException {
		Matcher m = extensionPattern.matcher(path);
		if (m.find()) {
			String ext;
			RipTemplateEngine engine = null;
			
			if (m.group(3)!=null) {
				ext = m.group(1) + "." + m.group(3); 
				engine = engines.get(ext);
			}
			
			if (engine==null) {
				ext = m.group(1).toLowerCase();
				engine = engines.get(ext);
			}

			if (engine==null) {
				if (m.group(3)!=null) {
					ext = m.group(3).toLowerCase();
					engine = engines.get(ext);
					if (engine==null) throw new RipException("no engine known for extension "+m.group(1)+"."+m.group(3)+" or "+m.group(1)+" or "+m.group(3));
				}
				else throw new RipException("no engine known for extension "+m.group(1));
			}			
			
			return engine;
		}
		else throw new IllegalArgumentException("no file extension found in path "+path);
	}

	protected void initApp() throws Exception {
		//noop
	}
	
	protected void initEngines() throws Exception {
		YatesEngine yates = new YatesEngine(this, "html");
		//addEngine("yt", yates);
		addEngine("yt", "html", yates);
		addEngine("yt", "htmlet", yates); //for snippets. not skinned.
		
		if (SystemUtils.isClassKnown("freemarker.template.Template", RipServlet.class)) {
			log("FreeMarker found, installing ftl support");
			addEngine("ftl", null, new FreeMarkerEngine(this));
		}
		else {
			log("FreeMarker not available, ftl support disabled");
		}

		//TODO: velocity engine

		//TODO: XSLT engine
	}
	
	protected void addEngine(String templateFormat, String outputFormat, RipTemplateEngine engine) throws ServletException, IOException, RipException {
		String name = templateFormat;
		if (outputFormat!=null) name += "." + outputFormat;
		engines.put(name, engine);
	}

	protected void initPages() throws Exception {
		//noop
	}

	protected void initActions() throws Exception {
		//noop
	}

	protected void addPage(String key, String name, RipPageHandler page) {
		page.attach(this, name);
		pages.put(key, page);
	}

	protected void addAction(String name, RipActionHandler action) {
		actions.put(name, action);
	}
	
	protected void setApplicationInfo(String name, Object value) {
		applicationInfo.put(name, value);
	}

	protected void applyConfig(String name, String value) throws RipException {
		int idx = name.indexOf('.');
		if (idx<0) return;
		
		String scope = name.substring(0, idx);
		name = name.substring(idx+1);

		applyConfig(scope, name, value);
	}

	protected void applyConfig(String scope, String name, String value) throws RipException {
		RipTemplateEngine engine = engines.get(scope);
		if (engine!=null) engine.putSetting(name, value); //XXX: doc!
		else if (scope.equals("app")) applicationInfo.put(name, value); //XXX: doc!
	}

	protected String getInitParameter(String name, String def) {
		//NOTE: servlet context takes precedence over servlet ini param.
		//      that's because context config is local, and web.xml comes from the war file.
		String v =  getServletContext().getInitParameter(name);
		if (v==null) v = getServletConfig().getInitParameter(name);
		if (v==null) v = def;
		return v;
	}

	@Override
	protected void doGet(HttpServletRequest rq, HttpServletResponse rp) throws ServletException, IOException {
		handleRequest(rq, rp);
	}
	
	protected RipRequest makeRequest(HttpServletRequest rq, HttpServletResponse rp) {
		RipRequest request = new RipRequest(this, rq, rp);
		return request;
	}
	
	protected void handleRequest(HttpServletRequest rq, HttpServletResponse rp) throws ServletException, IOException {
		//force encoding if not specified!
		String rqenc = rq.getCharacterEncoding();
		if (rqenc==null) rq.setCharacterEncoding(encoding);
		
		//apply default
		rp.setCharacterEncoding(encoding);

		RipRequest request = makeRequest(rq, rp);
		
		try {
			request.process();
		} catch (Exception e) {
			//TODO: show pretty error page!
			if (e instanceof RuntimeException) throw (RuntimeException)e;
			if (e instanceof IOException) throw (IOException)e;
			if (e instanceof ServletException) throw (ServletException)e;
			throw new ServletException(e);
		} finally {
			try {
				request.destroy();
			} catch (Exception e) {
				//TODO: show pretty error page!
				if (e instanceof RuntimeException) throw (RuntimeException)e;
				if (e instanceof IOException) throw (IOException)e;
				if (e instanceof ServletException) throw (ServletException)e;
				throw new ServletException(e);
			}
		}
	}

	/*
	protected Map<String, String> getRequestParameters(HttpServletRequest rq) {
		Map<String, String> m = new HashMap<String, String>();
		for(Object e: rq.getParameterMap().entrySet()) {
			Map.Entry<String, String[]> entry = (Map.Entry<String, String[]>)e;
			String n = entry.getKey();
			String[] v = entry.getValue();
			
			m.put(n, v.length==0?null:v[0]);
		}
		
		return m;
	}
	*/
	
	public URL getTemplateURL(String path, String kind, String format, String ext) throws IOException {
		//TODO: use StringBuilder to make this more efficient!
		
		String[] formats;
		if (format!=null) formats = new String[] { ext == null ? format : format + "." + ext };
		else formats = getTemplateExtensions(ext);
		
		//look for page template in context
		ServletContext ctx = getServletContext();
		for (String p: pageTemplatePathes) {
			if (!p.startsWith("/")) p = "/" + p;
			p = p+"/"+path.replace('/', '_')+".";
			if (kind!=null ) p += kind+".";
			
			for (String frm: formats) {
				String n = p+frm;
				
				URL u = ctx.getResource(n);
				if (u!=null) return u;
			}
		}
		
		//look for page template in classpath
		for (String pkg: pagePackages) {
			String p = pkg.replace('.', '/'); 
			p = p+ "/" + path.replace('/', '_')+"."; 
			if (kind!=null ) p += kind+".";
			
			for (String frm: formats) {
				String n = p+frm;
				
				URL u = pageClassLoader.getResource(n);
				if (u!=null) return u;
			}
		}
		
		return null;
	}

	private String[] getTemplateExtensions(String ext) {
		List<String> extensions = new ArrayList<String>();
		
		for (String e: engines.keySet()) {
			
			if (ext==null) extensions.add(e);
			else {
				int idx = e.indexOf('.');
		
				if (idx>=0) {
					if (e.substring(idx+1).equals(ext)) {
						extensions.add(e);
					}
				}
				else extensions.add(e+"."+ext);
			}
		}
		
		return (String[]) extensions.toArray(new String[extensions.size()]);
	}

	public RipPageHandler getPage(PathInfo info, String[] types) throws RipException, IOException {
		String key = info.name;
		
		//explicitely defined pages (and cached page objects)
		RipPageHandler page = pages.get(key);

		if (info.extension!=null) {
			key += "." + info.extension;
		
			//cached pages (and cached page objects)
			page = pages.get(key);
		}
		
		if (page!=null) return page;
		
		List<String> ext = new ArrayList<String>();
		if (info.extension!=null) ext.add(info.extension);
		else {
			for (String t: types) {
				String e = mime.guessExtensionForType(t);
				if (e!=null) ext.add(e);
			}
		}

		page = getProgrammedPage(info.name);
		
		if (page==null) {
			for (String e: ext) {
				page = getTemplatePage(info.name, e);
				if (page!=null) break;
			}
		}
		
		if (page==null) {
			for (String e: ext) {
				page = getStaticPage(info.name, e);
				if (page!=null) break;
			}
		}
		
		if (page!=null) {
			log("registered page "+key+": "+page);
			addPage(key, info.name, page); //cache for future use.
		}
		
		return page;
	}
	
	private RipPageHandler getStaticPage(String name, String ext) throws RipException, IOException {
		if (ext!=null) name += "." + ext;
		
		//evil path
		if (badPathes.matcher(name).matches()) return null;

		//find static file (simulate default servlet)
		URL u = getServletContext().getResource("/"+name);
		RipPageHandler page;
		
		if (u!=null) {
			page = new StaticPage(u);

			log("registered page "+name+": using static resource from "+u);
			return page;
		}
		
		//TODO: put miss marker into cache!
		return null;
		
	}
	
	private RipPageHandler getTemplatePage(String name, String ext) throws RipException, IOException {
		RipPageHandler page;

		//look for page template 
		URL u = getTemplateURL(name, "page", null, ext);
		if (u!=null) {
			RipTemplateEngine engine = guessEngine(u);
			Object template = engine.loadTemplate(name, u);
			
			page = makeTemplatedPage(engine, template, true, ext); //TODO: check if skinnable!

			log("registered page "+name+": using template at "+u);
			return page;
		}
				
		//TODO: put miss marker into cache!
		return null;
	}

	private RipPageHandler getProgrammedPage(String name) throws RipException, IOException {
		RipPageHandler page;

		//look for page class
		String cn = StringUtils.firstToUppercase(name.replace('/', '_')); //XXX: replace "/" by "."? or "$"? try each? frobidden? only last element?
		for (String pkg: pagePackages) {
			String n = pkg + "." + cn;
			
			try {
				Class<? extends RipPageHandler> cls = (Class<? extends RipPageHandler>)pageClassLoader.loadClass(n);
				int mod = cls.getModifiers();
				if (!RipPageHandler.class.isAssignableFrom(cls)) continue; 
				if (!Modifier.isPublic(mod) || Modifier.isAbstract(mod)) continue;
				if (cls.getAnnotation(RipPage.class)==null) continue; 
					
				page = (RipPageHandler)cls.newInstance();

				log("registered page "+name+": using instance of "+cls);
				return page;
			} catch (ClassNotFoundException ex) {
				//ignore, carry on
			} catch (InstantiationException e) {
				throw new RipException("Failed to instantiate page class "+n, e);
			} catch (IllegalAccessException e) {
				throw new RipException("Failed to instantiate page class "+n, e);
			}
		}
				
		//TODO: put miss marker into cache!
		return null;
	}

	/*
	
	protected RipPage makeFreeMarkerPage(String path) throws IOException {
		return new TemplatedPage(this, path, loadTemplate(path+".ftl"));
	}

	protected String loadTemplate(String name) throws IOException {
		URL u = getTemplateURL(name);
		if (u==null) throw new FileNotFoundException("template not found in servlet context or classpath: "+name);
		
		return IOUtil.slurp(u, getEncoding());
	}
	*/
	/*
	public TemplatedPage makeTemplatedPage(String pageName, String templatePath, boolean skinnable) throws IOException, RipException {
		RipTemplateEngine engine = guessEngine(templatePath);
		Object template = engine.loadTemplate(pageName, templatePath);
		return makeTemplatedPage(engine, template, skinnable);
	}
	*/
	public <T>TemplatedPage<T> makeTemplatedPage(RipTemplateEngine<T> engine, T template, boolean skinnable, String type) throws IOException, RipException {
		TemplatedPage<T> page;
		
		SkinDef skin = type == null ? null : skins.get(type);
		
		if (skin!=null) { //FIXME: engine must match skinTemplate!
			if (engine!=skin.engine) throw new IllegalArgumentException("template (endinge "+engine.getClass()+") is incomaptible with skin type (engine "+skin.engine.getClass()+"): ");
			page = new TemplatedPage<T>(engine, (T)skin.template);
			page.setPageOption(skin.contentName, template); 
		}
		else {
			page = new TemplatedPage<T>(engine, template);
		}
		
		if (type!=null) {
			type = mime.guessTypeForExtension(type);
			if (type!=null) page.setContentType(type);
		}
		
		return page;
	}
	
	/*
	public Object loadTemplate(String name, String path) throws IOException, RipException {
		RipTemplateEngine engine = guessEngine(path);
		Object template = engine.loadTemplate(name, path);
		return template;
	}
	*/
	
	public String getEncoding() {
		return encoding;
	}

	public Map<String, Object> getApplicationInfo() {
		return applicationInfo ;
	}

	public String getHomePage() {
		return homePage;
	}

	public void setHomePage(String homePage) {
		this.homePage = homePage;
	}

	public RipActionHandler getActionHandler(String action) {
		return actions.get(action);
	}

	public RipPageHandler getErrorPage() {
		return errorPage;
	}

	public RipActionHandler getDefaultActionHandler() {
		return defaultActionHandler;
	}

	public String guessContentType(URL u) {
		String n = u.getPath();
		String t = mime.guessTypeForName(n) ; 
		if (t==null) t = getServletContext().getMimeType(n);
		return t;
	}

	protected String getLocalizationBase() {
		return (String)applicationInfo.get("localizationBase");
	}

	protected ResourceBundle getMessages(Locale locale, String path) {
		ResourceBundle r = messages.get(locale);
		if (r!=null) return r==NO_BUNDLE ? null : r;
		
		try {
			String baseName = getLocalizationBase();
			if (baseName==null) baseName = getClass().getName(); 
			r = ResourceBundle.getBundle(baseName, locale);
		} catch (MissingResourceException e) {
			r = NO_BUNDLE;
		}
		
		messages.put(locale, r);
		return r==NO_BUNDLE ? null : r;
	}
	
	protected String getMessage(Locale locale, String path, String name) {
		ResourceBundle r = getMessages(locale, path);
		if (r==null) return "**"+name+"**";
		
		if (path!=null) {
			try {
				return r.getString(path+"/"+name);
			} catch (MissingResourceException e) {
				//ignore
			}
		}

		try {
			return r.getString(name);
		} catch (MissingResourceException e) {
			return "**"+name+"**";
		}
	}

	public boolean getUsePathPrefix() {
		return usePathPrefix;
	}

	public void setUsePathPrefix(boolean usePathPrefix) {
		this.usePathPrefix = usePathPrefix;
	}

	public boolean getUsePathSuffix() {
		return usePathSuffix;
	}

	public void setUsePathSuffix(boolean usePathSuffix) {
		this.usePathSuffix = usePathSuffix;
	}
	
	
	//TODO: support jnda-based database connections, etc!
}
