package de.brightbyte.web.rip;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import de.brightbyte.net.UrlUtils;
import de.brightbyte.util.CollectionUtils;

public class RipRequest {
	protected RipServlet servlet;
	protected Locale locale;

	protected HttpServletRequest request;
	protected HttpServletResponse response;
	
	protected RipPageHandler page;
	protected RipServlet.PathInfo pathInfo;
	//protected String pageName;
	
	protected Map<String, String[]> parameters;

	public RipRequest(RipServlet servlet, HttpServletRequest request, HttpServletResponse response) {
		super();
		this.servlet = servlet;
		this.request = request;
		this.response = response;

		//NOTE: decode query string manually, using the correct encoding
		//      the servlet spec is stupid about this
		String q = request.getQueryString();
		if (q==null || q.length()==0) parameters = Collections.emptyMap();
		else parameters = UrlUtils.splitParameters(q);
		
		request.setAttribute(RipServlet.parameterMapAttribute, parameters); //TODO: doc this hack!!
		
		locale = getLanguage(request);
	}
	
	public void process() throws Exception {
		String pagePath = getPagePath();
		if (handleRedirect(pagePath)) return;
		pathInfo = parsePath(pagePath, getParameter("as", null)); //XXX: doc
		
		page = getPage(pathInfo, getAcceptedContentTypes());
		String action = getRequestedAction();
		
		if (action==null && page!=null) {
			//no action explicitly requested, so ask the page
			action = page.getDefaultAction();
		}
		
		Map<String, Object> pageData = getDefaultPageData();
		if (action!=null) {
			//action changes data, may change page
			//may also set headers, and even commit response (redirect, etc)
			performAction(action, pageData); 
		}
		
		if (response.isCommitted()) {
			return; //already committed, abort. no forther output possible/wanted
		}
		
		if (page==null) { //page not found by servlet.getPage(path), and page not set by action
			page = servlet.getErrorPage();
			pageData.put("error", newErrorModel(404, "Not Found", "Path "+pagePath));
		}
		
		page.process(pageData, response);
	}

	protected static final Pattern extensionPattern = Pattern.compile("\\.(\\w[\\w\\d]*)$");
	
	protected RipServlet.PathInfo parsePath(String path, String ext) {
		String p = path;
		while (p.startsWith("/")) p = p.substring(1);
		while (p.endsWith("/")) p = p.substring(0, p.length()-1);
		if (p.length()==0) return new RipServlet.PathInfo(path, null, "", null, null);
		
		Matcher m = extensionPattern.matcher(p);
		if (m.find()) {
			if (ext==null) ext = m.group(1);
			p = p.substring(0, m.start());
		}
		
		String prefix = null;
		String suffix = null;
		String name = p;
		
		if (servlet.getUsePathPrefix()) {
			int idx = name.indexOf('/');
			if (idx>0) {
				prefix = name.substring(0, idx);
				name = name.substring(idx+1);
			}
			else prefix = "";
		}

		if (servlet.getUsePathSuffix()) {
			int idx = name.indexOf('/');
			if (idx>0) {
				suffix = name.substring(idx+1);
				name = name.substring(0, idx);
			}
			else suffix = "";
		}
		
		return new RipServlet.PathInfo( path, prefix, name, suffix, ext );
	}

	protected RipPageHandler getPage(RipServlet.PathInfo path, String[] contentTypes) throws RipException, IOException {
		return servlet.getPage(path, contentTypes);
	}
	
	
	public String[] getAcceptedContentTypes() {
		return getPreferenceListFromHeader("Accept");
	}

	public String[] getPreferenceListFromHeader(String name) {
		Enumeration<String> en = request.getHeaders(name);
		return getPreferenceListFromEnum(en);
	}
	
	public String[] getPreferenceListFromEnum(Enumeration<String> en) {
		class Entry implements Comparable<Entry> {
			int index;
			double pref;
			String value;
			
			public Entry(int index, double pref, String value) {
				super();
				this.index = index;
				this.pref = pref;
				this.value = value;
			}

			public int compareTo(Entry e) {
				if (pref == e.pref) {
					return index - e.index;
				}
				else if (pref<e.pref) return (int)(e.pref - pref) +1;
				else return (int)(e.pref - pref) -1;
			}
		}
		
		List<Entry> h = new ArrayList<Entry>();
		
		int i = 1;		
		while (en.hasMoreElements()) {
			String[] ss = en.nextElement().split("\\s*,\\s*");
			for (String n: ss) {
				double q = 1;
				int idx = n.indexOf(";q=");
				if (idx>=0) {
					q = Double.parseDouble(n.substring(idx+3));
					n = n.substring(0, idx);
				}
				
				h.add( new Entry(i++, q, n) );
			}
		}
		
		Collections.sort(h);
		
		i = 0;
		String[] v = new String[h.size()];
		for (Entry e: h) {
			v[i++] = e.value;
		}
		
		return v;
	}

	public void performAction(String action, Map<String, Object> pageData) throws Exception {
		RipActionHandler handler = null;
		handler = getActionHandler(action);
		
		if (handler == null) { //NOTE: do nothing. register a default handler if desired.
			return;
		}
			
		beforeAction(pageData);
		try {
			handler.perform(this, pageData);
		}
		finally {
			afterAction(pageData);
		}
	}

	protected static class MethodAction implements RipActionHandler {
		protected Method method;
		
		public MethodAction(Method method) {
			super();
			this.method = method;
		}

		@SuppressWarnings("unchecked")
		public void perform(RipRequest handler, Map<String, Object> data) throws Exception {
			try {
				Object obj;
				if (method.getParameterTypes().length == 0) {
					obj = method.invoke(handler);
				}
				else {
					obj = method.invoke(handler, data);
				}

				if (obj != null && obj!=data) {
					if (obj instanceof Map) {
						CollectionUtils.deepMerge(data, ((Map)obj)); //XXX: really merge?
					}
					else {
						data.put("result", obj);
					}
				}
			}
			catch (InvocationTargetException ex) {
				Throwable exx = ex.getCause();
				if (exx instanceof Exception) throw (Exception)exx;
				if (exx instanceof Error) throw (Error)exx;
				throw new Error("Odd Throwable", exx);
			}
		}
	}
	
	protected RipActionHandler getActionHandler(String action) {
		if (action==null) return servlet.getDefaultActionHandler();
		
		RipActionHandler a = servlet.getActionHandler(action);
		if (a!=null) return a;
		
		Class c = getClass();
		a = getMethodAction(action, c);
		
		if (a==null) a = servlet.getDefaultActionHandler();
		return a;
	}
	
	public static RipActionHandler getMethodAction(String name, Class cls) {
		try {
			Method m;
			
			try {
				m = cls.getMethod(name);
			}
			catch (NoSuchMethodException ex) {
				m = cls.getMethod(name, Map.class);
			}
			
			RipAction annot = m.getAnnotation(RipAction.class);
			
			if (annot!=null) {
				return new MethodAction(m);
			}
		}
		catch (NoSuchMethodException ex) {
			//carry on
		}
		
		return null;
	}

	protected void beforeAction(Map<String, Object> pageData) throws Exception{
		//noop
	}

	protected void afterAction(Map<String, Object> pageData) throws Exception {
		//noop
	}
	
	protected String getMessage(String name) {
		return servlet.getMessage(locale, pathInfo.name, name);
	}

	protected ResourceBundle getMessages() {
		return servlet.getMessages(locale, pathInfo.name);
	}

	protected Map<String, Object> getDefaultPageData() throws Exception {
		ResourceBundle msg = servlet.getMessages(locale, pathInfo==null ? null : pathInfo.name);
		String title = getMessage("title");
		if (title.equals("**title**")) title = pathInfo==null ? null : pathInfo.name; //HACK
		
		Map<String, Object> pageData = new HashMap<String, Object>();
		if (pathInfo!=null) pageData.put("self", getFullPath(pathInfo.name) );
		pageData.put("basepath", getFullPath(""));
		pageData.put("request", request);
		pageData.put("handler", this);
		pageData.put("locale", locale);
		pageData.put("language", locale.getLanguage()+"-"+locale.getCountry());
		pageData.put("messages", msg);
		pageData.put("title", title);

		pageData.put("settings.locale", locale); //XXX: for internal use by Yates. arcane knowledge. also, doesn't work (yet)
		
		initDefaultPageData(pageData);
		
		return pageData;
	}

	protected void initDefaultPageData(Map<String, Object> pageData) throws Exception {
		// noop
	}

	public String getFullPath() {
		return getFullPath(pathInfo.path);
	}

	protected String getFullPath(String path) {
		if (path.startsWith("/")) path = path.substring(1);
		return request.getContextPath()+"/"+path;
	}

	protected boolean handleRedirect(String path) throws IOException {
		if (path==null) return true; //redirect supplied elsewhere
		
		if (path.length()>0 && path.charAt(0)=='@') {
			String p = path.substring(1);
			if (p.startsWith("/")) p = p.substring(1);
			p = request.getContextPath()+"/"+p;
			response.sendRedirect(p);
			return true;
		}
		
		return false;
	}

	protected Object newErrorModel(int code, String reason, String text) throws ServletException, IOException {
		Map<String, Object> e = new HashMap<String, Object>();
		e.put("code", code);
		e.put("reason", reason);
		e.put("text", text);
		return e;
	}	

	public void destroy() throws Exception {
		//noop
	}
	
	public Map<String, String[]> getParameterMap() {
		return parameters;
	}

	public Locale getLocale() {
		return locale;
	}

	public RipPageHandler getPage() {
		return page;
	}

	public String getPageName() {
		return pathInfo.name;
	}

	public String getPath() {
		return pathInfo.path;
	}

	public HttpServletRequest getRequest() {
		return request;
	}

	public HttpServletResponse getResponse() {
		return response;
	}

	public RipServlet getServlet() {
		return servlet;
	}	
	
	protected Locale getLanguage(HttpServletRequest rq) {
		return rq.getLocale();
	}

	protected String getRequestedAction() throws ServletException, IOException, RipException {
		String action = getParameter("do", null); //XXX: doc!
		if (action!=null) return action;
		
		for(Object k: parameters.keySet()) {
			String n = (String)k;
			if (n.startsWith("do-") || n.startsWith("do_")) { //XXX: doc!
				action = n.substring(3);
				break;
			}
		}
		
		return action;
	}
	
	public String buildMangeledPath(String path, String[] removeParams, Map<String, String[]> setParams) {
		try {
			if (path==null) path = getRequestedPath(request);
			else if (!path.startsWith("/")) path = "/" + path;
			
			StringBuilder s = new StringBuilder();
			s.append(path);
			
			Map<String, String[]> p = new HashMap<String, String[]>(parameters);
			if (removeParams!=null) {
				for (String n: removeParams) {
					p.remove(n);
				}
			}

			if (setParams!=null) {
				p.putAll(setParams);
			}
			
			boolean first = true;
			for(Map.Entry<String, String[]> e: p.entrySet()) {
				if (first) {
					s.append('?');
					first = false;
				}
				else {
					s.append('&');
				}
				
				String k = e.getKey();
				String[] vv = e.getValue();
				
				for (String v: vv) {					
					s.append(URLEncoder.encode(k, getAppEncoding()));
					s.append('=');
					s.append(URLEncoder.encode(v, getAppEncoding()));
				}
			}
			
			return s.toString();
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("unexpected exception", e);
		}
	}

	public String getAppEncoding() {
		return servlet.getEncoding();
	}

	public int getIntParameter(String name, int def) {
		String v = getParameter(name, null);
		if (v==null || v.length()==0) return def;
		
		try {
			return Integer.parseInt(v);
		}
		catch (NumberFormatException ex) {
			//XXX: perhaps throwand trigger a 400? or 500? 
			return def;
		}
	}
	
	public String getParameter(String name, String def) {
		Map<String, String[]> m = getParameterMap();
		String[] a = m == null ? null : (String[])m.get(name); 
		return a==null || a.length == 0 ? def : a[0];
	}
	
	protected String getPagePath() {
		//XXX: doc: homepage, @ for redirect!
		
		//look for requested redirect
		String go = null;
		String p = getParameter("go", null); //XXX: doc!
		if (p!=null) go = "go";
		else {
			for(Object k: request.getParameterMap().keySet()) {
				String n = (String)k;
				if (n.startsWith("go-") || n.startsWith("go_")) { //XXX: doc!
					p = n.substring(3);
					go = n;
					break;
				}
			}
		}
		
		//if redirect was requested, build target path and return
		if (p!=null) {
			p = buildMangeledPath(p, new String[] { go }, null);
			return "@" + p;
		}

		//get requested path
		p = getRequestedPath(request);
		
		//redirect to home page if / was requested
		if (p.equals("") || p.equals("/") || p.equals("//")) {
			p = "@" + servlet.getHomePage(); //XXX: doc: homepage, @ for redirect!
		}
		
		return p;
	}
	
	protected String getRequestedPath(HttpServletRequest rq) {
		//check if / was requested (special case to bypass welcome-page cruft)
		String ru = rq.getRequestURI(); //FIXME: handle URL-encoding!
		if (ru.equals(rq.getContextPath()) || ru.equals(rq.getContextPath()+"/")) {
			return "/"; 
		}
		
		String sp = rq.getServletPath(); //FIXME: for some reason, this returns "/index.html" if "/" is requested :/
		String pi = rq.getPathInfo();
		String p = (sp==null?"":sp) + (pi==null?"":pi);
		return p;
	}
	
	
	
}
