package de.brightbyte.util;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

/**
 * Static system level utility functions, that is, things missing from java.lang.System and
 * related classes. 
 */
public class SystemUtils {

	/**
	 * Returns the given system property, or the def value if that property
	 * was not defined, or if accessing it is prevented by the security manager.
	 * Unlike System.getProperty, this method never throws a SecurityException.
	 * 
	 * @param name the name of the property to retrieve. Must not be null.
	 * @param def the value to return if the property was not defined or is not accessible. May be null.
	 * @return the value of the named system property, or the value of def if the property was not defined or is not accessible
	 */
	public static String getPropertySafely(String name, String def) {
		try {
			return System.getProperty(name, def);
		} catch (SecurityException e) {
			return def;
		}
	}
	
	/**
	 * Returns a new instance of java.util.Properties, filled with the values loaded
	 * from the given URL, and using the given parent properties for fallback.
	 * 
	 * @param url the URL to load the properties from. Must not be null.
	 * @param parent a Properties instance to use for fallback. May be null.
	 * @return a new instance of java.util.Properties, filled with the values loaded
	 * from the given URL, and using the given parent properties for fallback.
	 * @throws IOException if an error occurred while loading the properties from the given URL.
	 */
	public static Properties loadProperties(URL url, Properties parent) throws IOException {
		InputStream in = url.openStream();
		Properties p = loadProperties(in, parent);
		in.close();
		return p;
	}
	
	/**
	 * Returns a new instance of java.util.Properties, filled with the values loaded
	 * from the given URL, and using the given parent properties for fallback.
	 * 
	 * @param f the File to load the properties from. Must not be null.
	 * @param parent a Properties instance to use for fallback. May be null.
	 * @return a new instance of java.util.Properties, filled with the values loaded
	 * from the given URL, and using the given parent properties for fallback.
	 * @throws IOException if an error occurred while loading the properties from the given URL.
	 */
	public static Properties loadProperties(File f, Properties parent) throws IOException {
		InputStream in = new BufferedInputStream(new FileInputStream(f));
		Properties p = loadProperties(in, parent);
		in.close();
		return p;
	}
	
	/**
	 * Returns a new instance of java.util.Properties, filled with the values loaded
	 * from the given InputStream, and using the given parent properties for fallback.
	 * 
	 * @param in the InputStream to load the properties from. Must not be null.
	 * @param parent a Properties instance to use for fallback. May be null.
	 * @return a new instance of java.util.Properties, filled with the values loaded
	 * from the given InputStream, and using the given parent properties for fallback.
	 * @throws IOException if an error occurred while loading the properties from the given InputStream.
	 */
	public static Properties loadProperties(InputStream in, Properties parent) throws IOException {		
		Properties p = new Properties(parent);
		p.load(in);
		return p;
	}
	
	/**
	 * Returns true if the class with the given name can be loaded in the context
	 * of the class described by the ctx parameter. This method causes the named class
	 * to be loaded, and possibly to be initialized.
	 * 
	 * @param name the name of the class to check
	 * @param ctx the context to use for checking - ctx.getClassLoader() will be used to determine
	 *        the ClassLoader to use for loading the class. If ctx is null, the system class loader
	 *        will be used.
	 *        
	 * @return true if the named class can be loaded in the given context.
	 */
	public static boolean isClassKnown(String name, Class ctx) {
		return isClassKnown(name, ctx==null ? ClassLoader.getSystemClassLoader() : ctx.getClassLoader());
	}
	
	/**
	 * Returns true if the class with the given name can be loaded using the given
	 * ClassLoader. This method causes the named class to be loaded, and possibly to be initialized.
	 * 
	 * @param name the name of the class to check
	 * @param loader the class loader to use to load the class.
	 *        
	 * @return true if the named class can be loaded in the given context.
	 */
	public static boolean isClassKnown(String name, ClassLoader loader) {
		try {
			loader.loadClass(name);
			return true;
		} catch (ClassNotFoundException e) {
			//ignore
		}
		
		return false;
	}
}
