package de.brightbyte.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

public class StructuredDataCodecTest extends TestCase {
	public void testEncodeValue() {
		assertEquals("null literal", "null", StructuredDataCodec.instance.encodeValue(null));
		assertEquals("true literal", "true", StructuredDataCodec.instance.encodeValue(true));
		assertEquals("false literal", "false", StructuredDataCodec.instance.encodeValue(false));

		assertEquals("simple char", "'x'", StructuredDataCodec.instance.encodeValue('x'));
		assertEquals("nasty char", "'\\''", StructuredDataCodec.instance.encodeValue('\''));

		assertEquals("simple int", "I13", StructuredDataCodec.instance.encodeValue(13));
		assertEquals("negative int", "I-23", StructuredDataCodec.instance.encodeValue(-23));
		assertEquals("simple float", "F13.33", StructuredDataCodec.instance.encodeValue(13.33F));
		assertEquals("negative float", "F-23.33", StructuredDataCodec.instance.encodeValue(-23.33F));
		assertEquals("high precision", "D0.12345678", StructuredDataCodec.instance.encodeValue(0.12345678));
		assertEquals("byte", "B7", StructuredDataCodec.instance.encodeValue((byte)7));
		assertEquals("short", "S-22", StructuredDataCodec.instance.encodeValue((short)-22));
		assertEquals("long", "L987654321", StructuredDataCodec.instance.encodeValue(987654321L));

		assertEquals("simple string", "\"abc\"", StructuredDataCodec.instance.encodeValue("abc"));
		assertEquals("odd string", "\"1\02\"", StructuredDataCodec.instance.encodeValue("1\02"));
		assertEquals("nasty string", "\"foo \\\"x\\\" \\\\bar\"", StructuredDataCodec.instance.encodeValue("foo \"x\" \\bar"));
		assertEquals("very nasty string", "\"\\\"foo\\\\\\\"\\\\\"", StructuredDataCodec.instance.encodeValue("\"foo\\\"\\"));

		List<Object> x = new ArrayList<Object>();
		x.add("foo");
		x.add(true);
		x.add(3);
		x.add(null);
		assertEquals("simple list", "[\"foo\",true,I3,null]", StructuredDataCodec.instance.encodeValue( x ));

		List<Object> y = new ArrayList<Object>();
		y.add("bar");
		y.add(x);
		assertEquals("nested list", "[\"bar\",[\"foo\",true,I3,null]]", StructuredDataCodec.instance.encodeValue( y ));

		Map<Object, Object> m = new HashMap<Object, Object>();
		m.put((short)1, "one");
		m.put("two", 2.3F);
		m.put(true, null);
		assertEquals("simple map", "{true:null,\"two\":F2.3,S1:\"one\"}", StructuredDataCodec.instance.encodeValue( m ));

		Map<Object, Object> n = new HashMap<Object, Object>();
		n.put(m, 'x');
		n.put('x', m);
		assertEquals("nested map", "{{true:null,\"two\":F2.3,S1:\"one\"}:'x','x':{true:null,\"two\":F2.3,S1:\"one\"}}", StructuredDataCodec.instance.encodeValue( n ));
	}

	public void testDecodeValueString_simple() {
		assertEquals("null literal", null, StructuredDataCodec.instance.decodeValue("null"));
		assertEquals("true literal", true, StructuredDataCodec.instance.decodeValue("true"));
		assertEquals("false literal", false, StructuredDataCodec.instance.decodeValue("false"));

		assertEquals("simple char", 'x', StructuredDataCodec.instance.decodeValue("'x'"));
		assertEquals("nasty char", '\'', StructuredDataCodec.instance.decodeValue("'\\''"));

		assertEquals("simple int", 13, StructuredDataCodec.instance.decodeValue("I13"));
		assertEquals("negative int", -23, StructuredDataCodec.instance.decodeValue("I-23"));
		assertEquals("simple float", 13.33F, StructuredDataCodec.instance.decodeValue("F13.33"));
		assertEquals("negative float", -23.33F, StructuredDataCodec.instance.decodeValue("F-23.33"));
		assertEquals("double precision", 0.12345678D, StructuredDataCodec.instance.decodeValue("D0.12345678"));
		assertEquals("byte", (byte)7, StructuredDataCodec.instance.decodeValue("B7"));
		assertEquals("short", (short)-22, StructuredDataCodec.instance.decodeValue("S-22"));
		assertEquals("long", 987654321L, StructuredDataCodec.instance.decodeValue("L987654321"));

		assertEquals("sloppy int", 7, StructuredDataCodec.instance.decodeValue("7"));
		assertEquals("negative sloppy int", -11, StructuredDataCodec.instance.decodeValue("-11"));
		assertEquals("sloppy double", 7.3, StructuredDataCodec.instance.decodeValue("7.3"));
		assertEquals("negative sloppy double", -11.7, StructuredDataCodec.instance.decodeValue("-11.7"));
		
		assertEquals("simple string", "abc", StructuredDataCodec.instance.decodeValue("\"abc\""));
		assertEquals("simple string with spaces", "abc", StructuredDataCodec.instance.decodeValue("\t\"abc\" "));
		assertEquals("ident string", "foo_bar", StructuredDataCodec.instance.decodeValue("foo_bar"));
		assertEquals("odd string", "1\02", StructuredDataCodec.instance.decodeValue("\"1\02\""));
		assertEquals("nasty string", "foo \"x\" \\bar", StructuredDataCodec.instance.decodeValue("\"foo \\\"x\\\" \\\\bar\""));
		assertEquals("very nasty string", "\"foo\\\"\\", StructuredDataCodec.instance.decodeValue("\"\\\"foo\\\\\\\"\\\\\""));
		
		StructuredDataCodec.Lookup lookup = new StructuredDataCodec.Lookup() {
			public Object get(String key) {
				return key.equals("foo") ? "bar" : null;
			}
		};

		StructuredDataCodec codec = new StructuredDataCodec(lookup, null, null);
		assertEquals("reference", "bar", codec.decodeValue("$foo"));
		assertSame("empty reference", null, codec.decodeValue("$bar"));
	}
	
	public void testDecodeValue_structures() {
		List<Object> x = new ArrayList<Object>();
		x.add("foo");
		x.add(true);
		x.add(3);
		x.add(null);
		assertEquals("simple list",  x , StructuredDataCodec.instance.decodeValue("[\"foo\",true,I3,null]"));
		assertEquals("simple list, with spaces",  x , StructuredDataCodec.instance.decodeValue(" [ \"foo\"\t,\ntrue\r,\r\nI3 ,   \t null\r\n  ] "));

		List<Object> y = new ArrayList<Object>();
		y.add("bar");
		y.add(x);
		assertEquals("nested list",  y , StructuredDataCodec.instance.decodeValue("[\"bar\",[\"foo\",true,I3,null]]"));

		Map<Object, Object> m = new HashMap<Object, Object>();
		m.put((short)1, "one");
		m.put("two", 2.3F);
		m.put(true, null);
		assertEquals("simple map",  m , StructuredDataCodec.instance.decodeValue("{two:F2.3,S1:\"one\",true:null}"));

		Map<Object, Object> n = new HashMap<Object, Object>();
		n.put('x', x);
		n.put('m', m);
		assertEquals("nested map",  n , StructuredDataCodec.instance.decodeValue("{'x':[\"foo\",true,I3,null],'m':{\"two\":F2.3,S1:\"one\",true:null}}"));
	}

}
