package de.brightbyte.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Simple codec for encoding basic java data structures to strings, and decoding them again.
 * Supported are all java primitive types (resp. their wrapper classes), as well as null, 
 * Maps and Lists (may be nested arbitrarily). 
 * Caveat: cyclic strucures will cause the codec to recurse indefinitely, eventually resulting
 * in a StackOverflowError.
 */
public class StructuredDataCodec {
	
	public static interface Lookup {
		public Object get(String key);
	}
	
	protected static final Pattern defaultChunker = Pattern.compile(
			"\\s*([\\]\\[{},|:=]|null|true|false|[FD]?-?\\d+(\\.\\d+([eE]-?\\d+)?)?|[BSIL]?-?\\d+|\"(\\\\.|[^\"\\\\]+)*\"|'([^'\\\\]|\\\\'|\\\\\\\\)'|\\$?[\\w.]+)\\s*", Pattern.DOTALL);
	
	protected static final Pattern defaultIdentPattern = Pattern.compile("\\w+");
	
	protected static final Pattern specialCharPattern = Pattern.compile("([\\\\\"])");
	
	public static final String EOF = new String("eof");

	public static final StructuredDataCodec instance = new StructuredDataCodec();
	
	protected Pattern chunker;
	protected Pattern identPattern;
	protected Lookup lookup;
	
	public StructuredDataCodec() {
		this(null, null, null);
	}
	
	public StructuredDataCodec(Lookup lookup, Pattern chunker, Pattern identPattern) {
		if (chunker==null) chunker = defaultChunker;
		if (identPattern==null) identPattern = defaultIdentPattern;
		
		this.lookup = lookup;
		this.chunker = chunker;
		this.identPattern = identPattern;
	}

	protected String encodeMap(Map<?, ?> m) {
		if (m==null) return encodeValue(null);
		
		StringBuilder s = new StringBuilder();
		
		s.append('{');
		boolean first = true;
		for (Map.Entry<?, ?> e: m.entrySet()) {
			String k = encodeValue(e.getKey());
			String v = encodeValue(e.getValue());
			
			if (first) first = false;
			else s.append(',');
			
			s.append(k);
			s.append(':');
			s.append(v);
		}
		s.append('}');
		
		return s.toString();
	}
	
	protected String encodeList(List<?> l) {
		if (l==null) return encodeValue(null);
		
		StringBuilder s = new StringBuilder();
		
		s.append('[');
		boolean first = true;
		for (Object e: l) {
			String v = encodeValue(e);
			
			if (first) first = false;
			else s.append(',');
			
			s.append(v);
		}
		s.append(']');
		
		return s.toString();
	}
	
	public String encodeValue(Object obj) {
		if (obj==null) return "null";
		else if (obj instanceof Byte) return "B"+obj.toString();
		else if (obj instanceof Short) return "S"+obj.toString();
		else if (obj instanceof Integer) return "I"+obj.toString();
		else if (obj instanceof Long) return "L"+obj.toString();
		else if (obj instanceof Float) return "F"+obj.toString();
		else if (obj instanceof Double) return "D"+obj.toString();
		else if (obj instanceof Character) return "'"+((((Character)obj).charValue()=='\'')?"\\'":obj)+"'";
		else if (obj instanceof Boolean) return (((Boolean)obj).booleanValue())?"true":"false";
		else if (obj instanceof String) return quote((String)obj);
		else if (obj instanceof Map) return encodeMap((Map)obj);
		else if (obj instanceof List) return encodeList((List)obj);
		else if (obj instanceof Object[]) return encodeList(Arrays.asList((Object[])obj));
		else throw new IllegalArgumentException("unsupported data type "+obj.getClass().getName());
	}
	
	public static class Scanner {
		protected Matcher matcher;
		protected String text;
		protected int pos;
		protected int prev;
		
		public Scanner(Pattern chunker, String s) {
			text = s;
			pos = 0;
			prev = 0;
			matcher = chunker.matcher(s);
		}
		
		public void seek(int pos) {
			if (pos<0) throw new IllegalArgumentException("position must not be < 0: "+pos);
			this.prev = this.pos;
			this.pos = pos;
			matcher.region(pos, text.length());
		}
		
		public void pushback() {
			pos = prev;
		}
		
		public String next() {
			if (!matcher.lookingAt()) {
				if (matcher.hitEnd()) return EOF;
				else throw new IllegalArgumentException("malformed input: " + this);
			}
			else {
				String s = matcher.group(1);
				int end = matcher.end(); 
				if (end<matcher.regionEnd()) seek(end);
				return s;
			}
		}
		
		public String toString() {
			return getClass().getName()+"(@"+pos+")";
		}
		
		public int getPos() {
			return pos;
		}
		
		public int scanValue() {
			if (pos>=text.length()) return pos;
			String chunk = next();
			scanChunk(chunk, true);
			return pos;
		}
		
		protected Object scanChunk(String chunk, boolean allowStruct) {
			if (chunk == EOF) return pos;			
			char ch = chunk.charAt(0);

			if (ch=='{') {
				if (!allowStruct) throw new IllegalArgumentException("structure not allowed here!");
				return scanMap(null);
			}
			else if (ch=='[') {
				if (!allowStruct) throw new IllegalArgumentException("structure not allowed here!");
				return scanList(null);
			}
			else {
				return handleScalar(chunk);
			}
		}
		
		protected Object scanMap(Map<Object, Object> into) {
			boolean first = true;
			
			while (true) {
				String chunk = next();
				if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
				if (chunk.equals("}")) break;
				
				if (first) first = false;
				else {
					if (!chunk.equals(",") && !chunk.equals("|")) throw new IllegalArgumentException("bad input, ´,´ or ´|´ expected");
					chunk = next();
					if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
				}

				Object k = scanChunk(chunk, false);

				chunk = next();
				if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
				if (!chunk.equals(":") && !chunk.equals("=")) throw new IllegalArgumentException("bad input, ´:´ or ´=´ expected");

				chunk = next();
				if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
				Object v = scanChunk(chunk, true);
				
				if (into!=null) into.put(k, v);
			}
				
			return into;
		}
		
		protected Object scanList(List<Object> into) {
			boolean first = true;
			
			while (true) {
				String chunk = next();
				if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
				if (chunk.equals("]")) break;
				
				if (first) first = false;
				else {
					if (!chunk.equals(",") && !chunk.equals("|")) throw new IllegalArgumentException("bad input, ´,´ or ´|´ expected");
					chunk = next();
					if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
				}
					
				Object v = scanChunk(chunk, true);
				if (into!=null) into.add(v);
			}
				
			return into;
		}
		
		protected Object handleScalar(String chunk) {
			return null; //noop
		}
		
	}
	
	protected class Builder extends Scanner {
		protected Builder(Pattern chunker, String s) {
			super(chunker, s);
		}
		
		public Object buildValue() {
			String chunk = next();
			return scanChunk(chunk, true);
		}

		protected Object handleScalar(String chunk) {
			char ch = chunk.charAt(0);
			Object v;
			
			if (ch=='n' && chunk.equals("null")) v = null;
			else if (ch=='t' && chunk.equals("true")) v = Boolean.TRUE;
			else if (ch=='f' && chunk.equals("false")) v = Boolean.FALSE;
			else if (ch=='"') v = chunk.substring(1, chunk.length()-1).replaceAll("\\\\(.)", "$1");
			else if (ch=='\'') v = chunk.equals("'\\''") ? '\'' : chunk.charAt(1);
			else if (ch=='B') v = Byte.parseByte(chunk.substring(1));
			else if (ch=='S') v = Short.parseShort(chunk.substring(1));
			else if (ch=='I') v = Integer.parseInt(chunk.substring(1));
			else if (ch=='L') v = Long.parseLong(chunk.substring(1));
			else if (ch=='F') v = Float.parseFloat(chunk.substring(1));
			else if (ch=='D') v = Double.parseDouble(chunk.substring(1));
			else if (ch=='$') {
				if (lookup==null) throw new IllegalArgumentException("No lookup provided, references not supported! Found "+chunk);
				else v = lookup.get(chunk.substring(1));
			}
			else if ((ch>='0' && ch<='9') || ch=='-') {
				if (chunk.indexOf('.')>0) v = Double.parseDouble(chunk);
				else v = Integer.parseInt(chunk);
			}
			else if (identPattern.matcher(chunk).matches()) v = chunk;  
			else throw new IllegalArgumentException("illegal input (chunk: "+chunk+"): "+this);
			
			return v;
		}

		@Override
		protected Object scanList(List<Object> into) {
			into = new ArrayList<Object>();
			super.scanList(into);
			return into;
		}

		@Override
		protected Object scanMap(Map<Object, Object> into) {
			into = new HashMap<Object, Object>();
			super.scanMap(into);
			return into;
		}
				
	}
	
	public Scanner scanner(String s) {
		return new Scanner(chunker, s);
	}
	
	protected Builder builder(String s) {
		return new Builder(chunker, s);
	}
	
	public Object decodeValue(String s) {
		Builder builder = builder(s);
		return builder.buildValue();
		
		/*
		Scanner scanner = scanner(s);
		Object obj = decodeValue(scanner);
		if (obj == EOF) throw new IllegalArgumentException("premature end of input");
		return obj;
		*/
	}

	/*
	protected Object decodeValue(Scanner scanner) {
		String chunk = scanner.next();
		if (chunk==null) throw new IllegalArgumentException("bad input: "+scanner);
		
		return decodeChunk(chunk, scanner);
	}

	protected Object decodeChunk(String chunk, Scanner scanner) {
		if (chunk == EOF) throw new IllegalArgumentException("premature end of input");
		
		char ch = chunk.charAt(0);
		
		if (ch=='{') return decodeMap(scanner);
		else if (ch=='[') return decodeList(scanner);
		else if (ch=='n' && chunk.equals("null")) return null;
		else if (ch=='t' && chunk.equals("true")) return Boolean.TRUE;
		else if (ch=='f' && chunk.equals("false")) return Boolean.FALSE;
		else if (ch=='"') return chunk.substring(1, chunk.length()-1).replaceAll("\\\\(.)", "$1");
		else if (ch=='\'') return chunk.equals("'\\''") ? '\'' : chunk.charAt(1);
		else if (ch=='B') return Byte.parseByte(chunk.substring(1));
		else if (ch=='S') return Short.parseShort(chunk.substring(1));
		else if (ch=='I') return Integer.parseInt(chunk.substring(1));
		else if (ch=='L') return Long.parseLong(chunk.substring(1));
		else if (ch=='F') return Float.parseFloat(chunk.substring(1));
		else if (ch=='D') return Double.parseDouble(chunk.substring(1));
		else if (ch=='$') {
			if (lookup==null) throw new IllegalArgumentException("No lookup provided, references not supported! Found "+chunk);
			else return lookup.get(chunk.substring(1));
		}
		else if ((ch>='0' && ch<='9') || ch=='-') {
			if (chunk.indexOf('.')>0) return Double.parseDouble(chunk);
			else return Integer.parseInt(chunk);
		}
		else if (identPattern.matcher(chunk).matches()) return chunk;  
		else throw new IllegalArgumentException("illegal input (chunk: "+chunk+"): "+scanner);
		
	}

	protected Map<?, ?> decodeMap(Scanner scanner) {
		HashMap<Object, Object> m = new HashMap<Object, Object>();
		boolean first = true;
		
		while (true) {
			String chunk = scanner.next();
			if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
			if (chunk.equals("}")) break;
			
			if (first) first = false;
			else {
				if (!chunk.equals(",") && !chunk.equals("|")) throw new IllegalArgumentException("bad input, ´,´ or ´|´ expected");
				chunk = scanner.next();
				if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
			}

			Object k = decodeChunk(chunk, scanner);

			chunk = scanner.next();
			if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
			if (!chunk.equals(":") && !chunk.equals("=")) throw new IllegalArgumentException("bad input, ´:´ or ´=´ expected");

			chunk = scanner.next();
			if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
			Object v = decodeChunk(chunk, scanner);
			
			m.put(k, v);
		}
			
		return m;
	}

	protected List<?> decodeList(Scanner scanner) {
		List<Object> m = new ArrayList<Object>();
		boolean first = true;
		
		while (true) {
			String chunk = scanner.next();
			if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
			if (chunk.equals("]")) break;
			
			if (first) first = false;
			else {
				if (!chunk.equals(",") && !chunk.equals("|")) throw new IllegalArgumentException("bad input, ´,´ or ´|´ expected");
				chunk = scanner.next();
				if (chunk==EOF) throw new IllegalArgumentException("premature end of input");
			}
				
			Object v = decodeChunk(chunk, scanner);
			
			m.add(v);
		}
			
		return m;
	}
	*/
	
	public String quote(String s) {
		return "\"" + specialCharPattern.matcher(s).replaceAll("\\\\$1") + "\"";
	}

	/*
	protected static Number decodeNumber(String chunk) {
		int idx = chunk.indexOf('.');
		if (idx>=0) return Double.parseDouble(chunk);
		else return Long.parseLong(chunk);
	}
	*/
	
	public Map<String, Object> decodeValues(Properties props) {
		Map<String, Object> m = new HashMap<String, Object>();
		Enumeration names = props.propertyNames();
		while (names.hasMoreElements()) {
			String n = (String)names.nextElement();
			String s = props.getProperty(n);
			Object v = decodeValue(s);
			
			m.put(n, v);
		}
		
		return m;
	}
	
	public Map<String, Object> decodeValues(Map<String, String> props) {
		Map<String, Object> m = new HashMap<String, Object>();
		for (Map.Entry<String, String> e: props.entrySet()) {
			String n = e.getKey();
			String s = e.getValue();
			Object v = decodeValue(s);
			
			m.put(n, v);
		}
		
		return m;
	}
}
