package de.brightbyte.util;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Formatter;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class StringUtils {
	
	public static String join(String delimiter, Object array) throws IllegalArgumentException {
		return join(delimiter, new ArrayIterator<Object>(array));
	}
	
	public static String join(String delimiter, Iterable<? extends Object> iterable) {
		return join(delimiter, iterable.iterator());
	}
	
	public static String join(String delimiter, Iterator<? extends Object> it) {
		StringBuffer buffer = new StringBuffer();
		while (it.hasNext()) {
			if (buffer.length() > 0) buffer.append(delimiter);
			Object object = it.next();
			if (object != null)
				buffer.append(object.toString());
		}
		return buffer.toString();
	}
	
	protected static final Pattern patternFormatPlaceholderPattern = Pattern.compile("\\{.*?\\}");
	
	public static String patternFormat(String pattern, Map<String, Object> properties) {
		Matcher matcher = patternFormatPlaceholderPattern.matcher(pattern);
		
		String result = "";
		int ofs = 0;
		while (matcher.find()) {
			int idx = matcher.start();
			result += pattern.substring(ofs, idx);
			
			String n = matcher.group();
			n = n.substring(1, n.length()-1);
			
			Object v = properties.get(n);
			if (v==null) throw new IllegalArgumentException("bad pattern: the property "+n+" was not found. Pattern: "+pattern);
			
			result += v;
			
			ofs = matcher.end();
		}
		
		result += pattern.substring(ofs);
		
		return result;
	}
	
	public static String firstToUppercase(String s) {
		if (s==null || s.length()==0) return s;
		if (!Character.isLowerCase(s.charAt(0))) return s;
		s = s.substring(0, 1).toUpperCase() + s.substring(1);
		return s;
	}
	
	public static String clipString(String text, int max, String ellipsis) {
		if (text.length()>max) text = text.substring(0, max) + ellipsis;
		return text;
	}
	
	public static String describeLocation(String text, int start, int end) {
		int i = start - 16;
		int j = end + 16;
		
		if (i<0) i = 0;
		if (j>text.length()) j= text.length();
		
		//extract context
		String s = text.substring(i, j);
		if (i>0) s = "..." + s;
		if (j<text.length()) s += "...";
		
		//count lines
		boolean stopret = false;
		int line = 1;
		int ofs = 1;
		int p = start;
		if (p>=text.length()) p = text.length() -1;
		while (p>=0) {
			char ch = text.charAt(p);
			
			if (ch=='\n') {
				line++;
				stopret = true;
			}
			else if (ch=='\r' && !stopret) {
				line++;
			}
			else {
				stopret = false;
				if (line==1) ofs++;
			}
			
			p--;
		}
		
		//make message 
		String descr = " @" + start +" ("+line+":"+ofs+"): <" + s.replaceAll("\\s", " ") + ">";
		return descr;
	}

	public static int indexOf(char c, CharSequence text) {
		int length = text.length();
		for (int i=0; i<length; i++) {
			if (c==text.charAt(i)) return i;
		}
		
		return -1;
	}

	public static void strip(StringBuilder text) {
		int length = text.length();
		int i = 0;
		int j = length;
		
		for (; i<length; i++) {
			if (text.charAt(i)>'\u0020') break;
		}

		for (; j>i; j--) {
			if (text.charAt(j-1)>'\u0020') break;
		}
		
		if (i>0) text.delete(0, i);
		if (j<length) text.delete(j-i, length-i);
	}
	
	public static CharSequence trim(CharSequence text) {
		int length = text.length();
		int i = 0;
		int j = length;
		
		for (; i<length; i++) {
			if (text.charAt(i)>'\u0020') break;
		}

		for (; j>i; j--) {
			if (text.charAt(j-1)>'\u0020') break;
		}
		
		if (i==0 && j==length) return text;
		else return text.subSequence(i, j);
	}
	
	public static void subst(StringBuilder s, char replace, char by) {
		int length = s.length();
		for (int i=0; i<length; i++) {
			if (replace==s.charAt(i)) s.setCharAt(i, by);
		}
	}
	
	public static CharSequence replace(CharSequence s, char replace, char by) {
		StringBuilder t = null;
		int length = s.length();
		int pos = 0;
		for (int i=0; i<length; i++) {
			char ch = s.charAt(i);
			if (replace==ch) {
				if (t==null) t = new StringBuilder(s.length());
				t.append(s, pos, i);
				t.append(by);
				
				pos = i+1;
			}
		}
		
		if (t==null) return s;
		else {
			t.append(s, pos, s.length());
			return t;
		}
	}

	public static CharSequence toLowerCase(CharSequence name) {
		return name.toString().toLowerCase(); //TODO: optimize... do manuially? or check fro all-lower first?
	}

	public static int indexOf(CharSequence needle, CharSequence haystack) {
		return haystack.toString().indexOf(needle.toString()); //TODO: optimize!
	}

	public static CharSequence replace(CharSequence text, CharSequence pattern, CharSequence by) {
		return text.toString().replace(pattern, by); //TODO: optimize!
	}

	public static boolean equals(CharSequence a, CharSequence b) {
		if (a==b) return true;
		if (a==null || b==null) return false;
		if (a instanceof String && b instanceof String) return a.equals(b); //NOTE: StringBuilder uses identity!
		
		int v = a.length();
		int w = b.length();
		
		if (v!=w) return false;
		
		for (int i=0; i<v; i++) {
			if (a.charAt(i)!=b.charAt(i)) return false;
		}
		
		return true;
	}
	
	public static byte[] md5(String s) {
        try {
        	MessageDigest md5 = MessageDigest.getInstance("MD5");
            md5.update(s.getBytes("UTF-8"));
            return md5.digest();
        } catch (NoSuchAlgorithmException ex) {
            throw new Error("MD5 not supported");
        } catch (UnsupportedEncodingException e) {
        	throw new Error("UTF-8 not supported");
		}
	}

	public static byte[] sha1(String s) {
        try {
        	MessageDigest md5 = MessageDigest.getInstance("SHA-1");
            md5.update(s.getBytes("UTF-8"));
            return md5.digest();
        } catch (NoSuchAlgorithmException ex) {
            throw new Error("SHA-1 not supported");
        } catch (UnsupportedEncodingException e) {
        	throw new Error("UTF-8 not supported");
		}
	}

	public static String hex(byte[] bb) {
        StringBuilder result = new StringBuilder(32);
        Formatter f = new Formatter(result);
        
        for (byte b : bb) {
            f.format("%02x", b);
        }
        
        return result.toString();
	}

	public static boolean containsControlChar(CharSequence term, boolean strict) {
		int c = term.length();
		
		for (int i=0; i<c; i++) {
			char ch = term.charAt(i);
			if (ch<32) {
				if (!strict) {
					if (ch=='\t' || ch=='\r' || ch=='\n') continue;
				}
				
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Truncates the given string so that it will not exceed the given max number of bytes
	 * when encoded as UTF-8. Avoids actually encoding the string.
	 * 
	 * @param s the string to truncate
	 * @param maxbytes the maximum number of bytes the string may use in UTF-8 encoded form
	 * @returns the truncated string, or the original string, if there is no need to clip it
	 */
	public static String clipStringUtf8(String s, int maxbytes) {
		//NOTE: determine length in bytes when encoded as utf-8.
		char[] chars = s.toCharArray();
		int len = chars.length;
		if (len*4<maxbytes) return s; //#bytes can't be more than #chars * 4
		
		//int top = 0;
		int bytes = 0;
		for (int i = 0; i<len; i++) { //TODO: iterate codepoints instead of utf-16 chars.
			int ch = chars[i];
			//if (ch>top) top = ch;

			//XXX: hm, nice idea, but escape chars arn't counted anyway.
			//if (ch=='\'') bytes += 2; //plain ascii, but becomes 2 bytes by escaping
			//else if (ch=='\\') bytes += 2; //plain ascii, but becomes 2 bytes by escaping
			
			if (ch<128) bytes += 1; //plain ascii, 1 byte
			else if (ch<2048) bytes += 2; //2-byte encoding
			else if (ch<65536) bytes += 3; //3-byte encoding
			else bytes += 4; //3-byte encoding

			if (bytes>maxbytes) {
				s = s.substring(0, i);
				
				break;
			}
		}
		
		return s;
	}
	
}
