package de.brightbyte.util;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class SpiTools {
	
	public static class ProviderRegistry<T> {
		protected Map<String, T> spis = new HashMap<String, T>();
		protected Class<T> providerInterface;
		protected Class serviceClass;
		
		public ProviderRegistry(Class<T> providerInterface, Class serviceClass) {
			this.providerInterface = providerInterface;
			this.serviceClass = serviceClass;
		}

		public T getProvider(String name) {
			return spis.get(name);
		}
		
		public void registerProvider(String name, T sp) {
			spis.put(name, sp);
		}
		
		public void registerProvider(String name, Class<? extends T> spc) {
			T sp = instantiate(spc);
			spis.put(name, sp);
		}
		
		public void registerProvider(String name, String className) {
			T sp = newSPI(serviceClass, providerInterface, className);
			spis.put(name, sp);
		}
		
		public Set<String> getProviderNames() {
			return Collections.unmodifiableSet(spis.keySet());
		}
	}
	
	public static <T> T newDefaultSPI(Class<?> serviceClass, Class<T> providerInterface, String defaultClass) {
		String prop = serviceClass.getName() + ".spi";
		String className = defaultClass;
		
		try {
			className = System.getProperty(prop, defaultClass);
		}
		catch (SecurityException ex) {
			/* ignore, use default */
		}
		
		return newSPI(serviceClass, providerInterface, className);
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T newSPI(Class<?> serviceClass, Class<T> providerInterface, String className) {
		String[] nn = className.split("[ ,;:/|]+");
		
		Class<T> c = null;

		for (String n: nn) {
			try {
				c = (Class<T>)serviceClass.getClassLoader().loadClass(n);
			}
			catch (ClassNotFoundException ex) {
				try {
					c = (Class<T>)Thread.currentThread().getContextClassLoader().loadClass(n);
				}
				catch (ClassNotFoundException exx) {
					try {
						c = (Class<T>)ClassLoader.getSystemClassLoader().loadClass(n);
					}
					catch (ClassNotFoundException exxx) {
						//ignore
					}
				}
			}
			
			if (c!=null) {
				if (!providerInterface.isAssignableFrom(c)) {
					throw new RuntimeException("bad spi class "+n+" for service class "+serviceClass.getName()+", its incompatible with the SPI interface "+providerInterface.getName());
				}
				
				return instantiate(c);
			}
		}
		
		throw new RuntimeException("failed to load SPI class: one of "+className);
	}
	
	public static <T> T instantiate(Class<T> c) {
		try {
			T spi = c.newInstance();
			return spi;
		}
		catch (InstantiationException iex) {
			throw new RuntimeException("failed to instantiate class "+c.getName(), iex);
		}
		catch (IllegalAccessException iex) {
			throw new RuntimeException("could not instantiate class "+c.getName(), iex);
		}		
	}
}
