package de.brightbyte.util;

import java.awt.EventQueue;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URL;
import java.text.MessageFormat;
import java.util.Date;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;

/**
 * Wrapper class for java.util.logging.Logger providing an interface more like Log4J.
 * 
 * @author daniel
 */
public class Logger  {
	
	/**
	 * Single line formatter for log output. To activate, use this line in the logger configuration
	 * property file:
	 * <pre>
	 * java.util.logging.ConsoleHandler.formatter = com.braintribe.csp.workflow.client.Logger$LogFormatter
	 * </pre>
	 * 
	 * @author daniel
	 */
	public static class LogFormatter extends Formatter {

		private final static String newline = "\r\n"; //HACK! get from system property, but handle security exception...
	    private final static String format = "{0} [{1,date} {1,time}] {2}: {3}";

	    private MessageFormat formatter = new MessageFormat(format);

	    private Date dat = new Date();
	    private Object args[] = new Object[4];
	    private StringBuffer buffer = new StringBuffer();
	    
	    @Override
		public synchronized String format(LogRecord record) {
	    	buffer.setLength(0);

	    	args[0] = record.getLevel().getName();
	    	
	    	dat.setTime(record.getMillis());
	    	args[1] = dat;
	    	
	    	if (record.getSourceClassName() != null) {	
	    		args[2] = record.getSourceClassName();
	    	} else {
	    		args[2] = record.getLoggerName();
	    	}
	    	
	    	if (record.getSourceMethodName() != null) {	
	    		args[2] = ((String)args[2]) + record.getSourceMethodName();
	    	}
	    	
	    	args[3] = formatMessage(record);
	    	
	    	buffer.append( formatter.format(args) );
	    	
	    	if (record.getThrown() != null) {
	    	    try {
	    	    	StringWriter sw = new StringWriter();
	    	        PrintWriter pw = new PrintWriter(sw);
	    	        pw.println();
	    	        record.getThrown().printStackTrace(pw);
	    	        pw.close();
	    	        
	    	        buffer.append(sw.toString());
	    	    } catch (Exception ex) {
	    	    	//ignore
	    	    }
	    	}
	    
	    	buffer.append(newline);
	    	return buffer.toString();
	    }
	}
	
	public static class AwtExceptionHandler {
		public void handle(Throwable ex) {
			Logger logger = Logger.getLogger(EventQueue.class);
			logger.error("uncaught exception in AWT event dispatcher " + Thread.currentThread().getName(), ex);
		}
	}
	
	public static Logger getLogger(Class cls) {
		return new Logger(cls);
	}
	
	public static void readConfiguration(URL config) throws IOException {
		InputStream in = config.openStream();
		LogManager.getLogManager().readConfiguration(in);
		in.close();
	}
	
	public static void installAwtExceptionLogger() {
		installAwtExceptionHandler(AwtExceptionHandler.class);
	}
	
	public static void installAwtExceptionHandler(Class cls) {
		try {
			System.setProperty("sun.awt.exception.handler", cls.getName());
		}
		catch (SecurityException ex) {
			//ignore.
		}
	}
	
	protected String sourceClass;
	protected java.util.logging.Logger logger;
	
	public Logger(Class cls) {
		this(java.util.logging.Logger.getLogger(cls.getName()), cls);
	}

	public Logger(java.util.logging.Logger logger, Class sourceClass) {
		this.logger = logger;
		this.sourceClass = sourceClass.getName();
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.util.LogOutput#debug(java.lang.String)
	 */
	public void debug(String msg) {
		logger.logp(Level.FINE, sourceClass, null, msg);
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.util.LogOutput#trace(java.lang.String)
	 */
	public void trace(String msg) {
		logger.logp(Level.FINER, sourceClass, null, msg);
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.util.LogOutput#info(java.lang.String)
	 */
	public void info(String msg) {
		logger.logp(Level.INFO, sourceClass, null, msg);
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.util.LogOutput#warn(java.lang.String)
	 */
	public void warn(String msg) {
		logger.logp(Level.WARNING, sourceClass, null, msg);
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.util.LogOutput#warn(java.lang.String, java.lang.Throwable)
	 */
	public void warn(String msg, Throwable ex) {
		logger.logp(Level.WARNING, sourceClass, null, msg, ex);
	}

	/* (non-Javadoc)
	 * @see de.brightbyte.util.LogOutput#error(java.lang.String, java.lang.Throwable)
	 */
	public void error(String msg, Throwable ex) {
		logger.logp(Level.SEVERE, sourceClass, null, msg, ex);
	}
}
