package de.brightbyte.util;

import java.beans.BeanInfo;
import java.beans.EventSetDescriptor;
import java.beans.FeatureDescriptor;
import java.beans.IndexedPropertyDescriptor;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.MethodDescriptor;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public class BeanUtils {

	public static class NamedPropertyDescriptor extends PropertyDescriptor {

		private Method namedReadMethod;
		private Method namedWriteMethod;
		
		public NamedPropertyDescriptor(IndexedPropertyDescriptor descriptor) throws IntrospectionException {
			this( descriptor.getName(),
				  descriptor.getReadMethod(),
				  descriptor.getWriteMethod(),
				  descriptor.getIndexedReadMethod(),
				  descriptor.getIndexedWriteMethod() );
			
			setDisplayName(descriptor.getDisplayName());
		}
		
		public NamedPropertyDescriptor(String propertyName, Method readMethod, Method writeMethod, Method namedReadMethod, Method namedWriteMethod) throws IntrospectionException {
			super(propertyName, readMethod, writeMethod);
			
			this.namedReadMethod = namedReadMethod; 
			this.namedWriteMethod = namedWriteMethod; 
		}

		public Method getNamedReadMethod() {
			return namedReadMethod;
		}

		public void setNamedReadMethod(Method namedReadMethod) {
			this.namedReadMethod = namedReadMethod;
		}

		public Method getNamedWriteMethod() {
			return namedWriteMethod;
		}

		public void setNamedWriteMethod(Method namedWriteMethod) {
			this.namedWriteMethod = namedWriteMethod;
		}

	}
	
	public static class BagPropertyDescriptor extends PropertyDescriptor {

		private Method addMethod;
		private Method removeMethod;
		
		public BagPropertyDescriptor(EventSetDescriptor evd) throws IntrospectionException {
			this(evd.getName(),
					null,
					null,
					evd.getAddListenerMethod(),
					evd.getRemoveListenerMethod()
					);
		}

		public BagPropertyDescriptor(String propertyName, Method readMethod, Method writeMethod, Method addMethod, Method removeMethod) throws IntrospectionException {
			super(propertyName, readMethod, writeMethod);
			
			this.addMethod = addMethod; 
			this.removeMethod = removeMethod; 
		}

		public Method getAddMethod() {
			return addMethod;
		}

		public void setAddMethod(Method namedReadMethod) {
			this.addMethod = namedReadMethod;
		}

		public Method getRemoveMethod() {
			return removeMethod;
		}

		public void setRemoveMethod(Method namedWriteMethod) {
			this.removeMethod = namedWriteMethod;
		}

	}
	
	public static <T> Constructor<T> findConstructor(Class<T> cl, Object[] params) throws NoSuchMethodException {
		if (params==null) params= new Object[] {};

		Constructor[] constructors = cl.getConstructors();
		
		constructorLoop:
		for (Constructor constructor : constructors) {
			if (!Modifier.isPublic(constructor.getModifiers())) continue;
			
			Class[] paramTypes = constructor.getParameterTypes();
			
			if (paramTypes.length != params.length) continue constructorLoop;

			for (int i = 0; i < paramTypes.length; i++) {
				Class pt = paramTypes[i];
				Class ot = params[i] == null ? null : params[i].getClass();
				
				if (!isCompatibleTo(ot, pt)) continue constructorLoop;
			}
			
			return constructor;
		}
		
		throw new NoSuchMethodException("no constructor found in "+cl+" matching "+Arrays.asList(params));
	}

	public static boolean isCompatibleTo(Class concrete, Class declared) {
		if (declared==concrete) return true;
		if (concrete==null) return !declared.isPrimitive();
		if (declared.isAssignableFrom(concrete)) return true;

		if (declared.isPrimitive()) {
			if (declared == Byte.TYPE && concrete == Byte.class) return true;
			if (declared == Short.TYPE && concrete == Short.class) return true;
			if (declared == Integer.TYPE && concrete == Integer.class) return true;
			if (declared == Long.TYPE && concrete == Long.class) return true;
			if (declared == Float.TYPE && concrete == Float.class) return true;
			if (declared == Double.TYPE && concrete == Double.class) return true;
			if (declared == Character.TYPE && concrete == Character.class) return true;
			if (declared == Boolean.TYPE && concrete == Boolean.class) return true;
			
			//TODO: automatic numeric conversion
			/*
			if (Number.class.isAssignableFrom(concrete)) {
				if (declared != Character.TYPE && declared != Boolean.TYPE) {
					return true;
				}
			}
			*/
		}
		
		return false;
	}

	public static PropertyDescriptor findPropertyDescriptor(String propertyName, BeanInfo beanInfo, boolean getterRequired, boolean setterRequired) throws IntrospectionException {
		PropertyDescriptor[] props = beanInfo.getPropertyDescriptors();
		
		for (int i = 0; i < props.length; i++) {
			if (!propertyName.equals(props[i].getName())) continue;
			
			Method setter = props[i].getWriteMethod();
			Method getter = props[i].getReadMethod();
			
			if (setterRequired && setter == null) {
				throw new IntrospectionException("no setter defined for property "+propertyName+" in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
			}
			
			if (getterRequired && getter == null) {
				throw new IntrospectionException("no getter defined for property "+propertyName+" in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
			}
			
			return props[i];
		}
		
		throw new IntrospectionException("property "+propertyName+" not found in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
	}

	public static NamedPropertyDescriptor findNamedPropertyDescriptor(String propertyName, BeanInfo beanInfo, boolean getterRequired, boolean setterRequired) throws IntrospectionException {
		PropertyDescriptor[] props = beanInfo.getPropertyDescriptors();
		
		NamedPropertyDescriptor pd = null;
		Method setter = null;
		Method getter = null;
		
		for (int i = 0; i < props.length; i++) {
			if (!propertyName.equals(props[i].getName())) continue;
			if (!(props[i] instanceof IndexedPropertyDescriptor)) continue;
			
			pd = new NamedPropertyDescriptor((IndexedPropertyDescriptor)props[i]);
			break;
		}
		
		if (pd==null) {
			//NOTE: this search is reather lengthy and should be cached - in a BeanInfo??
			
			MethodDescriptor[] meths = beanInfo.getMethodDescriptors();
			
			String n = Character.toUpperCase(propertyName.charAt(0))+propertyName.substring(1);
			String gettername = "get" + n;
			String settername = "set" + n;
			String puttername = "put" + n;
			
			for (int i = 0; i < meths.length; i++) {
				Method m = meths[i].getMethod();
				
				if (getter==null && m.getName().equals(gettername)) {
					Class[] pt = m.getParameterTypes();
					if (pt.length!=1) continue;

					Class rt = m.getReturnType();
					if (rt==null || rt==Void.TYPE) continue;
					
					getter = m;
				}
				else if (setter==null && (m.getName().equals(settername) || m.getName().equals(puttername))) {
					Class[] pt = m.getParameterTypes();
					if (pt.length!=2) continue;

					setter = m;
				}
				
				if (getter!=null && setter!=null) break;
			}
			
			if (setter!=null || getter!=null) {
				pd = new NamedPropertyDescriptor(propertyName, null, null, getter, setter);
			}
		}

		if (pd==null) throw new IntrospectionException("indexed property "+propertyName+" not found in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());

		setter = pd.getNamedWriteMethod();
		getter = pd.getNamedReadMethod();
		
		if (setterRequired && setter == null) {
			throw new IntrospectionException("no indexed setter defined for property "+propertyName+" in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
		}
		
		if (getterRequired && getter == null) {
			throw new IntrospectionException("no indexed getter defined for property "+propertyName+" in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
		}

		return pd;
	}
	
	public static BagPropertyDescriptor findBagPropertyDescriptor(String propertyName, BeanInfo beanInfo, boolean addRequired, boolean removeRequired) throws IntrospectionException {
		EventSetDescriptor[] events = beanInfo.getEventSetDescriptors();
		
		BagPropertyDescriptor pd = null;
		Method adder = null;
		Method remover = null;
		
		for (int i = 0; i < events.length; i++) {
			if (!propertyName.equals(events[i].getName())) continue;
			
			pd = new BagPropertyDescriptor((EventSetDescriptor)events[i]);
			break;
		}
		
		if (pd==null) {
			//NOTE: this search is reather lengthy and should be cached - in a BeanInfo??
			
			MethodDescriptor[] meths = beanInfo.getMethodDescriptors();
			
			String n = Character.toUpperCase(propertyName.charAt(0))+propertyName.substring(1);
			String addername = "add" + n;
			String removername = "remove" + n;
			
			for (int i = 0; i < meths.length; i++) {
				Method m = meths[i].getMethod();
				
				if (adder==null && m.getName().equals(addername)) {
					Class[] pt = m.getParameterTypes();
					if (pt.length!=1) continue;

					adder = m;
				}
				else if (remover==null && m.getName().equals(removername)) {
					Class[] pt = m.getParameterTypes();
					if (pt.length!=1) continue;

					remover = m;
				}
				
				if (adder!=null && remover!=null) break;
			}
			
			if (adder!=null || remover!=null) {
				pd = new BagPropertyDescriptor(propertyName, null, null, adder, remover);
			}
		}

		if (pd==null) throw new IntrospectionException("bag property "+propertyName+" not found in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());

		adder = pd.getAddMethod();
		remover = pd.getRemoveMethod();
		
		if (addRequired && adder == null) {
			throw new IntrospectionException("no adder defined for property "+propertyName+" in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
		}
		
		if (removeRequired && remover == null) {
			throw new IntrospectionException("no remover defined for property "+propertyName+" in class "+beanInfo.getBeanDescriptor().getBeanClass().getName());
		}

		return pd;
	}
	
	public static void setBeanProperty(Object bean, BeanInfo beanInfo, String name, Object value) throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (beanInfo==null) beanInfo = Introspector.getBeanInfo(bean.getClass());
		
		PropertyDescriptor pd = findPropertyDescriptor(name, beanInfo, false, true);
		
		//NOTE: don't try to set primitive value to null.
		//TODO: optionally fail if that is tried.
		if (pd.getPropertyType().isPrimitive() && value==null) return; 
		
		Method setter = pd.getWriteMethod();
		if (setter==null) throw new IntrospectionException("no setter method found for property "+name+" in "+bean.getClass());
		
		setter.invoke(bean, new Object[] { value });
		//TODO: handle InvocationTargetException
	}
	
	public static Object getBeanProperty(Object bean, BeanInfo beanInfo, String name) throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (beanInfo==null) beanInfo = Introspector.getBeanInfo(bean.getClass());
		
		PropertyDescriptor pd = findPropertyDescriptor(name, beanInfo, true, false);
		
		Method getter = pd.getReadMethod();
		if (getter==null) throw new IntrospectionException("no getter method found for property "+name+" in "+bean.getClass());
		
		return getter.invoke(bean, (Object[])null);
		//TODO: handle InvocationTargetException
	}
	
	public static Object getNamedProperty(Object bean, BeanInfo beanInfo, String prop, String key) throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (beanInfo==null) beanInfo = Introspector.getBeanInfo(bean.getClass());
		
		NamedPropertyDescriptor pd = findNamedPropertyDescriptor(prop, beanInfo, true, false);
		
		Method getter = pd.getNamedReadMethod();
		if (getter==null) throw new IntrospectionException("no indexed getter method found for property "+prop+" in "+bean.getClass());
		
		return getter.invoke(bean, new Object[] { key });
		//TODO: handle InvocationTargetException
	}
	
	public static void setNamedProperty(Object bean, BeanInfo beanInfo, String prop, String key, Object value) throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (beanInfo==null) beanInfo = Introspector.getBeanInfo(bean.getClass());
		
		NamedPropertyDescriptor pd = findNamedPropertyDescriptor(prop, beanInfo, false, true);
		
		Method setter = pd.getNamedWriteMethod();
		if (setter==null) throw new IntrospectionException("no indexed setter method found for property "+prop+" in "+bean.getClass());
		
		setter.invoke(bean, new Object[] { key, value });
		//TODO: handle InvocationTargetException
	}
	
	public static void addToBeanProperty(Object bean, BeanInfo beanInfo, String prop, Object value) throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (beanInfo==null) beanInfo = Introspector.getBeanInfo(bean.getClass());
		
		BagPropertyDescriptor pd = findBagPropertyDescriptor(prop, beanInfo, true, false);
		
		Method adder = pd.getAddMethod();
		if (adder==null) throw new IntrospectionException("no adder method found for property "+prop+" in "+bean.getClass());
		
		adder.invoke(bean, new Object[] { value });
		//TODO: handle InvocationTargetException
	}
	
	public static void configureBean(Object bean, BeanInfo beanInfo, Map<String, Object> properties) throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		if (beanInfo==null) beanInfo = Introspector.getBeanInfo(bean.getClass());

		for (Map.Entry<String,Object> entry: properties.entrySet()) {
			String propName = entry.getKey();
			
			int idx = propName.indexOf('.'); 
			if (idx >= 0) {
				String path = propName.substring(0,idx);
				propName = propName.substring(idx+1);
				
				bean = evaluatePath(bean, path);
				if (bean==null) 
					throw new NullPointerException("path "+path+" evaluates to null");

				beanInfo = null;
			}
			
			idx = propName.indexOf('#');
			if (idx>=0) {
				String key= propName.substring(idx+1);
				propName = propName.substring(0, idx);
				
				setNamedProperty(bean, beanInfo, propName, key, entry.getValue());
			}
			else if (propName.endsWith("+")) {
				propName = propName.substring(0, propName.length()-1);
				addToBeanProperty(bean, beanInfo, propName, entry.getValue());
			}
			else {
				setBeanProperty(bean, beanInfo, propName, entry.getValue());
			}
		}
	}

	public static Object createBean(Class cl, List<Object> parameters, Map<String, Object> properties) throws IntrospectionException, InstantiationException, IllegalAccessException, NoSuchMethodException, IllegalArgumentException, InvocationTargetException {
		Object[] params = parameters == null ? null : parameters.toArray();
		
		Constructor ctor = findConstructor(cl, params);
		Object bean = ctor.newInstance( params );
		//TODO: handle InvocationTargetException
		
		if (properties!=null && properties.size()>0) {
			BeanInfo beanInfo = Introspector.getBeanInfo(cl);
			configureBean(bean, beanInfo, properties);
		}
		
		return bean;
	}
	
	public static Object evaluateConstant(String qname) throws ClassNotFoundException, SecurityException, NoSuchFieldException, IntrospectionException, IllegalArgumentException, IllegalAccessException {
		int idx = qname.lastIndexOf('.');
		if (idx<0) throw new IllegalArgumentException("qname must contain a dot (.): "+qname);
		
		String name = qname.substring(idx+1);
		String cls = qname.substring(0, idx);
		
		Class c = Class.forName(cls); //TODO: ClassLoader, etc...
		Field f = c.getField(name);
		
		int mods = f.getModifiers();
		if (!Modifier.isStatic(mods)) throw new IntrospectionException("field "+name+" in class "+cls+" is not static");
		if (!Modifier.isPublic(mods)) throw new IntrospectionException("field "+name+" in class "+cls+" is not public");
		//TODO: check final?...
		
		Object v = f.get(null);
		return v;
	}

	public static Object evaluatePath(Object root, String qname) throws IllegalArgumentException, IntrospectionException, IllegalAccessException, InvocationTargetException {
		int idx = qname.indexOf('#');
		String key = null;
		
		if (idx>=0) {
			key = qname.substring(idx+1);
			qname = qname.substring(0, idx);
		}

		idx = qname.lastIndexOf('.');

		Object base = root;
		String name = qname;
		
		if (idx>=0) {
			name = qname.substring(idx+1);
			String path = qname.substring(0, idx);
			
			base = evaluatePath(root, path);
			if (base==null) 
				throw new NullPointerException("path "+path+" evaluates to null");
		}

		if (key == null) {
			return getBeanProperty(base, null, name);
		}
		else {
			return getNamedProperty(base, null, name, key);
		}
	}

	public static Class[] collectInterfaces(Class clazz, Class... classes) {
		List<Class> interfaces = new ArrayList<Class>();
		collectInterfaces(clazz, interfaces);
		for (Class curClass: classes) {
			interfaces.add(curClass);
		}
		return (Class[]) interfaces.toArray(new Class[interfaces.size()]);
	}

	protected static void collectInterfaces(Class clazz, List<Class> into) {
		if (clazz==Object.class) return;
		if (clazz.isPrimitive()) return;
		if (clazz.isArray()) return;
		
		Class[] interfaces = clazz.getInterfaces();
		for (int i = 0; i < interfaces.length; i++) {
			if (!into.contains(interfaces[i])) {
				into.add(interfaces[i]);
				collectInterfaces(interfaces[i], into);
			}
		}
		
		if (!clazz.isInterface()) {
			Class p = clazz.getSuperclass();
			if (p!=null && !into.contains(p)) {
				collectInterfaces(p, into);
			}
		}
	}
	
	public static Method getMethod(Class cls, String name, Class... paramTypes) throws SecurityException, NoSuchMethodException {
		Method m = cls.getMethod(name, paramTypes);
		
		return getDefinedMethod(m);
	}
	
	public static Method getDefinedMethod(Method m) throws SecurityException {
		if (!Modifier.isPublic(m.getModifiers())) return m;
		
		Class c = m.getDeclaringClass();
		String name = m.getName();
		Class[] paramTypes = m.getParameterTypes();
		
		while (c!=null) {
			Class[] ii = c.getInterfaces();
			for (Class i : ii) {
				try {
					Method n = i.getMethod(name, paramTypes);
					if (Modifier.isPublic(n.getModifiers())) return n;
				}
				catch (NoSuchMethodException ex) {
					//ignore
				}
			}

			if (!Modifier.isPublic(c.getModifiers())) {
				c = c.getSuperclass();
				continue;
			}

			try {
				Method n =  c.getMethod(name, paramTypes);
				if (Modifier.isPublic(n.getModifiers())) return n;
			}
			catch (NoSuchMethodException ex) {
				//ignore
			}
			
			c = c.getSuperclass();
		}
		
		return m;
	}
	
	public static <T> T cloneObject(T obj) throws CloneNotSupportedException {
		if (!(obj instanceof Cloneable)) 
			throw new CloneNotSupportedException("clone not supported by "+obj.getClass());
		
		try {
			Class c = obj.getClass();
			Method m = getMethod(c,"clone", new Class[0]); 
			
			T clone = (T)m.invoke(obj, new Object[0]);
			return clone;
		} catch (IllegalAccessException e) {
			throw new CloneNotSupportedException("clone not public in "+obj.getClass());
		} catch (NoSuchMethodException e) {
			throw new Error("method clone() not found");
		} catch (InvocationTargetException e) {
			Throwable tx = e.getTargetException();

			if (tx instanceof Error) throw (Error)tx;
			if (tx instanceof RuntimeException) throw (RuntimeException)tx;
			if (tx instanceof CloneNotSupportedException) throw new Error("unexpected exception: clone not supported by cloneable "+obj.getClass(), e);
			throw new Error("unexpected exception in clone() of "+obj.getClass(), e);
		}
	}

	public static List<String> getPropertyNames(Object bean, BeanInfo info) throws IntrospectionException {
		if (info==null) info = Introspector.getBeanInfo(bean.getClass());
		
		
		PropertyDescriptor[] properties = info.getPropertyDescriptors();
		List<String> names = new ArrayList<String>(properties.length);
		for (int i = 0; i < properties.length; i++) {
			String n = properties[i].getName();
			
			names.add(n);
		}
		
		return names;
	}
	
	public String capitalize(String n) {
		if (!Character.isLowerCase(n.charAt(0))) return n;
		else return n.substring(0, 1).toUpperCase() + n.substring(1); 
	}

	public static <T extends FeatureDescriptor>Map<String, T> makeFeatureMap(T[] features) {
		Map<String, T> m = new HashMap<String, T>();
		return fillFeatureMap(features, m);
	}

	public static <T extends FeatureDescriptor>Map<String, T> fillFeatureMap(T[] features, Map<String, T> m) {
		for (T d: features) {
			String n = Introspector.decapitalize(d.getName());
			m.put(n, d);
		}
		
		return m;
	}

	public static void main(String[] args) throws SecurityException, NoSuchMethodException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		Properties p = new Properties();
		p.setProperty("foo", "bar");
		
		Iterator it = p.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry e = (Map.Entry)it.next();
			Class c = e.getClass();
			Method m = getMethod(c, "getKey");
			Object k = m.invoke(e);
			System.out.println(k);
		}
	}
}
