package de.brightbyte.spa;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeoutException;

import de.brightbyte.abstraction.Disposer;
import de.brightbyte.abstraction.Instantiator;

public class SimplePool<T> implements Pool<T> {

	protected static class Entry<T> {
		public final T object;
		public long timestamp;

		public boolean leased;
		
		public Entry(T object) {
			super();
			this.object = object;
			this.leased = false;
		}

		@Override
		public int hashCode() {
			final int PRIME = 31;
			int result = 1;
			result = PRIME * result + ((object == null) ? 0 : object.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final Entry other = (Entry) obj;
			if (object == null) {
				if (other.object != null)
					return false;
			} else if (!object.equals(other.object))
				return false;
			return true;
		}
		
		private void touch() {
			timestamp = System.currentTimeMillis();
		}
		
		public void leased() {
			leased = true;
			touch();
		}
		
		public void released() {
			leased = false;
			touch();
		}
		
	}
	
	protected int minIdle = 0;
	protected int minTotal = 0;
	protected int maxIdle = 10;
	protected int maxTotal = 100;
	protected int keepAlive = 60;
	
	protected Instantiator<T> instantiator;
	protected Disposer<T> disposer; 
	
	protected BlockingQueue<Entry<T>> idle = new LinkedBlockingQueue<Entry<T>>();
	protected Set<Entry<T>> busy = new HashSet<Entry<T>>();
	protected Map<T, Entry<T>> entries = new HashMap<T, Entry<T>>();
	
	protected Object lock = new Object();
	
	public SimplePool(Instantiator<T> instantiator, Disposer<T> disposer) {
		if (instantiator==null) throw new NullPointerException();
		
		if (disposer==null && instantiator instanceof Disposer) {
			disposer = (Disposer<T>)instantiator;
		}
		
		this.instantiator = instantiator;
		this.disposer = disposer;
	}

	
	public int getKeepAlive() {
		return keepAlive;
	}


	public void setKeepAlive(int keepAlive) {
		if (keepAlive<0) throw new IllegalArgumentException(keepAlive+" < 0");
		this.keepAlive = keepAlive;
	}


	public int getMaxIdle() {
		return maxIdle;
	}


	public void setMaxIdle(int maxIdle) {
		if (maxIdle<0) throw new IllegalArgumentException(maxIdle+" < 0");
		this.maxIdle = maxIdle;
	}


	public int getMaxTotal() {
		return maxTotal;
	}


	public void setMaxTotal(int maxTotal) {
		if (maxTotal<1) throw new IllegalArgumentException(maxTotal+" < 1");
		this.maxTotal = maxTotal;
	}


	public int getMinIdle() {
		return minIdle;
	}


	public void setMinIdle(int minIdle) {
		if (minIdle<0) throw new IllegalArgumentException(minIdle+" < 0");
		this.minIdle = minIdle;
	}


	public int getMinTotal() {
		return minTotal;
	}


	public void setMinTotal(int minTotal) {
		if (minTotal<0) throw new IllegalArgumentException(minTotal+" < 0");
		this.minTotal = minTotal;
	}


	public T lease(int timeout) throws InterruptedException, InstantiationException, TimeoutException {
		Entry<T> e = leaseEntry(timeout);
		return e.object;
	}
	
	protected Entry<T> leaseEntry(int timeout) throws InterruptedException, InstantiationException, TimeoutException {
		long t = timeout==0 ? Long.MAX_VALUE : timeout;
		Entry<T> e = null;
	
		synchronized (lock) {
			while (e==null) {
				e = idle.poll();
				if (e!=null) break;
				
				if (busy.size()<maxTotal) {
					e = newEntry(); //XXX: might be low...
				}
				else {
					if (t<=0) throw new TimeoutException("all resources busy");
					
					long from = System.currentTimeMillis();
					lock.wait(t);
					t -= (System.currentTimeMillis() - from);
				}
			}
				
			e.leased();
			busy.add(e);
		}
		
		return e;
	}
	
	protected Entry<T> makeEntry(T rc) {
		return new Entry<T>(rc);
	}
	
	protected Entry<T> newEntry() throws InstantiationException {
		T obj = instantiator.newInstance(); //NOTE: keep creation outside sync

		synchronized (lock) {
			Entry<T> e = makeEntry(obj);
			entries.put(obj, e);
			return e;
		}
	}

	public void release(T obj) {
		releaseEntry(obj);
	}
	
	protected Entry<T> releaseEntry(T obj) {
		synchronized (lock) {
			Entry<T> e = entries.get(obj);
			if (e==null) throw new IllegalArgumentException("not registered with this pool: "+obj);
	
			busy.remove(e);
			e.released();
			
			if (keepAlive<=0 || idle.size()>=maxIdle) {
				remove(e.object);
			}
			else {
				idle.add(e);
			}

			lock.notify();
			
			return e;
		}		
	}
	
	public void remove(T obj) {
		synchronized (lock) {
			Entry<T> e = entries.remove(obj);
			if (e==null) throw new IllegalArgumentException("not registered with this pool: "+obj);
	
			busy.remove(e);
			idle.remove(e);
			e.released();
			
			if (disposer != null) disposer.dispose(e.object);
			
			lock.notify();
		}
	}
	
	public void janitize() throws InstantiationException {
		boolean add;
		
		synchronized (lock) {
			long now = System.currentTimeMillis();
			
			//remove old idle resources
			Iterator<Entry<T>> it = idle.iterator();
			while (it.hasNext()) {
				Entry<T> e = it.next();
				long t = now - e.timestamp;
				if (t>keepAlive*1000L) {
					it.remove();
					remove(e.object);
				}
			}
			
			//remove excess idle resources (cvan this even happen?)
			while (idle.size()>maxIdle) {
				Entry<T> e = idle.remove();
				remove(e.object);
			}
			
			//check if we want to create resources preemtively
			add = (idle.size()<minIdle || idle.size()+busy.size()<minTotal);
		}

		//NOTE: do actual creation of new entries unsynchronized, because it may be slow.
		//      might end up with one too many, but that's not real problem.
		while (add) {
			Entry<T> e = newEntry();
			synchronized (lock) {
				idle.add(e);
				lock.notify();
				
				add = (idle.size()<minIdle || idle.size()+busy.size()<minTotal);
			}
		}
		
	}

}
