package de.brightbyte.net;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import de.brightbyte.util.StringUtils;

public class UrlUtils {
	
	//TODO: encodeCritical(String s) for encoding [?&=#] only
	
	public static String encode(String s) {
		try {
			return URLEncoder.encode(s, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			throw new Error("UTF-8 not supported");
		}
	}
	
	protected static final char[] hex = "0123456789ABCDEF".toCharArray();

	public static final String criticalChars = "%?&#=";
	
	public static String encodeCritical(String s, String critical) {
		StringBuilder b = null;
		
		int c = s.length();
		for (int i=0; i<c; i++) {
			char ch = s.charAt(i);
			
			if (critical.indexOf(ch)>=0) {
				int cv = (int)ch;
				if (cv>127) throw new IllegalArgumentException("encodeCritical() can not be used to encode non-asscii characters! Found code 0x"+Integer.toHexString(cv));
				if (b==null) b = new StringBuilder(s.substring(0, i));
				
				int hi = (cv & 0xF0) >>> 4;
				int lo = (cv & 0x0F);
				b.append('%');
				b.append(hex[hi]);
				b.append(hex[lo]);
			}
			else if (b!=null) {
				b.append(ch);
			}
		}
		
		if (b==null) return s;
		else return b.toString();
	}

	public static String decode (String text) {
		return decode(text, true);
	}
	
	public static String decode (String text, boolean allowControlChars) {
		//NOTE: this is more lenient than URLDecoder.decode!
		//TODO: test case!
		
		if (text==null) return null;
		if (text.length()<3 || StringUtils.indexOf('%', text)<0) return text;
		
		try {
			byte[] a = text.getBytes("UTF-8"); //original bytes //XXX: rough toString
			byte[] b = new byte[a.length];     //target buffer 
			byte x = 0; //accumulator for encoded byte
			int i = 0;  //index on original bytes
			int c = 0;  //size/index on target buffer
			int st = 0; //parser state (bytes into encoded section)
			
			for (i=0; i<a.length; i++) {
				if (st==0) { //normal bytes
					if (a[i] == '%') st = 1; //start of potential code section 
					else b[c++] = a[i];
				}
				else if (st==1) { //first digit
					if (a[i] >= '0' && a[i] <= '9') {
						st = 2;
						x = (byte)((a[i] - '0') << 4); 
					}
					else if (a[i] >= 'A' && a[i] <= 'F') {
						st = 2;
						x = (byte)((a[i] - 'A' + 10) << 4); 
					}
					else if (a[i] >= 'a' && a[i] <= 'f') {
						st = 2;
						x = (byte)((a[i] - 'a' + 10) << 4); 
					}
					else {
						x = 0;
						b[c++] = a[i - st--];
						b[c++] = a[i];
					}
				}
				else if (st==2) { //second digit
					if (a[i] >= '0' && a[i] <= '9') {
						st = 0;
						x += (a[i] - '0'); 
						if (allowControlChars || x>=32 || x<0) b[c++] = x; //NOTE: codes < 32 are control chars (also true in utf-8) 
					}
					else if (a[i] >= 'A' && a[i] <= 'F') {
						st = 0;
						x += (a[i] - 'A' + 10); 
						if (allowControlChars || x>=32 || x<0) b[c++] = x; //NOTE: codes < 32 are control chars (also true in utf-8)
					}
					else if (a[i] >= 'a' && a[i] <= 'f') {
						st = 0;
						x += (a[i] - 'a' + 10); 
						if (allowControlChars || x>=32 || x<0) b[c++] = x; //NOTE: codes < 32 are control chars (also true in utf-8)
					}
					else {
						b[c++] = a[i - st--];
						b[c++] = a[i - st--];
						b[c++] = a[i];
					}
					
					x = 0;
				}
				else {
					throw new Error("oops");
				}
			}
			
			while (st>0) {
				b[c++] = a[i - st--];
			}
			
			return new String(b, 0, c, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			throw new Error("UTF-8 not supported", e);
		}
	}

	public static String appendParameters(String base, Map<String, ? extends Object>... params) {
		if (base.indexOf('?')>=0) base += "&";
		else base += "?";
		
		return base + joinParameters(params);
	}
	
	public static String joinParameters(Map<String, ? extends Object>... params) {
		StringBuilder b = new StringBuilder();

		//XXX: merge maps first?
		
		char separator = '\0';
		for (Map<String, ? extends Object> p: params) {
			appendParameters(b, p, separator);
			if (b.length()>0) separator = '&';
		}
		
		return b.toString();
	}
	
	public static void appendParameters(StringBuilder builder, Map<String, ? extends Object> params) {
		appendParameters(builder, params, builder.indexOf("?")>=0 ? '&' : '?');
	}
	
	public static void appendParameters(StringBuilder builder, Map<String, ? extends Object> params, char separator) {
		try {
			appendParameters((Appendable)builder, params, separator);
		} catch (IOException e) {
			throw new Error("unexpected IOException while appending to StringBuilder");
		}
	}
	
	@SuppressWarnings("unchecked")
	public static void appendParameters(Appendable appendable, Map<String, ? extends Object>  params, char separator) throws IOException {
		String[] b = null;
		
		for (Map.Entry<String, ? extends Object> e: params.entrySet()) {
			String k = (String)e.getKey();
			Object v = e.getValue();
			
			if (v instanceof String) {
				if (b==null) b = new String[1];
				b[0] = (String)v;
				v = b;
			}
			
			for (String s: (String[])v) {
				if (separator!='\0') appendable.append(separator);
				separator = '&';
					
				appendable.append(encode(k));
				appendable.append('=');
				if (v!=null) appendable.append(encode(s));
			}
		}
	}
	
	public static Map<String, String[]> completeParameterMap(Map<String, String> p) {
		if (p==null) return null;
		if (p.isEmpty()) return Collections.emptyMap();
		
		Map<String, String[]> m = new HashMap<String, String[]>(p.size());
		for (Map.Entry<String, String> e: p.entrySet()) {
			String s = e.getValue();
			String[] v;
			
			if (s==null) v = new String[] {};
			else v = new String[] { s };
			
			m.put(e.getKey(), v);
		}
		
		return m;
	}
	
	public static Map<String, String> simplifyParameterMap(Map<String, String[]> p) {
		if (p==null) return null;
		if (p.isEmpty()) return Collections.emptyMap();
		
		Map<String, String> m = new HashMap<String, String>(p.size());
		for (Map.Entry<String, String[]> e: p.entrySet()) {
			String[] v = e.getValue();
			String s;
			
			if (v==null) continue;
			else if (v.length==0) s = "";
			else s = v[0];
			
			m.put(e.getKey(), s);
		}
		
		return m;
	}
	
	public static Map<String, String[]> splitParameters(String q) {
		try {
			return splitParameters(q, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			throw new Error("UTF-8 not supported");
		}
	}
	
	public static Map<String, String[]> splitParameters(String q, String encoding) throws UnsupportedEncodingException {
		if (q==null) return null;
		if (q.length()==0) return Collections.emptyMap();

		String[] qq = q.split("&");
		Map<String, String[]> m = new HashMap<String, String[]>(qq.length);
		
		for (String s: qq) {			
			String k;
			String v;
			
			int idx = s.indexOf('=');
			if (idx>=0) {
				k = s.substring(0, idx);
				v = s.substring(idx+1);
			}
			else {
				k = s;
				v = "";
			}
			
			k = URLDecoder.decode(k, encoding);
			v = URLDecoder.decode(v, encoding);
			
			String[] a = (String[])m.get(k);
			if (a!=null) {
				//NOTE: this is a bit inefficient generally, but considering that multi-value
				//      params are rare, and don't usually have *many* values, it should be reasonable.
				a = new String[ a.length + 1 ]; 
				a[a.length-1] = v;
			}
			else {
				a = new String[] { v };
			}
			
			m.put(k, a);
		}
		
		return m;
	}
}
