package de.brightbyte.net;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketException;

public class UdpReceiver {
	public static interface Sink {
		public void packetReceived(UdpReceiver receiver, DatagramPacket packet);
		public void errorOccurred(UdpReceiver receiver, IOException ex);
		public void socketClosed(UdpReceiver receiver);
	}
	
	public static class StreamSink implements Sink {
		protected OutputStream out;
		protected byte[] prefix;
		protected byte[] suffix;
		
		protected IOException receiverError;
		protected IOException streamError;
		
		public StreamSink(OutputStream out, byte[] prefix, byte[] suffix) {
			if (out==null) throw new NullPointerException();
			
			this.out = out;
			this.prefix = prefix;
			this.suffix = suffix;
		}

		public void errorOccurred(UdpReceiver receiver, IOException ex) {
			receiverError = ex;
		}

		public void packetReceived(UdpReceiver receiver, DatagramPacket packet) {
			try {
				if (prefix!=null && prefix.length>0) out.write(prefix);
				out.write(packet.getData(), packet.getOffset(), packet.getLength());
				if (suffix!=null && suffix.length>0) out.write(suffix);
			} catch (IOException ex) {
				streamError = ex;
			}
		}

		public void socketClosed(UdpReceiver receiver) {
			try {
				out.close(); //xxx: really? optional?
			} catch (IOException ex) {
				streamError = ex;
			}
		}

		public IOException getReceiverError() {
			return receiverError;
		}

		public IOException getStreamError() {
			return streamError;
		}
		
	}
	
	protected int packetBufferSize = 1024;
	protected SocketAddress address;
	
	private volatile boolean running = false; 
	
	protected DatagramSocket socket;
	protected Thread worker;
	protected Sink sink;
	
	public UdpReceiver(SocketAddress address, Sink sink) {
		if (address==null) throw new NullPointerException();
		if (sink==null) throw new NullPointerException();
		
		this.address = address;
		this.sink = sink;
	}
	
	public void bind() throws SocketException {
		if (isBound()) throw new IllegalStateException("already bound");
		socket = new DatagramSocket(address);
	}
	
	public synchronized boolean isBound() {
		return socket != null && socket.isBound() && !socket.isClosed();
	}
	
	public void close() {
		boolean closed = false;
		
		synchronized (this) {
			if (worker!=null) worker.interrupt();
			
			if (socket!=null) {
				socket.close();
				closed = true;
			}
			
			socket = null;
		}
		
		if (closed) sink.socketClosed(this);
	}
	
	public void run() throws IOException {
		if (sink==null) throw new IllegalStateException("no sink specified");
		if (!isBound()) bind();
		
		running = true;
		
		try {
			byte[] buffer = new byte[packetBufferSize];
			DatagramPacket packet = new DatagramPacket(buffer, buffer.length); 
			while (true) {
				socket.receive(packet);
				
				sink.packetReceived(this, packet);
			}
		}
		catch (IOException ex) {
			if (ex instanceof InterruptedIOException && Thread.currentThread().isInterrupted()) /* ignore */;
			else sink.errorOccurred(this, ex);
		}
		finally {
			running = false;
			close();
		}
	}
	
	public synchronized boolean isRunning() {
		return running;
	}
	
	public synchronized void start() {
		if (isRunning()) throw new IllegalStateException("already running");
		if (!isBound()) throw new IllegalStateException("not bound");
		
		worker = new Thread("UdpReceiver#"+hashCode()) {
			public void run() {
				try {
					UdpReceiver.this.run();
				} catch (IOException e) {
					// ignore, sink has already been informed
				}
			}
		};
		
		worker.start();
	}

	public int getPacketBufferSize() {
		return packetBufferSize;
	}

	public void setPacketBufferSize(int packetBufferSize) {
		this.packetBufferSize = packetBufferSize;
	}

	public SocketAddress getAddress() {
		return address;
	}

	public SocketAddress getLocalSocketAddress() {
		return socket.getLocalSocketAddress();
	}

	public int getReceiveBufferSize() throws SocketException {
		return socket.getReceiveBufferSize();
	}

	public void setReceiveBufferSize(int size) throws SocketException {
		socket.setReceiveBufferSize(size);
	}
	
	
	public static void main(String[] args) throws IOException {
		int port = Integer.parseInt(args[0]);
		SocketAddress a = new InetSocketAddress(port);
		
		StreamSink sink = new StreamSink(System.out, null, null); 
		
		UdpReceiver receiver = new UdpReceiver(a, sink);
		receiver.run();
		receiver.close();
	}
}
