package de.brightbyte.job;

import java.util.ArrayList;
import java.util.List;

public class Progress {
	
	public static final int NONE = 0;
	public static final int START = 1;
	public static final int PROGRESS = 5;
	public static final int DONE = 10;
	public static final int FAILED = 100;
	
	public static String getTypeText(int type) {
		switch (type) {
			case NONE: return "NONE";
			case START: return "START";
			case PROGRESS: return "PROGRESS";
			case DONE: return "DONE";
			case FAILED: return "FAILED";
		}
		
		throw new IllegalArgumentException("not a valid progress event type code: "+type);
	}
	
	public static class Event {
		protected int type;
		protected Source source;
		protected String item;
		protected long total;
		protected long position;
		protected Throwable error;
				
		public Event(int type, Source source, String item, long total, long position, Throwable error) {
			super();
			this.type = type;
			this.source = source;
			this.item = item;
			this.total = total;
			this.position = position;
			this.error = error;
		}
		
		public Throwable getError() {
			return error;
		}
		
		public long getPosition() {
			return position;
		}
		
		public Source getSource() {
			return source;
		}
		
		public String getItem() {
			return item;
		}
		
		public long getTotal() {
			return total;
		}
		
		public int getType() {
			return type;
		}
		
		public void deliver(Listener li) {
			li.progress(this);
		}
		
		public String toString() {
			return getClass().getName()+": ["+type+"]: "+item+" at "+position+"/"+total;
		}
	}

	public static interface Listener {
		public void progress(Event event);
	}
	
	public static interface Source {
		public void addProgressListener(Listener li);
		public void removeProgressListener(Listener li);
	}
	
	public interface Monitor {
		public void fireStep(long step);
		public void fireStart(long total);
		public void fireDone(long total);
		public void fireFailed(Throwable ex);
	}
	
	public static abstract class AbstractMonitor implements Monitor {
		protected String item;
		protected Source source;
		private long total;
		private long position;
		private Throwable error;
		private int status;
		private boolean ignoreDone = false;
		private boolean freezeTotal;
		
		public AbstractMonitor(Source source, String item) {
			super();
			this.item = item;
			this.source = source;
		}
		
		public void setTotal(long length, boolean freeze) {
			this.total = length;
			this.freezeTotal = freeze;
		}
		
		public void reset() {
			total = -1;
			position = -1;
			status = NONE;
			error = null;
			freezeTotal = false;
		}

		protected void fire() {
			Event e = new Event(this.status, this.source, this.item, this.total, this.position, this.error);
			fire(e);
		}
		
		protected abstract void fire(Event e);

		public void fireStart(long total) {
			if (status != NONE) return;
			//if (status != NONE) throw new IllegalStateException("already started");
			
			this.position = 0;
			if (total>=0 && !freezeTotal) this.total = total;
			this.status = START;
			
			fire();
		}

		public void fireStep(long step) {
			fireProgress(position + step, -1);
		}
		
		public void fireProgress(long pos) {
			fireProgress(pos, -1);
		}
		
		public void fireProgress(long pos, long total) {
			if (isTerminated()) throw new IllegalStateException("terminated");
			if (status == NONE) fireStart(total);
			//if (status != START && status != PROGRESS) throw new IllegalStateException("not running");
			
			if (total>=0 && !freezeTotal) this.total = total;
			if (pos>=0) this.position = pos;
			this.status = PROGRESS;
			
			fire();
		}
		
		public void fireDone(long total) {
			if (ignoreDone) {
				fireProgress(-1, total);
				return;
			}
			//if (status != START && status != PROGRESS) throw new IllegalStateException("not running");
			
			if (total>=0 && !freezeTotal) this.total = total;
			this.position = this.total;
			this.status = DONE;
			
			fire();
		}

		public void fireFailed(Throwable error) {
			if (status == DONE || status == FAILED) return;
			//if (status == DONE || status == FAILED) throw new IllegalStateException("already finished");
			
			this.status = FAILED;
			this.error = error;
			
			fire();
		}

		public Throwable getError() {
			return error;
		}

		public String getItem() {
			return item;
		}

		public long getPosition() {
			return position;
		}

		public Source getSource() {
			return source;
		}

		public int getStatus() {
			return status;
		}

		public long getTotal() {
			return total;
		}

		public boolean isRunning() {
			return status == START || status == PROGRESS;
		}

		public boolean isTerminated() {
			return status == Progress.DONE || status == Progress.FAILED;
		}
		
		public void setIgnoreDone(boolean ignore) {
			ignoreDone = ignore;
		}
	}
	
	public static class GenericMonitor extends AbstractMonitor {
		protected Listener listener;
		
		public GenericMonitor(Source source, String item, Listener listener) {
			super(source, item);
			this.listener = listener;
		}
		
		public void fire(Event ev) {
			if (listener==null) return;
			listener.progress(ev);
		}
		
	}

	public static class State extends AbstractMonitor implements Source {
		protected List<Listener> listeners;
		
		public State(Source source, String item) {
			super(source, item);
			if (this.source==null) this.source = this;
		}

		public void addProgressListener(Listener li) {
			if (li==null) return;
			if (listeners==null) listeners = new ArrayList<Listener>();
			listeners.add(li);
		}
		
		public void removeProgressListener(Listener li) {
			if (li==null) return;
			if (listeners==null) return;
			listeners.remove(li);
			if (listeners.size()==0) listeners = null;
		}
		
		protected void fire() {
			if (listeners==null) return;
			super.fire();
		}
		
		public void fire(Event ev) {
			if (listeners==null) return;
			
			Listener[] ll = (Listener[]) listeners.toArray(new Listener[listeners.size()]); //TODO: optionally sync...
			
			for (int i = 0; i < ll.length; i++) {
				try {
					ev.deliver(ll[i]); 
				}
				catch (RuntimeException ex) {
					ex.printStackTrace(); //FIXME: generic reporting interface
				}
			}
		}
	}
	
}
