package de.brightbyte.job;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executor;

import de.brightbyte.io.Output;
import de.brightbyte.io.PrintStreamOutput;

public class NiceThreadPoolExecutor implements Executor {

	protected Output log;

	public interface Job extends Runnable {

		public abstract String getResource();
		
	}

	protected class Worker extends Thread {

		private long since = 0;
		private boolean idle = false;
		private Job job;
		
		public Worker() {
			since = System.currentTimeMillis();
		}
		
		public void run() {
			try {
				Job j;
				synchronized (this) {
					j = job;
				}
				
				while (!isInterrupted()) {
					if (j!=null) {
						try {
							j.run();
						}
						catch (RuntimeException ex) {
							ex.printStackTrace();
						}
						finally {
							jobFinished(j, this);							
						}
					}
					
					Job nextJob = getNextJob(); //note: outside sync block
					synchronized (this) {
						job = nextJob;
						
						if (job!=null) since = System.currentTimeMillis();
						else while (job==null) idle();

						j = job;
					}
				}
			} catch (InterruptedException e) {
				//interrupted...
			} /* finally {
				synchronized (jobQueue) {
					workers.remove(this);
					log("removed thread "+getName()+" on termination");
					
					if (shutdown && workers.isEmpty() && ) {
						terminated = true;
						jobQueue.notify();
						log("terminated");
					}
				}
			}*/
		}
		
		protected synchronized void idle() throws InterruptedException {
			idle = true;
			since = System.currentTimeMillis();
			log("thread "+getName()+" went idle");
			wait();
			idle = false;
			since = System.currentTimeMillis();
			log("thread "+getName()+" now busy");
			
		}

		public synchronized void assign(Job job) {
			if (!isIdle()) throw new IllegalArgumentException();
			
			this.job = job;
			
			//FIXME: in-limbo; real "busy" state should be set inside worker thread.
			idle = false;
			since = System.currentTimeMillis();

			log("thread "+getName()+" assigned job");
			
			notify();
		}

		public synchronized boolean isIdle() {
			return idle;
		}
		
		public synchronized long getIdleTime() {
			if (!isIdle()) return -1;
			else return Math.max(System.currentTimeMillis() - since, 1);
		}
		
		public synchronized long getBusyTime() {
			if (isIdle()) return -1;
			else return Math.max(System.currentTimeMillis() - since, 1);
		}
		
	}

	protected LinkedList<Job> jobQueue = new LinkedList<Job>(); //TODO: use blocking queue. or polling?
	protected HashSet<Worker> workers = new HashSet<Worker>(); //TODO: use blocking queue. or polling?
	protected HashSet<Object> resourceLocks = new HashSet<Object>();
	
	
	protected long maxIdleTime = 60 * 1000;
	protected long jobQueuePatience = 10 * 1000;
	protected int minWorkers = 4;
	protected int maxWorkers = 16;
	
	private boolean shutdown = false;
	private boolean terminated = true; 
	
	public void execute(Runnable command) {
		Job job = (Job)command;
		
		synchronized (jobQueue) {
			Worker w = null;
			
			String rc = job.getResource();
			if (rc!=null && !resourceLocks.contains(rc)) {
				w = getIdleWorker();
			}
			
			if (w==null) jobQueue.add(job);
			else assignJob(job, w);
		}
	}
	
	protected void assignJob(Job job, Worker worker) {
		synchronized (jobQueue) {
			String rc = job.getResource();
			if (rc!=null) resourceLocks.add(rc);
			worker.assign(job);
		}
	}
	
	protected void jobFinished(Job job, Worker worker) {
		synchronized (jobQueue) {
			String rc = job.getResource();
			if (rc!=null) resourceLocks.remove(rc);
			
			if (shutdown && jobQueue.isEmpty()) {
				terminated = true;
				jobQueue.notifyAll();
			}
		}
	}
	
	protected Worker getIdleWorker() {
		synchronized (jobQueue) {
			for (Worker w : workers) {
				if (w.isIdle() && w.isAlive()) return w;
			}
			
			return null;
		}
	}

	protected Job getNextJob() {
		synchronized (jobQueue) {
			for (Job job : jobQueue) {
				String rc = job.getResource();
				if (rc==null || !resourceLocks.contains(rc)) {
					jobQueue.remove(job);
					return job;
				}
			}
			
			return null;
		}
	}
	
	protected int checkWorkers() {
		synchronized (jobQueue) {
			long minBusy = Integer.MAX_VALUE;
			long maxIdle = 0;
			int busy = 0;
			
			Iterator<Worker> it = workers.iterator();
			while (it.hasNext()) {
				Worker w = it.next();
				if (!w.isAlive()) {
					log("removing dead thread");
					it.remove();
				}
				else if (w.isIdle()) {
					long t = w.getIdleTime();
					log(w.getName()+" idle since "+t+"ms");
					if (t>=0) {
						if (t > maxIdleTime && workers.size() > minWorkers) {
							w.interrupt();
							it.remove();
							log("killed idle worker after "+t+"ms");
						}
						else {
							maxIdle = Math.max(maxIdle, t);
						}
					}
				}
				else {
					long t = w.getBusyTime();
					log(w.getName()+" busy since "+t+"ms");
					if (t>=0) {
						minBusy = Math.min(minBusy, t);
						busy ++;
					}
				}
			}
			
			if (shutdown && jobQueue.isEmpty()) {
				terminated = true;
				jobQueue.notifyAll();
			}
			else {
				if (shutdown && workers.isEmpty()) {
					terminated = true;
					jobQueue.notify();
					log("terminated");
				}
	
				if (!shutdown &&
						workers.size() < maxWorkers 
						&& busy > 0 
						&& jobQueue.size()>0 
						&& minBusy > jobQueuePatience) {
					startWorker();
				}
	
				while (!shutdown && workers.size() < minWorkers) {
					startWorker();
				}
			}
			
			log("Queue: "+jobQueue.size()+"; Workers: "+busy+"/"+workers.size()+" busy, "+jobQueue.size()+" queued, maxIdle = "+maxIdle+"ms, minBusy = "+minBusy+"ms");
			
			return busy + jobQueue.size();
		}
	}
	
	public void janitorLoop(int interval) throws InterruptedException {
		while (checkWorkers() > 0 && !shutdown) { //FIXME: exits too early, if workers have not yet picked up new jobs.
			Thread.sleep(interval);
		}
	}
	
	protected void startWorker() {
		synchronized (jobQueue) {
			Worker w = new Worker();
			w.start();
			workers.add(w);
			log("started worker");
		}
	}
	
	protected void log(String msg) {
		if (log!=null) log.println(msg);
	}

	public void setLog(Output log) {
		this.log = log;
	}

	public void awaitTermination() throws InterruptedException {
		synchronized (jobQueue) {
			while (!terminated) {
				jobQueue.wait();
			}

			log("awaitTermination complete");
		}
	}

	public void shutdown() {
		synchronized (jobQueue) {
			shutdown = true;
		}

		log("shutdown");
	}
	
	public static void main(String[] args) throws InterruptedException {
		NiceThreadPoolExecutor executor = new NiceThreadPoolExecutor();
		
		executor.log = new PrintStreamOutput(System.out);
		
		executor.maxIdleTime = 3 * 1000;
		executor.jobQueuePatience = 1 * 1000;
		executor.maxWorkers = 5;
		executor.minWorkers = 2;
		
		class TestJob implements Job {
			
			protected String resource;
			protected int delay;
			
			public TestJob(int delay, String resource) {
				this.resource = resource;
				this.delay = delay;
			}

			public String getResource() {
				return resource;
			}

			public void run() {
				System.out.println("using "+resource+"...");
				try {
					Thread.sleep(delay);
					System.out.println("done using "+resource+".");				
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			
		};
		
		executor.execute( new TestJob(2000, "one") );
		executor.execute( new TestJob(2000, "two") );
		executor.execute( new TestJob(2000, "three") );
		executor.execute( new TestJob(2000, "three") );
		executor.execute( new TestJob(2000, "four") );
		
		executor.execute( new TestJob(600, "five") );
		executor.execute( new TestJob(600, "six") );
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "eight") );
		executor.execute( new TestJob(600, "nine") );
		executor.execute( new TestJob(600, "ten") );
		
		executor.execute( new TestJob(600, "five") );
		executor.execute( new TestJob(600, "six") );
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "eight") );
		executor.execute( new TestJob(600, "nine") );
		executor.execute( new TestJob(600, "ten") );
		
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "seven") );
		executor.execute( new TestJob(600, "seven") );
		
		executor.execute( new TestJob(600, "ten") );
		executor.execute( new TestJob(600, "ten") );
		executor.execute( new TestJob(600, "ten") );
		executor.execute( new TestJob(600, "ten") );
		executor.execute( new TestJob(600, "ten") );
		executor.execute( new TestJob(600, "ten") );
		
		executor.janitorLoop(300);
		executor.shutdown();
		executor.awaitTermination();
		System.exit(0);
	}

}
