package de.brightbyte.job;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import de.brightbyte.job.Progress.Listener;

public class JobQueue<J extends Job> extends ThreadPoolExecutor implements Progress.Source {
	protected ExecutorService executor;
	protected MultiProgress progress;
	private boolean aborted;
	
	public JobQueue(String id) {
		super(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>());
		progress = new MultiProgress(this, id); 
	}
	
	public void addJob(J job) {
		String item = job.getProgressItemId();
		if (item!=null) progress.addItem(item, job.getProgressTotal());
		
		//System.out.println(progress.getItem()+" queued item "+item+": "+job);
		submit((Callable<?>)job);
	}

	public void addProgressListener(Listener li) {
		if (li==null) return;
		progress.addProgressListener(li);
	}

	public void removeProgressListener(Listener li) {
		if (li==null) return;
		progress.addProgressListener(li);
	}
	
	public void resetProgress() {
		progress.reset();
	}

	public boolean isAborted() {
		return aborted;
	}

	public void shutdown() {
		if (isShutdown()) return;
		
		progress.close();
		super.shutdown();
	}
	
	public List<Runnable> shutdownNow() {
		if (isAborted()) return Collections.emptyList();
		
		aborted = true;
		progress.close();
		return super.shutdownNow();
	}

	public int getCompletedJobCount() {
		return progress.getCompletedItemCount();
	}

	public int getJobCount() {
		return progress.getItemCount();
	}

	public void close() {
		if (isShutdown() || isTerminated()) return;
		
		shutdown();
		progress.close();
	}

	@Override
	protected void terminated() {
		progress.close();
		
		if (!progress.isTerminated()) {
			if (isAborted()) progress.fireFailed(null);
			else progress.fireDone(-1);
		}
		
		super.terminated();
	}

	@Override
	protected void afterExecute(Runnable r, Throwable t) {
		if (t!=null) {
			System.err.println("Error in job "+r);
			t.printStackTrace(System.err); //FIXME: nice reporting!
		}
		super.afterExecute(r, t);
	}

	@Override
	protected void beforeExecute(Thread t, Runnable r) {
		super.beforeExecute(t, r);
	}

	@Override
	public void setCorePoolSize(int corePoolSize) {
		throw new UnsupportedOperationException("imutable");
	}

	@Override
	public void setKeepAliveTime(long time, TimeUnit unit) {
		throw new UnsupportedOperationException("imutable");
	}

	@Override
	public void setMaximumPoolSize(int maximumPoolSize) {
		throw new UnsupportedOperationException("imutable");
	}

	@Override
	public void setRejectedExecutionHandler(RejectedExecutionHandler handler) {
		throw new UnsupportedOperationException("imutable");
	}

	public long getProgressSteps() {
		return progress.getTotal();
	}

	public long getProgressPosition() {
		return progress.getPosition();
	}
		
}
