package de.brightbyte.io;

import java.io.PrintStream;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Fixed-size buffer for multiple consumers, each of which should receive all data.
 *  
 * Implemented as a ring buffer with a view for each consumer. 
 * Features a skipping mechanism to handle slow consumers.
 *  
 * @author daniel
 * @param <E>
 */
public class MultiStreamBuffer<E> {
	public static interface ViewMonitor<E> {
		public void skippedElement(View<E> view);
		public void removedElement(View<E> view);
		public void becameEmpty(View<E> view);
		public void becameAvailable(View<E> view);
		public void becameClosed(View<E> view);
	}
	
	public static interface View<E> {
		public int size();
		public boolean isEmpty();
		
		public E element() throws NoSuchElementException; 
		public E remove() throws NoSuchElementException; 
		public E poll(); 
		public E peek();
		
		public void close();
		
		public int getSkippedEntryCount();
		public int resetSkippedEntryCount();
		
		public void setMonitor(ViewMonitor<E> monitor);
		public boolean isClosed();
	}
	
	protected abstract static class ChainLink {
		private ChainLink previous;
		private ChainLink next;
		
		public ChainLink() {
			previous = this;
			next = this;
		}
		
		protected void unlink() {
			previous.next = next;
			next.previous = previous;
			
			next.checkChain(null);
			previous.checkChain(null);
			
			next = this;
			previous = this;
		}

		protected void insertAfter(ChainLink prev) {
			unlink();
			
			previous = prev;
			next = previous.next;
			
			previous.next = this;
			next.previous = this;
			
			checkChain(null);
		}

		protected void checkChain(Class stopClass) {
			ChainLink link = this;
			while (true) {
				if (link == null) throw new RuntimeException("broken chain (next==null)");
				if (link!=this && stopClass!=null && stopClass.isInstance(link)) throw new RuntimeException("broken chain (chained to wrong base!)");
				if (link.next.previous != link) throw new RuntimeException("broken chain (link.next.previous != link)");
				if (link.previous.next != link) throw new RuntimeException("broken chain (link.previous.next != link)");
				
				link = link.next;
				if (link==this) break;
			}
		}

		protected ChainLink getNextLink() {
			return next;
		}

		protected ChainLink getPreviousLink() {
			return previous;
		}

		protected void insertChainAfter(ChainLink n) {
			n.previous.next = this.next;
			this.next.previous = n.previous;
			
			n.previous = this;
			this.next = n;
			
			checkChain(null);
		}
	}
	
	protected abstract static class Handle<E> extends ChainLink implements View<E> {
		protected Object lock;
		protected volatile int position;
		protected volatile int skipped;

		public Handle(int pos, Object lock) {
			if (pos<0) throw new IllegalArgumentException("position must not be negative");
			
			this.lock = lock;
			this.position = pos;
			this.skipped = 0;
		}

		public abstract void onAvailable();
		public abstract void onSkip();
	}
	
	protected class BufferHandle extends Handle<E> {
		protected ViewMonitor<E> monitor; 
		protected String name = null; //for debugging and testing

		public BufferHandle(int pos, Object lock) {
			super(pos, lock);
			
			if (pos>=entries.length) throw new IllegalArgumentException("position out of range: "+pos+" >= "+entries.length);
			
			entries[pos].append(this);
		}
		
		public void setName(String name) {
			this.name = name;
		}
		
		public boolean isClosed() {
			return position < 0;
		}
		
		protected void checkClosed() {
			if (isClosed()) throw new IllegalStateException("view closed");
		}

		public E element() throws NoSuchElementException {
			checkClosed();
				
			E e = peek();
			if (e==null) throw new NoSuchElementException();
			return e;
		}

		public E remove() throws NoSuchElementException {
			E e = poll();
			if (e==null) throw new NoSuchElementException();
			return e;
		}

		public int getSkippedEntryCount() {
			return skipped;
		}

		public int resetSkippedEntryCount() {
			synchronized (lock) {
				int sk = skipped;
				skipped = 0;
				return sk;
			}
		}

		public boolean isEmpty() {
			return size() == 0;
		}

		public int size() {
			synchronized (lock) {
				checkClosed();
				
				if (position <= top) return top - position;
				else return entries.length - position + top;
			}
		}

		public E peek() {
			synchronized (lock) {
				if (isEmpty()) return null;
				else {
					if (entries[position].value==null) throw new RuntimeException("inconsistent inner state: no value in supposedly used entry"); //TODO: assert! 
					return entries[position].value;
				}
			}				
		}

		public E poll() { 
			E v;
			boolean empty = false;
			
			synchronized (lock) {
				v = peek();
				
				if (v!=null) {				
					int newPos = (position +1) % entries.length;
					
					entries[newPos].append(this);
					
					position = newPos;
					empty = isEmpty();
				}
			}
			
			if (v!=null && monitor!=null) {
				monitor.removedElement(this);
				if (empty) monitor.becameEmpty(this);
			}
			
			return v;
		}

		public void setMonitor(ViewMonitor<E> monitor) {
			this.monitor = monitor;
		}

		public void close() {
			if (position<0) return; //already closed
			
			synchronized (lock) {
				unlink();
				position = -1;
			}
			
			if (monitor!=null) monitor.becameClosed(this);
		}

		@Override
		public void onAvailable() {
			if (monitor!=null) monitor.becameAvailable(this);
		}

		@Override
		public void onSkip() {
			if (monitor!=null) monitor.skippedElement(this);
		}

		public String toString() {
			if (name!=null) return name;
			else return super.toString();
		}
	}
	
	protected static class Entry<E> extends ChainLink implements Iterable<Handle<E>> {
		protected Object lock;
		protected E value = null;
		protected ChainLink iterPos = null;
		
		public Entry(Object lock) {
			this.lock = lock;
		}
		
		public void append(Handle<E> handle) {
			handle.insertAfter(getPreviousLink());
		}

		public void transferHandlesTo(Entry<E> other) {
			if (other==this) return;
			
			synchronized (lock) {
				ChainLink n = getNextLink();
				if (n==this) return; //nothing to transfer 
				
				unlink();
				
				other.getPreviousLink().insertChainAfter(n);
				
				other.checkChain(Entry.class);
				this.checkChain(Entry.class);
			}
		}

		public boolean isUnused() {
			synchronized (lock) {
				return getNextLink() == this;
			}
		}

		protected Iterator<Handle<E>> iter = new Iterator<Handle<E>>() {
			public boolean hasNext() {
				return iterPos.getNextLink() != Entry.this;
			}

			public Handle<E> next() {
				iterPos = iterPos.getNextLink();
				return (Handle<E>)iterPos;
			}

			public void remove() {
				iterPos.unlink();
			}
		};
		
		/**
		 * Always returns the same iterator instance! Not thread safe, not re-entrant!
		 * Iterator must be used in a synchronized block!
		 */
		public Iterator<Handle<E>> iterator() {  
			checkChain(Entry.class);
			iterPos = this;
			return iter;
		}
		
	}
	
	protected boolean allowSkip = false; 
	protected int capacity;
	protected Entry<E>[] entries;
	
	protected int top;
	protected int base;
	
	protected Object lock;
	
	public MultiStreamBuffer(int capacity) {
		this.capacity = capacity;
		
		this.lock = this;
		this.entries = new Entry[capacity +1];
		
		for (int i=0; i<entries.length; i++) {
			entries[i] = new Entry<E>(lock);
		}
		
		this.top = 0;
		this.base = 0;
	}
	
	public boolean isAllowSkip() {
		return allowSkip;
	}

	public void setAllowSkip(boolean allowSkip) {
		this.allowSkip = allowSkip;
	}

	public int getCapacity() {
		return capacity;
	}

	public boolean isFull() {
		synchronized (lock) {
			purge();
			return top == (base-1) || (base == 0 && top == (entries.length-1));
		}
	}
	
	public boolean isEmpty() {
		return size() == 0;
	}

	public int size() {
		synchronized (lock) {
			purge();
			
			if (base <= top) return top - base;
			else return entries.length - base + top;
		}
	}
	
	public View<E> view(boolean lookBack) {
		synchronized (lock) {
			return new BufferHandle(lookBack ? base : top, lock);
		}
	}
	
	public boolean add(E element) {
		if (!offer(element)) throw new RuntimeException("queue overflow");
		return true;
	}
	
	protected Handle<E>[] handleBuffer = (Handle<E>[])new Handle[10]; //NOTE: not reentrant (obviously), use in sync only!

	protected void growHandleBuffer() {
		Handle<E>[] buff = (Handle<E>[])new Handle[handleBuffer.length + handleBuffer.length/2];
		System.arraycopy(handleBuffer, 0, buff, 0, handleBuffer.length);
		handleBuffer = buff;
	}
	
	public boolean offer (E element) {
		int handleCount = 0;
		synchronized (lock) {
			if (element==null) throw new IllegalArgumentException("element must not be null");
			
			if (isFull()) { //NOTE: implies purge!
				if (!allowSkip) return false;
				else shove(); //NOTE: causes monitor notification inside sync section! 
				              //XXX: inline shove here and defer notifications? would need two bufers...
			}
			
			Entry<E> e = entries[top];
			
			if (e.value!=null) throw new RuntimeException("inconsistent inner state: found a value in supposedly empty entry"); //TODO: assert! 
			e.value = element;
			top = (top + 1) % entries.length;
			
			//XXX: crass hack: keep monitor notification out of sync block, avoid creating a new buffer every time. 
			for (Handle<E> h: e) {
				if (handleCount > handleBuffer.length) growHandleBuffer();
				
				handleBuffer[handleCount] = h;
				handleCount++;
			}
		}
		
		for (int i = 0; i<handleCount; i++) {
			handleBuffer[i].onAvailable();
			handleBuffer[i] = null; //kill reference, help GC
		}
		
		return true;
	}
	
	protected void shove() {
		int handleCount = 0;
		synchronized (lock) {
			if (base == top) return;
			
			Entry<E> e = entries[base];
			if (e.value==null) throw new RuntimeException("inconsistent inner state: no value in supposedly used entry"); //TODO: assert! 
	
			int next = (base + 1) % entries.length;
			
			for (Handle<E> h : e) {
				if (handleCount > handleBuffer.length) growHandleBuffer();
				
				h.position = next;
				h.skipped ++;
				handleBuffer[handleCount] = h;
				handleCount++;
			}

			e.transferHandlesTo(entries[next]);

			e.value = null;
			base = next;
		}
		
		for (int i = 0; i<handleCount; i++) {
			handleBuffer[i].onSkip();
			handleBuffer[i] = null; //kill reference, help GC
		}
	}
	
	public int purge() {
		synchronized (lock) {
			int c = 0;
			while (base != top) {
				if (!entries[base].isUnused()) break;
				
				entries[base].value = null;
				base = (base + 1) % entries.length;
			}
			
			return c;
		}
	}
	
	public void dumpInternalState(PrintStream out) {
		synchronized (lock) {
			int i = 0;
			for (Entry e: entries) {
				out.printf("[%d] = %s ", i, e.value);
				if (i == top) out.print(" <-- TOP");
				if (i == base) out.print(" <-- BASE");
				out.println();
				i++;
			}
		}
	}
	
	protected static <E> void printViewStep(String name, View<E> v) {
		if (v.getSkippedEntryCount()>0) {
			//System.out.println(name+": SKIPPED "+v.getSkippedEntryCount());
			v.resetSkippedEntryCount();
		}
		
		//if (v.isEmpty()) System.out.println(name+": EMPTY ");
		//else System.out.println(name+": VALUE " + v.remove());
	}
	
	public static void main(String[] args) {
		MultiStreamBuffer<String> buffer = new MultiStreamBuffer<String>(3);
		buffer.setAllowSkip(true);
		
		ViewMonitor<String> monitor = new ViewMonitor<String>() {

			public void becameAvailable(View<String> view) {
				System.out.println("* "+view+" becameAvailable");
			}

			public void becameEmpty(View<String> view) {
				System.out.println("* "+view+" becameEmpty");
			}

			public void removedElement(View<String> view) {
				System.out.println("* "+view+" removedElement");
			}

			public void skippedElement(View<String> view) {
				System.out.println("* "+view+" skippedElement");
			}

			public void becameClosed(View<String> view) {
				System.out.println("* "+view+" becameClosed");
			}
			
		};
		
		System.out.println("-- start");
		MultiStreamBuffer.View<String> v1 = buffer.view(false);
		((MultiStreamBuffer.BufferHandle)v1).setName("V1");
		v1.setMonitor(monitor);
		
		System.out.println("--> one, two");
		buffer.add("one");
		buffer.add("two");
		buffer.dumpInternalState(System.out);
		
		printViewStep("V1", v1);
		
		MultiStreamBuffer.View<String> v2 = buffer.view(false);
		((MultiStreamBuffer.BufferHandle)v2).setName("V2");
		v2.setMonitor(monitor);

		printViewStep("V2", v2);
		printViewStep("V2", v2);

		System.out.println("--> three");
		buffer.add("three");
		buffer.dumpInternalState(System.out);
		
		System.out.println("--> four");
		buffer.add("four");
		buffer.dumpInternalState(System.out);

		printViewStep("V1", v1);

		printViewStep("V2", v2);
		printViewStep("V2", v2);
		
		System.out.println("--> five, six");
		buffer.add("five");
		buffer.add("six");
		buffer.dumpInternalState(System.out);

		printViewStep("V1", v1);

		System.out.println("--> seven");
		buffer.add("seven");
		buffer.dumpInternalState(System.out);

		printViewStep("V2", v2);		
		printViewStep("V2", v2);		
		printViewStep("V2", v2);		
		printViewStep("V2", v2);		
	
		MultiStreamBuffer.View<String> v3 = buffer.view(false);
		((MultiStreamBuffer.BufferHandle)v3).setName("V3");
		v3.setMonitor(monitor);

		MultiStreamBuffer.View<String> v4 = buffer.view(true);
		((MultiStreamBuffer.BufferHandle)v4).setName("V4");
		v4.setMonitor(monitor);

		printViewStep("V3", v3);
		printViewStep("V4", v4);

		System.out.println("--> eight, nine, then");
		buffer.add("eight");
		buffer.add("nine");
		buffer.add("ten");
		buffer.dumpInternalState(System.out);


		System.out.println("-- no more");
		printViewStep("V1", v1);
		printViewStep("V2", v2);
		printViewStep("V3", v3);
		printViewStep("V4", v4);
		
		System.out.println("-- end");
	}
}
