package de.brightbyte.io;

import java.text.ParseException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class MimeType implements Comparable<MimeType> {
	private String major;
	private String minor;
	private Map<String, String> params;
	
	private String external;
	private int hash;
	
	protected static final Pattern mimePattern = Pattern.compile("^([-\\w\\d]+)/([-\\w\\d]+)(;.*)?$");
	protected static final Pattern delimPattern = Pattern.compile("\\s*;\\s*");
	protected static final Pattern paramPattern = Pattern.compile("^\\s*([-\\w\\d]+)\\s*=\\s*([^;]*)\\s*$");
	
	public static MimeType parse(String mime) throws ParseException {
		Matcher m = mimePattern.matcher(mime);
		if (!m.matches()) throw new ParseException(mime, 0);
		
		String major = m.group(1);
		String minor = m.group(2);
		String p = m.groupCount()>2 ? m.group(3) : null;
		Map<String, String> params = null;
		
		if (p!=null) {
			String[] pp= delimPattern.split(p);
			if (pp.length>0) {
				params = new HashMap<String, String>(pp.length);
				for (int i = 0; i < pp.length; i++) {
					Matcher n = paramPattern.matcher(pp[i]);
					if (!n.matches()) throw new ParseException(mime, 0);
					
					String k = n.group(1);
					String v = n.group(2);
					
					params.put(k, v);
				}
			}
		}
		
		return new MimeType(major, minor, params);
	}
	
	public MimeType(String major, String minor, Map<String, String> params) {
		if (major == null || minor == null) throw new NullPointerException();
		this.major = major.toLowerCase().trim();
		this.minor = minor.toLowerCase().trim();

		if (params!=null) this.params = Collections.unmodifiableMap(params);
		else this.params = Collections.emptyMap(); 
		
		external = major + "/" + minor;
		if (params!=null && params.size()>0) {
			StringBuffer s = new StringBuffer();
			for (Map.Entry<String, String> e: params.entrySet()) {
				s.append(';');
				s.append(e.getKey());
				s.append('=');
				s.append(e.getValue());
			}
			external += s;
		}
		
		final int PRIME = 31;
		int hash = 1;
		hash = PRIME * hash + major.hashCode();
		hash = PRIME * hash + minor.hashCode();
		hash = PRIME * hash + ((params == null) ? 0 : params.hashCode());
	}
	
	public String toString() {
		return toExternalForm();
	}

	public String toExternalForm() {
		return external;
	}

	public String getMajorType() {
		return major;
	}

	public String getMinorType() {
		return minor;
	}

	public Map<String, String> getParameterMap() {
		return params;
	}

	public String getParameter(String name) {
		return params.get(name);
	}
	
	public boolean matches(String major, String minor) {
		return this.major.equals(major) && this.minor.equals(minor);
	}

	public boolean matches(MimeType m) {
		return matches(m.getMajorType(), m.getMinorType());
	}

	@Override
	public int hashCode() {
		return hash;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		
		final MimeType other = (MimeType) obj;
		if (hash!=other.hashCode())
			return false;
		if (!major.equals(other.getMajorType()))
			return false;
		if (!minor.equals(other.getMinorType()))
			return false;
		if (!params.equals(other.getParameterMap()))
			return false;
		
		return true;
	}

	public int compareTo(MimeType o) {
		return external.compareTo(o.toExternalForm());
	}
	
}
