package de.brightbyte.data.graph;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import de.brightbyte.data.Pair;
import de.brightbyte.data.UnmodifiableException;

public class MapGraph<V, E> implements Graph<V, E> {
	
	protected static class CombinedEdges<V, E> implements Set<EdgeEntry<V, E>> {
		protected Set<EdgeEntry<V, E>> edges = new HashSet<EdgeEntry<V, E>>();
		
		public CombinedEdges(Collection<EdgeEntry<V, E>> outgoingEdges, Collection<EdgeEntry<V, E>> incommingEdges, Relation rel) {

			if (rel != Relation.IN) edges.addAll(outgoingEdges);
			
			if (rel == Relation.STRONG) {
				Iterator<EdgeEntry<V, E>>it = edges.iterator();
				while (it.hasNext()) {
					EdgeEntry<V, E> ee= it.next();
					if (!incommingEdges.contains( new MapEdgeEntry<V, E>(ee.getDestination(), ee.getOrigin(), null))) {
						it.remove();
					}
				}
			}
			else if (rel != Relation.OUT && rel != Relation.WEAK) { 
				for(EdgeEntry<V, E> ee: incommingEdges) {
					edges.add(new MapEdgeEntry<V, E>(ee.getDestination(), ee.getOrigin(), ee.getValue()));
				}
			}
		}

		public boolean add(EdgeEntry<V, E> e) {
			throw new UnmodifiableException();
		}

		public boolean addAll(Collection<? extends EdgeEntry<V, E>> c) {
			throw new UnmodifiableException();
		}

		public void clear() {
			throw new UnmodifiableException();
		}

		public boolean contains(Object o) {
			return edges.contains(o);
		}

		public boolean containsAll(Collection<?> c) {
			return edges.containsAll(c);
		}

		public boolean equals(Object o) {
			return edges.equals(o);
		}

		public int hashCode() {
			return edges.hashCode();
		}

		public boolean isEmpty() {
			return edges.isEmpty();
		}

		public Iterator<EdgeEntry<V, E>> iterator() {
			final Iterator<EdgeEntry<V, E>> it = edges.iterator();
			
			return new Iterator<EdgeEntry<V, E>>() {
				public boolean hasNext() {
					return it.hasNext();
				}

				public EdgeEntry<V, E> next() {
					return it.next();
				}

				public void remove() {
					throw new UnmodifiableException();
				}
			};
		}

		public boolean remove(Object o) {
			throw new UnmodifiableException();
		}

		public boolean removeAll(Collection<?> c) {
			throw new UnmodifiableException();
		}

		public boolean retainAll(Collection<?> c) {
			throw new UnmodifiableException();
		}

		public int size() {
			return edges.size();
		}

		public Object[] toArray() {
			return edges.toArray();
		}

		public <T> T[] toArray(T[] a) {
			return edges.toArray(a);
		}
		
		
	}
	
	protected static class MapVertexEntry<V, E> implements VertexEntry<V, E> {

		protected HashMap<V, EdgeEntry<V, E>> outgoing;
		protected HashMap<V, EdgeEntry<V, E>> incomming;
		protected V value;
		
		public MapVertexEntry(V v) {
			this.value = v;
		}

		public Collection<EdgeEntry<V, E>> getEdges(Relation rel) {
			switch (rel) {
			case IN: 
				return incomming == null ? Collections.<EdgeEntry<V, E>>emptySet() : incomming.values();
			case OUT: 
				return outgoing == null ? Collections.<EdgeEntry<V, E>>emptySet() : outgoing.values();
			default: 
				return new CombinedEdges<V, E>(getEdges(Relation.OUT), getEdges(Relation.IN), rel);
			}
		}
		
		public E getEdgeValue(V vertex, Relation rel) {
			EdgeEntry<V, E> e;
			E v;
			
			switch (rel) {
			case IN: 
				if (incomming==null) return null;
				e = incomming.get(vertex);
				return e == null ? null : e.getValue();
			case OUT: 
				if (outgoing==null) return null;
				e = outgoing.get(vertex);
				return e == null ? null : e.getValue();
			case WEAK: 
				v = getEdgeValue(vertex, Relation.IN);
				if (v!=null) return v; 
				v = getEdgeValue(vertex, Relation.OUT);
				return v;
			case STRONG: 
				v = getEdgeValue(vertex, Relation.IN);
				if (v==null) return null; 
				v = getEdgeValue(vertex, Relation.OUT);
				return v;
			default:
				throw new IllegalArgumentException("unknown relation "+rel);
			}
		}
		
		protected void addIncommingEdge(V origin, E value) {
			if (incomming==null) incomming = new HashMap<V, EdgeEntry<V, E>>();
			incomming.put(origin, new MapEdgeEntry<V, E>(origin, this.value, value));
		}

		protected void addOutgoingEdge(V destination, E value) {
			if (outgoing==null) outgoing = new HashMap<V, EdgeEntry<V, E>>();
			outgoing.put(destination, new MapEdgeEntry<V, E>(this.value, destination, value));
		}

		protected void removeIncommingEdge(V origin) {
			if (incomming==null) return;
			incomming.remove(origin);
		}

		protected void removeOutgoingEdge(V destination) {
			if (outgoing==null) return;
			outgoing.remove(destination);
		}

		public V getValue() {
			return value;
		}
		
		protected V setValue(V v) {
			V old = value;
			value = v;
			return old;
		}
		
		public String toString() {
			return value.toString();
		}

		public int getDegree(de.brightbyte.data.graph.Graph.Relation rel) {
			HashSet<V> n;
			
			switch (rel) {
			case IN: 
				return incomming == null ? 0 : incomming.size();
			case OUT: 
				return outgoing == null ? 0 : outgoing.size();
			case WEAK: 
				n = new HashSet<V>( (incomming == null ? 0 : incomming.size()) + (outgoing == null ? 0 : outgoing.size()));
				if (incomming!=null) n.addAll(incomming.keySet());
				if (outgoing!=null) n.addAll(outgoing.keySet());
				return n.size();
			case STRONG: 
				n = new HashSet<V>( (incomming == null ? 0 : incomming.size()) + (outgoing == null ? 0 : outgoing.size()));
				if (incomming!=null) n.addAll(incomming.keySet());
				if (outgoing!=null) n.retainAll(outgoing.keySet());
				return n.size();
			default:
				throw new IllegalArgumentException("unknown relation "+rel);
			}
		}
	}
	
	protected static class MapEdgeEntry<V, E> implements EdgeEntry<V, E> {

		protected V destination;
		protected V origin;
		protected E value;
		
		public MapEdgeEntry(V origin, V destenation, E value) {
			this.origin = origin;
			this.destination = destenation;
			this.value = value;
		}

		public V getDestination() {
			return destination;
		}

		public V getOrigin() {
			return origin;
		}

		public E getValue() {
			return value;
		}
		
		public String toString() {
			return origin + " ->(" + value + ")-> " + destination;
		}

		@Override
		public int hashCode() {
			final int PRIME = 31;
			int result = 1;
			result = PRIME * result + ((destination == null) ? 0 : destination.hashCode());
			result = PRIME * result + ((origin == null) ? 0 : origin.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final MapEdgeEntry other = (MapEdgeEntry) obj;
			if (destination == null) {
				if (other.destination != null)
					return false;
			} else if (!destination.equals(other.destination))
				return false;
			if (origin == null) {
				if (other.origin != null)
					return false;
			} else if (!origin.equals(other.origin))
				return false;
			return true;
		}
		
		
	}
	
	protected Map<V, VertexEntry<V, E>> vertecies = new HashMap<V, VertexEntry<V, E>>();
	protected Map<Pair<V, V>, EdgeEntry<V, E>> edges = new HashMap<Pair<V, V>, EdgeEntry<V, E>>();
	
	protected boolean ignoreLoops;

	public Collection<EdgeEntry<V, E>> edges() {
		return Collections.unmodifiableCollection(edges.values());
	}

	public int getEdgeCount() {
		return edges.size();
	}

	public Collection<EdgeEntry<V, E>> getEdges(V v, Relation rel) {
		VertexEntry<V, E> ve = vertecies.get(v);
		if (ve==null) throw new NoSuchElementException("vertex "+v);
		return ve.getEdges(rel);
	}

	public E getEdgeValue(V v, V w, Relation rel) {
		VertexEntry<V, E> ve = vertecies.get(v);
		if (ve==null) throw new NoSuchElementException("vertex "+v);
		return ((MapVertexEntry<V, E>)ve).getEdgeValue(w, rel);
	}

	public int getVertexCount() {
		return vertecies.size();
	}

	public Graph<V, E> neighbourhood(V vertex, Relation rel) {
		VertexEntry<V, E> ve = vertecies.get(vertex);
		if (ve==null) throw new NoSuchElementException("vertex "+vertex);

		MapGraph<V, E> g = new MapGraph<V, E>();
		g.putVertex(vertex);
		
		for (EdgeEntry<V, E> ee: ve.getEdges(rel)) {
			g.putVertex(ee.getDestination());
		}
		
		Relation nrel = rel == Relation.STRONG ? Relation.STRONG : Relation.OUT; 
		g.copyEdges(this, nrel);
		
		return g;
	}
	
	public int copyEdges(Graph<V, E> g, Relation rel) {
		int c = 0;
		
		for (VertexEntry<V, E> we: vertices()) {
			VertexEntry<V, E> wwe = g.getVertex(we.getValue());
			
			for (EdgeEntry<V, E> ee: wwe.getEdges(rel)) {
				if ( getVertex(ee.getDestination()) != null ) {
					putEdge(ee.getOrigin(), ee.getDestination(), ee.getValue());
					c++;
				}
			}
		}
		
		return c;
	}

	public E putEdge(V v, V w, E e) {
		if (ignoreLoops) {
			if (v==w || v.equals(w)) return null;
		}
		
		VertexEntry<V, E> ve = vertecies.get(v);
		if (ve==null) throw new NoSuchElementException("vertex "+v);

		VertexEntry<V, E> we = vertecies.get(w);
		if (we==null) throw new NoSuchElementException("vertex "+w);

		((MapVertexEntry<V, E>)ve).addOutgoingEdge(w, e);
		((MapVertexEntry<V, E>)we).addIncommingEdge(v, e);
		
		EdgeEntry<V, E> old = edges.put(new Pair<V, V>(v, w), new MapEdgeEntry<V, E>(v, w, e));
		return old == null ? null : old.getValue();
	}

	public void putGraph(Graph<V, E> g) {
		for (VertexEntry<V, E> ve: g.vertices()) {
			putVertex(ve.getValue());
		}

		for (EdgeEntry<V, E> ee: g.edges()) {
			putEdge(ee.getOrigin(), ee.getDestination(), ee.getValue());
		}
	}

	public V putVertex(V v) {
		VertexEntry<V, E> ve = vertecies.get(v);
		if (ve==null) {
			vertecies.put(v, new MapVertexEntry<V, E>(v));
			return null;
		}
		else {
			return ((MapVertexEntry<V, E>)ve).setValue(v);
		}
	}

	public E removeEdge(V v, V w) {
		VertexEntry<V, E> ve = vertecies.get(v);
		if (ve==null) throw new NoSuchElementException("vertex "+v);

		VertexEntry<V, E> we = vertecies.get(w);
		if (we==null) throw new NoSuchElementException("vertex "+w);

		EdgeEntry<V, E> ee = edges.remove(new Pair<V, V>(v, w));
		if (ee==null) return null;

		((MapVertexEntry<V, E>)ve).removeOutgoingEdge(w);
		((MapVertexEntry<V, E>)we).removeIncommingEdge(v);
		
		return ee.getValue();
	}

	public int removeVertecies(Collection<V> remove) {
		int c = 0;
		for (V v: remove) {
			if (removeVertex(v)) c++;
		}
		
		return c;
	}

	public int retainVertecies(Collection<V> retain) {
		int c = 0;
		Set<V> remove = new HashSet<V>(getVertexCount()); //NOTE: avoid concurrent modification error
		
		for (V v: this.vertecies.keySet()) {
			if (!retain.contains(v)) {
				remove.add(v);
			}
		}
		
		for (V v: remove) {
			if (removeVertex(v)) c++;
		}
		
		return c;
	}

	public boolean removeVertex(V v) {
		VertexEntry<V, E> ve = vertecies.remove(v);
		if (ve==null) return false;
		
		//NOTE: avoid concurrent mod!
		Collection outc = ve.getEdges(Relation.OUT);
		Collection inc = ve.getEdges(Relation.IN);
		EdgeEntry<V, E>[] out = (EdgeEntry<V, E>[])outc.toArray(new EdgeEntry[outc.size()]);
		EdgeEntry<V, E>[] in = (EdgeEntry<V, E>[])inc.toArray(new EdgeEntry[inc.size()]);
		
		for(EdgeEntry<V, E> e: out) {
			removeEdge(v, e.getDestination());
		}
		
		for(EdgeEntry<V, E> e: in) {
			removeEdge(e.getOrigin(), v);
		}
		
		return false;
	}

	public Collection<VertexEntry<V, E>> vertices() {
		return Collections.unmodifiableCollection(vertecies.values());
	}

	public Set<V> vertexSet() {
		return Collections.unmodifiableSet(vertecies.keySet());
	}

	public VertexEntry<V, E> getVertex(V v) {
		return vertecies.get(v);
	}

	public void dump(PrintWriter output) {
		for (EdgeEntry<V, E> e: edges.values()) {
			output.println(e);
		}
	}

	public void dump(PrintStream output) {
		for (EdgeEntry<V, E> e: edges.values()) {
			output.println(e);
		}
	}

	public boolean getIgnoreLoops() {
		return ignoreLoops;
	}

	public void setIgnoreLoops(boolean ignoreLoops) {
		this.ignoreLoops = ignoreLoops;
	}

}
