package de.brightbyte.data.graph;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import de.brightbyte.data.graph.Graph.Relation;
import de.brightbyte.data.graph.Graph.VertexEntry;
import de.brightbyte.data.measure.Measure;

public class GraphGenerator<V, E> {
	
	public static interface GraphFactory<V, E> {
		public Graph<V, E> newGraph();
	}

	public static interface VertexFactory<V> {
		public V newVertex(int num);
	}

	public static class VertexGeneratingIteratore<V> implements Iterator<V> {
		protected int count = 0;
		protected int max;
		protected VertexFactory<V> factory;
		
		public VertexGeneratingIteratore(VertexFactory<V> factory, int c) {
			this.count = 0;
			this.max = c;
			this.factory = factory;
		}

		public boolean hasNext() {
			return count < max;
		}

		public V next() {
			count++;
			return factory.newVertex(count);
		}

		public void remove() {
			throw new UnsupportedOperationException();
		}
	}
	
	public static final GraphFactory<Integer, Boolean> dummyGraphFactory = new GraphFactory<Integer, Boolean>() {
		public Graph<Integer, Boolean> newGraph() {
			return new MapGraph<Integer, Boolean>();
		}
	};
	
	public static final VertexFactory<Integer> dummyVertexFactory = new VertexFactory<Integer>() {
		public Integer newVertex(int num) {
			return num;
		}
	};
	
	public static GraphGenerator<Integer, Boolean> dummyInstance() {
		return new GraphGenerator<Integer, Boolean>(dummyGraphFactory, dummyVertexFactory, true); 
	}
	
	protected GraphFactory<V, E> graphFactory;
	protected E defaultEdgeValue;
	protected VertexFactory<V> vertexFactory;
	
	public GraphGenerator(GraphFactory<V, E> graphFactory, VertexFactory<V> vertexFactory, E defaultEdgeValue) {
		this.graphFactory = graphFactory;
		this.defaultEdgeValue = defaultEdgeValue;
		this.vertexFactory = vertexFactory;
	}

	protected Graph<V, E> newGraph() {
		return graphFactory.newGraph();
	}
	
	protected E getDefaultEdgeValue() {
		return defaultEdgeValue;
	}
	
	protected Iterable<V> getVertexGenerator(final int size) {
		return new Iterable<V>() {
			public Iterator<V> iterator() {
				return new VertexGeneratingIteratore<V>(vertexFactory, size);
			}
		};
	}
	
	public Graph<V, E> randomGraph(int size, int degree) {
		Graph<V, E> g = newGraph();
		
		for (V v: getVertexGenerator(size)) {
			g.putVertex(v);
		}
		
		addRandomLinks(g, size*degree, getDefaultEdgeValue());
		return g;
	}

	public void addRandomLinks(Graph<V, E> g, int n, E edgeValue) {
		for (int i = 0; i<n; i++) {
			//XXX: dog slow...
			V a = GraphRandom.randomVertex(g).getValue();
			V b = GraphRandom.randomVertex(g).getValue();
			
			g.putEdge(a, b, edgeValue);
		}
	}

	public Graph<V, E> completeGraph(int size, E edgeValue) {
		Graph<V, E> g = unconnectedGraph(size);
		
		for (V i: g.vertexSet()) {
			for (V j: g.vertexSet()) {
				g.putEdge(i, j, edgeValue); //TODO: loops, or no loops? strong or weak?
			}
		}
		
		return g;
	}

	public Graph<V, E> unconnectedGraph(int size) {
		Graph<V, E> g = newGraph();
		
		for (V i: getVertexGenerator(size)) {
			g.putVertex(i);
		}
		
		return g;
	}

	public Graph<V, E> regularCircleGraph(int size, int neighbourhood) {
		Graph<V, E> g = newGraph();
		E edgeValue = getDefaultEdgeValue();
		
		List<V> vertecies = new ArrayList<V>(size); 
		
		for (V v: getVertexGenerator(size)) {
			vertecies.add(v);
			g.putVertex(v);
		}
		
		for (int i = 0; i < size; i++) {
			V iv = vertecies.get(i);

			for (int j = 0; j < neighbourhood; j++) {
				int a = (i - j) % size;
				int b = (i + j) % size;
				
				if (a<0) a = size +a;
				
				V av = vertecies.get(a);
				V bv = vertecies.get(b);

				g.putEdge(iv, av, edgeValue);
				g.putEdge(iv, bv, edgeValue); //TODO: optional!
			}
		}
		
		return g;
	}
	
	public Graph<V, E> regularGridGraph(int width, int height) {
		Graph<V, E> g = newGraph();
		E edgeValue = getDefaultEdgeValue();
		
		int size = width * height;
		List<V> vertecies = new ArrayList<V>(size); 
		
		Iterator<V> gen = getVertexGenerator(size).iterator();
		
		for (int y = 0; y<height; y++) {
			for (int x = 0; x<width; x++) {
				V v = gen.next(); 
				vertecies.add(v);
				g.putVertex(v);
			}
		}
		
		for (int y = 0; y<height; y++) {
			for (int x = 0; x<width; x++) {
				V v = vertecies.get(y * width + x);

				int xa = (x - 1) % width;
				int xb = (x + 1) % width;
				if (xa<0) xa = width -xa;
				
				int ya = (y - 1) % height;
				int yb = (y + 1) % height;
				if (ya<0) ya = height -ya;
				
				//V v1 = vertecies.get(y * width + xa);
				V v2 = vertecies.get(y * width + xb);
				//V v3 = vertecies.get(ya * width + x);
				V v4 = vertecies.get(yb * width + x);
				
				//g.putEdge(v, v1, true); //TODO: optional!
				g.putEdge(v, v2, edgeValue); 
				//g.putEdge(v, v3, true); //TODO: optional! 
				g.putEdge(v, v4, edgeValue); 
			}
		}
		
		return g;
	}
	
	public void randomize(Graph<V, E> g, double p) {
		Map<Graph.EdgeEntry<V, E>, V> todo = new HashMap<Graph.EdgeEntry<V, E>, V>(g.getEdgeCount());
		
		for (Graph.EdgeEntry<V, E> ee: g.edges()) {
			if (Math.random() > p) continue;
			V v = GraphRandom.randomVertex(g).getValue();
			todo.put(ee, v);
		}
		
		for (Map.Entry<Graph.EdgeEntry<V, E>, V> e: todo.entrySet()) {
			Graph.EdgeEntry<V, E> ee = e.getKey();
			g.removeEdge(ee.getOrigin(), ee.getDestination());
			g.putEdge(ee.getOrigin(), e.getValue(), ee.getValue());
		}
	}
	
	public Graph<V, E> kumarGraph(int size, int degree, double copyFactor) {
		Graph<V, E> g = completeGraph(degree, getDefaultEdgeValue());
		E edgeValue = getDefaultEdgeValue();
		
		Iterable<V> generator = getVertexGenerator(size);

		boolean first = true;
		for (V v: generator) {
			g.putVertex(v);
			
			if (first) {
				first = false;
				break;
			}
			
			Graph.VertexEntry<V, E> prototype = GraphRandom.randomVertex(g);
			
			for (int k=0; k<degree; k++) {
				V dest = null;

				if (Math.random()<=copyFactor) {
					Graph.EdgeEntry<V, E> e = GraphRandom.randomEdge(prototype, Graph.Relation.OUT);
					dest = e==null ? null : e.getDestination();
					//if (dest!=null) System.out.println("Kumar: copy: "+v+" -> "+dest);
				}
				
				if (dest==null)  {
					Graph.VertexEntry<V, E> ve = GraphRandom.randomVertex(g); 
					dest = ve.getValue();
					//System.out.println("Kumar: random: "+v+" -> "+dest);
				}
				
				g.putEdge(v, dest, edgeValue);
			}
		}
		
		return g;
	}
	
	public Graph<V, E> barabasiGraph(int size, int degree) {
		E edgeValue = getDefaultEdgeValue();
		final Graph<V, E> g = completeGraph(degree, edgeValue);
		
		Measure<Graph.VertexEntry<V, E>> prob = new Measure<Graph.VertexEntry<V, E>>() {
			public double measure(VertexEntry<V, E> ve) {
				//NOTE: add-one-smoothed relative in-degree
				double d = ve.getDegree(Relation.IN);
				d = (d+1)/(double)(g.getEdgeCount()+g.getVertexCount());
				return d;
			}
		};

		for (V v: getVertexGenerator(size - degree)) {
			Graph.VertexEntry<V, E>[] others = GraphRandom.randomVertecies(g, degree, prob);

			g.putVertex(v);
			for (Graph.VertexEntry<V, E> e: others) {
				//System.out.println("Barabasi: "+v+" -> "+others[k].getValue());
				g.putEdge(v, e.getValue(), edgeValue);
			}
		}
		
		return g;
	}
}
