package de.brightbyte.data.graph;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

import de.brightbyte.data.LabeledMatrix;
import de.brightbyte.data.MapLabeledMatrix;
import de.brightbyte.data.graph.Graph.Relation;
import de.brightbyte.util.PowerMath;

public class GraphConnectivity {
	
	public static <V, E> double density(Graph<V, E> g, Graph.Relation rel) {
		//FIXME: if rel==STRONG, use only symmetric edges!!!
		//       ---> note: number of possible edges is only half if rel==STRONG
		//       ---> how about loops, eh?
		int v = g.getVertexCount();
		if (v<2) return 1; //FIXME: is this The Right Thing?!
		
		long max = PowerMath.binomial(v, 2);
		if (rel!=Graph.Relation.STRONG) max = max * 2;
		
		return g.getEdgeCount() / (double)max;
	}
	
	public static <V, E> double clusteringCoefficient(V v, Graph<V, E> g, Graph.Relation rel) {
		Graph<V, E> n = g.neighbourhood(v, rel);
		return density(n, rel);
	}
	
	public static <V, E> double averageClusteringCoefficient(Graph<V, E> g, Graph.Relation rel) {
		double c = 0;
		for (Graph.VertexEntry<V, E> ve: g.vertices()) {
			c += clusteringCoefficient(ve.getValue(), g, rel);
		}
		return c / g.getVertexCount();
	}
	
	public static <V, E> long totalDegree(Graph<V, E> g, Graph.Relation rel) {
		long d = 0;
		for (Graph.VertexEntry<V, E> ve: g.vertices()) {
			d += ve.getDegree(rel);
		}
		
		return d;
	}
	
	public static <V, E> double averageDegree(Graph<V, E> g, Graph.Relation rel) {
		return totalDegree(g, rel) / (double)g.getVertexCount();
	}

	public static <V, E>LabeledMatrix<V, V>adjacency(Graph<V, E> g, Graph.Relation rel) {
		LabeledMatrix<V, V> m = new MapLabeledMatrix<V, V>();
		Set<V> keys = g.vertexSet();
		
		for (V i: keys) {
			for (V j: keys) {
				double w;
				if (i==j || i.equals(j)) w = 0;
				else {
					E v = g.getEdgeValue(i, j, rel);
					if (v==null) w = Double.POSITIVE_INFINITY;
					else if (v instanceof Number) w = ((Number)v).doubleValue();
					else w = 1;
				}
				
				m.set(i, j, w);
			}
		}

		return m;
	}
	
	public static <V, E>boolean isConnected(Graph<V, E> g, Graph.Relation rel) {
		LabeledMatrix<V, V> m = adjacency(g, rel);
		Set<V> keys = g.vertexSet();
		
		for (V i: keys) {
			for (V j: keys) {
				if (m.get(i, j) == Double.POSITIVE_INFINITY) return false;
			}
		}
		
		return true;
	}
	
	//lenient wrt unconnected nodes
	public static <V, E>double averagePathLength(Graph<V, E> g, Graph.Relation rel) {
		LabeledMatrix<V, V> m = shortestPathes(g, rel);
		Set<V> keys = g.vertexSet();
		
		double total = 0;
		int c = 0;
		
		for (V i: keys) {
			for (V j: keys) {
				if (i==j || i.equals(j)) continue;

				double w = m.get(i, j);
				if (w == Double.POSITIVE_INFINITY) continue;
				
				total += w;
				c++;
			}
		}
		
		if (c==0) return Double.NaN;
		return total/c;
	}
	
	public static <V, E>LabeledMatrix<V, V>shortestPathes(Graph<V, E> g, Graph.Relation rel) {
		//Floyd-Warshall
		LabeledMatrix<V, V> m = adjacency(g, rel);
		Set<V> keys = g.vertexSet();
		
		for (V i: keys) {
			for (V j: keys) {
				double w;
				if (i==j || i.equals(j)) w = 0;
				else {
					E v = g.getEdgeValue(i, j, rel);
					if (v==null) w = Double.POSITIVE_INFINITY;
					else if (v instanceof Number) w = ((Number)v).doubleValue();
					else w = 1;
				}
				
				m.set(i, j, w);
			}
		}
		
		for (V k: keys) {
			for (V i: keys) {
				for (V j: keys) {
					double wij = m.get(i, j);
					double wik = m.get(i, k);
					double wkj = m.get(k, j);
					
					double w = Math.min ( wij, wik+wkj );
					m.set(i, j, w); 
				}
			}
		}
		
		return m;
	}

	public static <V, E> List<Graph<V, E>> getConnectedParts(Graph<V, E> g, Graph.Relation rel) {
		List<Graph<V, E>> parts = new ArrayList<Graph<V, E>>();
		Set<V> pool = new HashSet<V>( g.vertexSet() );
		
		Relation prel = rel == Relation.STRONG ? Relation.STRONG : Relation.OUT; 

		while (!pool.isEmpty()) {
			Iterator<V> it = pool.iterator();
			V some = it.next();
			it.remove();
			
			MapGraph<V, E> part = new MapGraph<V, E>();
			
			Queue<V> todo = new LinkedList<V>( );
			todo.add(some);

			while (!todo.isEmpty()) {
				V v = todo.poll();
				part.putVertex(v);
				
				Graph.VertexEntry<V, E> ve = g.getVertex(v);
				
				for (Graph.EdgeEntry<V, E> ee: ve.getEdges(rel)) {
					V next = ee.getDestination();
					if (pool.contains(next)) {
						pool.remove(next);
						todo.add(next);
					}
				}
			}
			
			part.copyEdges(g, prel);
			
			if (part.getVertexCount()>0) parts.add(part);
		}
		
		Collections.sort(parts, Graph.byVertexCount);
		Collections.reverse(parts);
		
		return parts;
	}
	
	public static <V, E> Map<Integer, Integer> degreeRankDistribution(Graph<V, E> g, Relation rel) {
		Collection<Graph.VertexEntry<V, E>> vertices = g.vertices();
		List<Integer> ranking = new ArrayList<Integer>(vertices.size());
		
		for (Graph.VertexEntry<V, E> ve: vertices) {
			int dg = ve.getDegree(rel);
			ranking.add(dg);
		}
		
		Collections.sort(ranking);
		Collections.reverse(ranking);		
		
		int v = -1;
		int n = ranking.size();
		HashMap<Integer, Integer> distribution = new HashMap<Integer, Integer>(n);
		for (int r = 0; r<n; r++) {
			int w = ranking.get(r);
			if (v>0 && w!=v) {
				distribution.put(v, r);
			}
			
			v = w;
		}

		if (v>0) distribution.put(v, n-1);
		return distribution;
	}

	public static <V, E> double normalizedDegreeRankDeviation(Graph<V, E> g, Relation rel) {
		Map<Integer, Integer> dist = degreeRankDistribution(g, rel);
		int n = dist.size();
		int[] dr = new int[n];
		long total = 0;
		
		//grab array of rank * value 
		int i = 0;
		for (Map.Entry<Integer, Integer> e: dist.entrySet()) {
			int v = e.getKey();
			int r = e.getValue();
			int k = v * r;
			
			total += k;
			dr[i++] = k;
		}
		
		int avg = (int)(total/dr.length);

		//grab variance: (x1^2+...+xn^2)/n - avg^2
		long sq = 0;
		for (int x : dr) {
			sq += x*x;
		}
		
		double variance = sq/n - avg*avg; 
		double deviation = Math.sqrt(variance);
		double normalized = deviation / avg;
		
		return normalized;
	}
	
	public static <V, E> void dumpStats(Graph<V, E> g, PrintStream out) {
		int v = g.getVertexCount();
		int e = g.getEdgeCount();
		out.println(String.format("Nodes: %1$d, Edges: %2$d", v, e));

		double den = density(g, Graph.Relation.OUT);
		out.println(String.format("Density: %1$05.3f", den));
		
		double adg = averageDegree(g, Graph.Relation.OUT);
		out.println(String.format("Average Node Degree: %1$06.3f", adg));

		double apl = averagePathLength(g, Graph.Relation.OUT);
		out.println(String.format("Average Path Length: %1$06.3f", apl));

		double acc = averageClusteringCoefficient(g, Graph.Relation.OUT);
		out.println(String.format("Average Clustering Coefficient: %1$05.3f", acc));
		
		double drd = normalizedDegreeRankDeviation(g, Graph.Relation.OUT);
		out.println(String.format("Normalized Degree Rank Deviation: %1$05.3f", drd));
		
		List<Graph<V, E>> scc = getConnectedParts(g, Graph.Relation.OUT);
		List<Graph<V, E>> wcc = getConnectedParts(g, Graph.Relation.WEAK);

		out.println(String.format("Strongly Connected Clusters: %1$d; Largest: %2$d", scc.size(), scc.get(0).getVertexCount()));
		out.println(String.format("Weakly Connected Clusters: %1$d; Largest: %2$d", wcc.size(), scc.get(0).getVertexCount()));
	}
	
	public static void main(String[] args) {
		int w = 16;
		int h = 16;
		int n = w*h;
		int d = 4;
		Graph<Integer, Boolean> g;
		
		/*
		System.out.println("== Grid Graph, Width: "+w+", Height: "+h+" ==");
		g = GraphGenerator.regularGridGraph(w, h);
		dumpStats(g, System.out);

		int b = 2;
		System.out.println("== Circle Graph, Size: "+n+", Neighbourhood: "+b+" ==");
		g = GraphGenerator.regularCircleGraph(n, b);
		dumpStats(g, System.out);

		System.out.println("== Complete Graph, Size: "+n+" ==");
		g = GraphGenerator.completeGraph(n);
		dumpStats(g, System.out);
		*/
		
		GraphGenerator<Integer, Boolean> generator = GraphGenerator.dummyInstance(); 

		System.out.println("== Random Graph, Size: "+n+", Degree: "+d+" ==");
		g = generator.randomGraph(n, d);
		dumpStats(g, System.out);
		
		double cf = 0.3;
		System.out.println("== Kumar Graph, Size: "+n+", Degree: "+d+", CopyFactor: "+cf+" ==");
		g = generator.kumarGraph(n, d, cf);
		dumpStats(g, System.out);

		System.out.println("== Barabasi Graph, Size: "+n+", Degree: "+d+" ==");
		g = generator.barabasiGraph(n, d);
		dumpStats(g, System.out);

	}
}
