package de.brightbyte.data;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;

public class SpectralBloomSet<T> extends AbstractBloomSet<T>{
	private int[] counters;
	
	public SpectralBloomSet(String algorithm) throws NoSuchAlgorithmException {
		this(MessageDigest.getInstance(algorithm));
	}
	
	public SpectralBloomSet(MessageDigest digest) {
		super(digest);
		int width = digest.getDigestLength();
		if (width==0) throw new IllegalArgumentException("digest does not report width: "+digest);
		
		this.counters = new int[width * 8];
	}
	
	protected boolean[] bools(byte[] bytes) {
		boolean[] bools = new boolean[bytes.length*8];
		
		int j = 0;
		for (byte b : bytes) {
			for (int i=0; i<8; i++) {
				int m = 1 << i;
				
				bools[j] = (b & m) > 0;
				
				j++;
			}
		}
		
		return bools;
	}
	
	public void add(T value) {
		put(value, 1);
	}
	
	public void put(T value, int count) {
		boolean[] h = bools(hash(value));
		
		for (int i=0; i<counters.length; i++) {
			if (h[i]) counters[i]+= count;
		}
	}
	
	public void remove(T value) {
		int c = get(value);
		if (c>0) remove(value, c);
	}
	
	public void remove(T value, int count) {
		if (count==0 || !contains(value)) return;
		
		boolean[] h = bools(hash(value));
		
		for (int i=0; i<counters.length; i++) {
			if (h[i]) counters[i]= Math.max(0, counters[i] - count);
		}
	}
	
	public boolean contains(T value) {
		boolean[] h = bools(hash(value));
		
		for (int i=0; i<counters.length; i++) {
			if (h[i] && counters[i]==0) return false;
		}
		
		return true;
	}

	public int get(T value) {
		boolean[] h = bools(hash(value));
		
		int v = Integer.MAX_VALUE;
		
		for (int i=0; i<counters.length; i++) {
			if (h[i]) {
				if (counters[i]==0) return 0;
				else if (v > counters[i]) v = counters[i];
			}
		}
		
		return v;
	}
	
	public static void main(String[] args) throws NoSuchAlgorithmException, IOException {
		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		
		SpectralBloomSet<String> set = new SpectralBloomSet<String>("MD5");
		System.out.printf("BloomSet bits: %d\n", set.getBits());
		
		while (true) {
			String s = in.readLine();
			if (s==null) break;
			
			s= s.trim();
			if (s.length()==0) break;
			
			set.add(s);
		}

		while (true) {
			String s = in.readLine();
			if (s==null) break;

			s= s.trim();
			if (s.length()==0) break;
			
			System.out.printf("%d\n", set.get(s));
		}
	}
}
