package de.brightbyte.data;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import de.brightbyte.util.PersistenceException;
import de.brightbyte.util.StringUtils;

public class PersistentIdManager implements IdManager<String> {

	private StringBuilder buffer = new StringBuilder();
	private int bufferSize;
	
	private Map<String, Integer> ids;
	private File store;
	private String encoding;
	private Writer out;
	
	private int nextId = 1;
	
	public PersistentIdManager(File store, String enc, int bufferSize) {
		this(new HashMap<String, Integer>(), store, enc, bufferSize);
	}
	
	public PersistentIdManager(Map<String, Integer> ids, File store, String enc, int bufferSize) {
		if (ids==null) throw new NullPointerException();
		if (store==null) throw new NullPointerException();
		
		this.ids = ids;
		this.store = store;
		this.encoding = enc;
		this.bufferSize = bufferSize;
	}
	
	public void flush() throws PersistenceException {
		try {
			if (buffer.length()==0) return;
			
			if (out==null) {
				OutputStream f = new FileOutputStream(store, true);
				out = encoding!=null ? new OutputStreamWriter(f, encoding) : new OutputStreamWriter(f);
			}
			
			out.write(buffer.toString());
			buffer.setLength(0);
			
			out.flush();
		} catch (IOException e) {
			out = null;
			throw new PersistenceException(e);
		}
	}
	
	public void clear() throws PersistenceException {
		ids.clear();

		close();
		store.delete();
		
		nextId = 1;
	}
	
	protected void close() throws PersistenceException {
		flush();
		
		try {
			if (out!=null) out.close();
			out = null;
		} catch (IOException e) {
			out = null;
			throw new PersistenceException(e);
		}
	}
	
	public int load() throws PersistenceException {
		if (!store.exists()) return 0; //nothing to load
		
		close();
		
		try {
			InputStream f = new FileInputStream(store);
			Reader rd = encoding!=null ? new InputStreamReader(f, encoding) : new InputStreamReader(f);
			BufferedReader in = new BufferedReader(rd); 
			
			String s;
			while ((s = in.readLine()) != null) {
				int idx = s.indexOf('\t');
				if (idx<0) {
					break; //FIXME: remove broken record before appending!
				}
				
				try {
					String n = s.substring(0, idx);
					int i = Integer.parseInt(s.substring(idx+1));
					
					if (nextId<=i) nextId = i +1;
					
					Integer old = ids.put(n, i);
					if (old != null && !old.equals(i)) throw new RuntimeException("multiple entries for key "+n+": was "+old+", found "+i);
				} catch (NumberFormatException e) {
					break; //FIXME: remove broken record before appending!
				}
			}
			
			in.close();
			
			return ids.size();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}
	
	protected void storeId(String name, int id) throws PersistenceException {
		Integer old = ids.put(name, id);
		if (old != null && !old.equals(id)) throw new RuntimeException("multiple entries for key "+name+": was "+old+", setting "+id);
		
		buffer.append(name);
		buffer.append('\t');
		buffer.append(String.valueOf(id));
		buffer.append('\n');
		
		if (buffer.length()>bufferSize) flush();
	}
	
	public int getMaxId() {
		return nextId -1;
	}
	
	public int aquireId(String x) throws PersistenceException {
		if (StringUtils.containsControlChar(x, true)) throw new IllegalArgumentException("name must not contain control characters (code < 32)");
		
		Integer id = ids.get(x);
		if (id==null) {
			id = nextId;
			nextId ++;
			
			storeId(x, id);
		}
		
		return id;
	}

	public int size() {
		return ids.size();
	}
}
