package de.brightbyte.data;

import java.util.Arrays;
import java.util.Iterator;

import junit.framework.TestCase;

public class IntRelationTest extends TestCase {
	
	protected IntRelation newIntRelation(int capacity) {
		return new IntRelation(capacity);
	}

	public void testFind() {
		int[] data = new int[] { 10, 20, 30, 40, 40 };
		
		assertEquals(-1, IntRelation.find( 5, data, 0));
		
		assertEquals(-1, IntRelation.find( 5, data, 1));
		assertEquals( 0, IntRelation.find(10, data, 1));
		assertEquals(-2, IntRelation.find(15, data, 1));
		
		assertEquals(-1, IntRelation.find( 5, data, 2));
		assertEquals( 0, IntRelation.find(10, data, 2));
		assertEquals(-2, IntRelation.find(15, data, 2));
		assertEquals( 1, IntRelation.find(20, data, 2));
		assertEquals(-3, IntRelation.find(25, data, 2));
		
		assertEquals(-1, IntRelation.find( 5, data, 3));
		assertEquals( 0, IntRelation.find(10, data, 3));
		assertEquals(-2, IntRelation.find(15, data, 3));
		assertEquals( 1, IntRelation.find(20, data, 3));
		assertEquals(-3, IntRelation.find(25, data, 3));
		assertEquals( 2, IntRelation.find(30, data, 3));
		assertEquals(-4, IntRelation.find(35, data, 3));
		
		assertEquals(-1, IntRelation.find( 5, data, 4));
		assertEquals( 0, IntRelation.find(10, data, 4));
		assertEquals(-2, IntRelation.find(15, data, 4));
		assertEquals( 1, IntRelation.find(20, data, 4));
		assertEquals(-3, IntRelation.find(25, data, 4));
		assertEquals( 2, IntRelation.find(30, data, 4));
		assertEquals(-4, IntRelation.find(35, data, 4));
		assertEquals( 3, IntRelation.find(40, data, 4));
		assertEquals(-5, IntRelation.find(45, data, 4));
		
		assertEquals(-1, IntRelation.find( 5, data, 5));
		assertEquals( 0, IntRelation.find(10, data, 5));
		assertEquals(-2, IntRelation.find(15, data, 5));
		assertEquals( 1, IntRelation.find(20, data, 5));
		assertEquals(-3, IntRelation.find(25, data, 5));
		assertEquals( 2, IntRelation.find(30, data, 5));
		assertEquals(-4, IntRelation.find(35, data, 5));
		assertEquals( 3, IntRelation.find(40, data, 5));
		assertEquals(-6, IntRelation.find(45, data, 5));
	}
	
	
	public void testGet() {
		IntRelation r = newIntRelation(3);
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );
		
		r.put(10, 30);
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );
		
		r.put(10, 130);
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );

		r.put(11, 33);
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );

		r.put(9, 27);
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );

		r.put(7, 21);
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );
		
		assertTrue( Arrays.equals(r.get(9), new int[] { 27 }) );
		assertTrue( Arrays.equals(r.get(7), new int[] { 21 }) );
		assertTrue( Arrays.equals(r.get(11), new int[] { 33 }) );
		assertTrue( Arrays.equals(r.get(17), new int[] { }) );
		
		r.put(7, new int[] {1,2,3,4,5,6,7,8,9,0,1,2,3,4,5,6,7,8,9,0,1,2,3,4,5,6,7,8,9,0});
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );
		
		int[] ten = r.get(10);
		assertTrue( Arrays.equals(ten, new int[] { 30, 130 }) || Arrays.equals(ten, new int[] { 130, 30 }) );
	}

	public void testContainsKey() {
		IntRelation r = newIntRelation(3);
		assertFalse( r.containsKey(1) );
		
		r.put(10, 30);
		assertTrue( r.containsKey(10) );
		
		r.put(10, 130);
		assertTrue( r.containsKey(10) );

		r.put(17, 33);
		assertTrue( r.containsKey(17) );

		r.remove(10, 130);
		assertTrue( r.containsKey(10) );

		r.remove(10, 30);
		assertFalse( r.containsKey(10) );
	}

	public void testPutIntIntArrayIntInt() {
		IntRelation r = newIntRelation(3);
		
		r.put(5, new int[] {2,4,8,16}, 1, 2);
		
		assertEquals(2, r.size());
		assertTrue( Arrays.equals(r.get(5), new int[] { 4, 8 }) );
		
		r.put(4, new int[] {3,5,7,11}, 1, 2);
		
		assertEquals(4, r.size());
		assertTrue( Arrays.equals(r.get(4), new int[] { 5, 7 }) );
	}

	public void testPutIntInt() {
		IntRelation r = newIntRelation(3);
		
		r.put(5, 4);
		r.put(5, 8);
		
		assertEquals(2, r.size());
		assertTrue( Arrays.equals(r.get(5), new int[] { 4, 8 }) || Arrays.equals(r.get(5), new int[] { 8, 4 }));
		
		r.put(4, 5);
		r.put(4, 7);
		
		assertEquals(4, r.size());
		assertTrue( Arrays.equals(r.get(4), new int[] { 5, 7 }) || Arrays.equals(r.get(4), new int[] { 7, 5 }));
	}

	public void testPutAll() {
		IntRelation r = newIntRelation(5);
		IntRelation s = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		
		s.put(7, 14);
		s.put(3, 18);
		s.put(4, 17);
		s.put(2, 12);

		
		r.putAll(s);
		assertEquals(8, r.size());
		assertTrue( Arrays.equals(r.get(5), new int[] { 4, 8 }) || Arrays.equals(r.get(5), new int[] { 8, 4 }));
		assertTrue( Arrays.equals(r.get(3), new int[] { 7, 18 }) || Arrays.equals(r.get(3), new int[] { 18, 7 }));		
		assertTrue( Arrays.equals(r.get(1), new int[] { 2 }) );		
		assertTrue( Arrays.equals(r.get(7), new int[] { 14 }) );		
		assertTrue( Arrays.equals(r.get(4), new int[] { 17 }) );		
		assertTrue( Arrays.equals(r.get(2), new int[] { 12 }) );		
	}

	public void testRemoveInt() {
		IntRelation r = newIntRelation(3);
		
		r.put(1, 4);
		r.put(2, 5);
		r.put(3, 6);
		r.put(2, 55);
		
		r.remove(100);
		assertEquals(4, r.size());
		
		r.remove(1);
		assertEquals(3, r.size());
		assertEquals(0, r.get(1).length);
		assertTrue( Arrays.equals(r.get(2), new int[] { 5, 55 }) || Arrays.equals(r.get(2), new int[] { 55, 5 }));
		assertTrue( Arrays.equals(r.get(3), new int[] { 6 }) );
		
		r.remove(2);
		assertEquals(1, r.size());
		assertEquals(0, r.get(2).length);
		assertTrue( Arrays.equals(r.get(3), new int[] { 6 }) );
	}

	public void testRemoveIntInt() {
		IntRelation r = newIntRelation(3);
		
		r.put(1, 4);
		r.put(2, 5);
		r.put(3, 6);
		r.put(2, 55);
		
		r.remove(1, 6);
		assertEquals(4, r.size());
		
		r.remove(1, 4);
		assertEquals(3, r.size());
		assertEquals(0, r.get(1).length);
		assertTrue( Arrays.equals(r.get(2), new int[] { 5, 55 }) || Arrays.equals(r.get(2), new int[] { 55, 5 }));
		assertTrue( Arrays.equals(r.get(3), new int[] { 6 }) );
		
		r.remove(2, 55);
		assertEquals(2, r.size());
		assertTrue( Arrays.equals(r.get(2), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.get(3), new int[] { 6 }) );
	}

	public void testKeys() {
		IntRelation r = newIntRelation(5);
		Iterator<Integer> it = r.keys().iterator();
		assertEquals(false, it.hasNext());
		
		r.put(1, 4);
		r.put(2, 5);
		r.put(5, 6);
		r.put(2, 55);
		
		it = r.keys().iterator();
		
		assertEquals(true, it.hasNext());
		assertEquals(1, (int)it.next());
		
		assertEquals(true, it.hasNext());
		assertEquals(2, (int)it.next());
		
		it.remove();
		
		assertEquals(true, it.hasNext());
		assertEquals(5, (int)it.next());
		
		assertEquals(false, it.hasNext());
		
		assertEquals(0, r.get(2).length);
	}

	public void testGetKeys() {
		IntRelation r = newIntRelation(5);
		
		r.put(1, 4);
		r.put(2, 5);
		r.put(5, 6);
		r.put(2, 55);
		
		IntList keys = r.getKeys();
		
		assertEquals(3, keys.size());
		assertEquals(1, keys.getInt(0));
		assertEquals(2, keys.getInt(1));
		assertEquals(5, keys.getInt(2));
	}

	public void testRemoveValue() {
		IntRelation r = newIntRelation(10);
		
		r.put(3, 2);
		r.put(3, 4);
		r.put(3, 7);
		r.put(4, 5);
		r.put(6, 6);
		r.put(7, 2);
		r.put(7, 4);
		r.put(7, 4);
		r.put(7, 6);
		r.put(9, 2);

		r.removeValue(4);
		
		assertEquals(7, r.size());
		assertTrue( Arrays.equals(r.get(3), new int[] { 2, 7 }) || Arrays.equals(r.get(3), new int[] { 7, 2 }));
		assertTrue( Arrays.equals(r.get(4), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.get(6), new int[] { 6 }) );
		assertTrue( Arrays.equals(r.get(7), new int[] { 2, 6 }) || Arrays.equals(r.get(7), new int[] { 6, 2 }) );
		assertTrue( Arrays.equals(r.get(9), new int[] { 2 }) );
		assertEquals(5, r.getKeys().size());
		
		r.removeValue(2);
		
		assertEquals(4, r.size());
		assertTrue( Arrays.equals(r.get(3), new int[] { 7 }) );
		assertTrue( Arrays.equals(r.get(4), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.get(6), new int[] { 6 }) );
		assertTrue( Arrays.equals(r.get(7), new int[] { 6 }) );
		assertEquals(4, r.getKeys().size());
	}

	public void testGetSinks() {
		//simple tree --------------------------
		IntRelation r = newIntRelation(10);

		r.put(1, 11);
		r.put(1, 12);
		r.put(11, 111);
		r.put(11, 112);
		r.put(11, 113);
		r.put(12, 121);
		r.put(12, 122);
		r.put(122, 1221);
		r.put(1221, 12211);
		r.put(12211, 122111);
		r.put(12211, 122112);
		
		int[] sinks = r.getSinks().toIntArray();
		Arrays.sort(sinks);
		assertTrue(Arrays.equals(sinks, new int[] { 111, 112, 113, 121, 122111, 122112 }));

		//tree with looped leaf --------------------------
		r = newIntRelation(10);
		
		r.put(1, 11); 
		r.put(1, 12); //kept
		r.put(11, 111);
		r.put(11, 112); 
		r.put(11, 113);
		r.put(12, 121);
		r.put(12, 122); //kept
		r.put(122, 1221); //kept
		r.put(1221, 12211); //kept
		r.put(12211, 122111); 
		r.put(12211, 122112); //kept
		r.put(122112, 122112); //loop!
		
		sinks = r.getSinks().toIntArray();
		Arrays.sort(sinks);
		assertTrue(Arrays.equals(sinks, new int[] { 111, 112, 113, 121, 122111 }));
				
		//circle --------------------------
		r = newIntRelation(10);
		
		r.put(1, 2); 
		r.put(2, 3); 
		r.put(3, 4); 
		r.put(4, 1); 
		
		sinks = r.getSinks().toIntArray();
		assertTrue(Arrays.equals(sinks, new int[] { }));
		
		//pseudo-tree with inner cycle an d cross-links --------------------------
		r = newIntRelation(10);
		
		r.put(1, 11); 
		r.put(1, 12); //kept
		r.put(11, 111); 
		r.put(11, 112); 
		r.put(11, 113); 
		r.put(12, 121); 
		r.put(12, 122);  //live
		r.put(122, 11); //cross-link 
		r.put(122, 1221); //live 
		r.put(122, 1222); 
		r.put(1221, 12211); 
		r.put(1221, 12); //backlink!
		r.put(1222, 113); //cross-link 
		r.put(12211, 122111); 
		r.put(12211, 122112); 
		
		sinks = r.getSinks().toIntArray();
		Arrays.sort(sinks);
		assertTrue(Arrays.equals(sinks, new int[] { 111, 112, 113, 113, 121, 122111, 122112 })); //WARNING: non-uniq		
	}

	public void testCount() {
		IntRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		r.put(1, 8);
		
		assertEquals(0, r.count(100));

		assertEquals(1, r.count(3));
		assertEquals(2, r.count(5));
		assertEquals(2, r.count(1));
		
		r.remove(1, 8);
		assertEquals(1, r.count(1));
		
		r.remove(5);
		assertEquals(0, r.count(5));
	}
	
	public void testCountIgnoring() {
		IntRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		r.put(1, 8);
		
		assertEquals(0, r.countIgnoring(100, 77));

		assertEquals(2, r.countIgnoring(5, 77));
		assertEquals(1, r.countIgnoring(5, 4));
		assertEquals(1, r.countIgnoring(3, 77));
		assertEquals(0, r.countIgnoring(3, 7));
	}
	
	public void testContainsKeyIgnoring() {
		IntRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		r.put(1, 8);
		
		assertEquals(false, r.containsKeyIgnoring(100, 77));

		assertEquals(true, r.containsKeyIgnoring(5, 77));
		assertEquals(true, r.containsKeyIgnoring(5, 4));
		assertEquals(true, r.containsKeyIgnoring(3, 77));
		assertEquals(false, r.containsKeyIgnoring(3, 7));
	}
	
	public void testReplace() {
		IntRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		
		r.replace(5, 4, 40);
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(5), new int[] {8, 40}) || Arrays.equals(r.get(5), new int[] {40, 8}) );
		
		r.replace(3, 9, 90);
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(3), new int[] {7}) );
		
		r.replace(9, 7, 70);
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(9), new int[] {}) );
	}
	
	public void testFill() {
		IntRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		
		r.fill(5, 40);
		assertEquals( 2, r.count(5) );
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(5), new int[] { 40, 40 }) );
		
		r.fill(3, 90);
		assertEquals( 1, r.count(3) );
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(3), new int[] { 90 }) );
		
		r.fill(9, 70);
		assertEquals( 0, r.count(9) );
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(9), new int[] { }) );
	}
	
	public void testRemoveIntArray() {
		IntRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		r.put(1, 8);
		r.put(7, 5);
		
		r.remove( new int[] {3, 1} );
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.get(5), new int[] { 4, 8 }) || Arrays.equals(r.get(5), new int[] { 8, 4 }) );
		assertTrue( Arrays.equals(r.get(7), new int[] { 5 }) );		
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );		
		assertTrue( Arrays.equals(r.get(3), new int[] { }) );		

		r.remove( new int[] { 11 } );
		assertEquals( 3, r.size() );

		r.remove( new int[] { 13, 1, 7 } );
		assertEquals( 2, r.size() );
		assertTrue( Arrays.equals(r.get(5), new int[] { 4, 8 }) || Arrays.equals(r.get(5), new int[] { 8, 4 }) );
		assertTrue( Arrays.equals(r.get(7), new int[] { }) );		
		assertTrue( Arrays.equals(r.get(1), new int[] { }) );		
		assertTrue( Arrays.equals(r.get(3), new int[] { }) );		
	}
	

	public void testGetSources() {
		//simple tree --------------------------
		IntRelation r = newIntRelation(10);

		r.put(11, 1);
		r.put(12, 1);
		r.put(111, 11);
		r.put(112, 11);
		r.put(113, 11);
		r.put(121, 12);
		r.put(122, 12);
		r.put(1221, 122);
		r.put(12211, 1221);
		r.put(122111, 12211);
		r.put(122112, 12211);
		
		int[] sinks = r.getSources().toIntArray();
		Arrays.sort(sinks);
		assertTrue(Arrays.equals(sinks, new int[] { 111, 112, 113, 121, 122111, 122112 }));

		//tree with looped leaf --------------------------
		r = newIntRelation(10);
		
		r.put(11, 1); 
		r.put(12, 1); //kept
		r.put(111, 11);
		r.put(112, 11); 
		r.put(113, 11);
		r.put(121, 12);
		r.put(122, 12); //kept
		r.put(1221, 122); //kept
		r.put(12211, 1221); //kept
		r.put(122111, 12211); 
		r.put(122112, 12211); //kept
		r.put(122112, 122112); //loop!
		
		sinks = r.getSources().toIntArray();
		Arrays.sort(sinks);
		assertTrue(Arrays.equals(sinks, new int[] { 111, 112, 113, 121, 122111 }));
				
		//circle --------------------------
		r = newIntRelation(10);
		
		r.put(2, 1); 
		r.put(3, 2); 
		r.put(4, 3); 
		r.put(1, 4); 
		
		sinks = r.getSources().toIntArray();
		assertTrue(Arrays.equals(sinks, new int[] { }));
		
		//pseudo-tree with inner cycle an d cross-links --------------------------
		r = newIntRelation(10);
		
		r.put(11, 1); 
		r.put(12, 1); //kept
		r.put(111, 11); 
		r.put(112, 11); 
		r.put(113, 11); 
		r.put(121, 12); 
		r.put(122, 12);  //live
		r.put(11, 122); //cross-link 
		r.put(1221, 122); //live 
		r.put(1222, 122); 
		r.put(12211, 1221); 
		r.put(12, 1221); //backlink!
		r.put(113, 1222); //cross-link 
		r.put(122111, 12211); 
		r.put(122112, 12211); 
		
		sinks = r.getSources().toIntArray();
		Arrays.sort(sinks);
		assertTrue(Arrays.equals(sinks, new int[] { 111, 112, 113, 121, 122111, 122112 }) || Arrays.equals(sinks, new int[] { 111, 112, 113, 113, 121, 122111, 122112 })); 		
	}
	
	public void testPruneSinks() {
		int c;
		IntRelation r;
		
		//small --------------------------
		r = newIntRelation(10);
		r.put(1, 11);
		
		c = r.pruneSinks();
		assertEquals(1, c);
		assertEquals(0, r.size());

		//-----------
		r = newIntRelation(10);
		r.put(1, 11);
		r.put(11, 111);
		
		c = r.pruneSinks();
		assertEquals(2, c);
		assertEquals(0, r.size());

		//-----------
		r = newIntRelation(10);
		r.put(1, 11);
		r.put(11, 111);
		r.put(111, 11);
		
		c = r.pruneSinks();
		assertEquals(0, c);
		assertEquals(3, r.size());

		//-----------
		r = newIntRelation(10);
		r.put(1, 11);
		r.put(11, 111);
		r.put(11, 1);
		
		c = r.pruneSinks();
		assertEquals(1, c);
		assertEquals(2, r.size());

		//simple tree --------------------------
		r = newIntRelation(10);

		r.put(1, 11);
		r.put(1, 12);
		r.put(11, 111);
		r.put(11, 112);
		r.put(11, 113);
		r.put(12, 121);
		r.put(12, 122);
		r.put(122, 1221);
		r.put(1221, 12211);
		r.put(12211, 122111);
		r.put(12211, 122112);
		
		c = r.pruneSinks();
		assertEquals(11, c);
		assertEquals(0, r.size());
		assertEquals(0, r.get(1).length);

		//tree with looped root --------------------------
		r = newIntRelation(10);
		
		r.put(1, 1); //loop!
		r.put(1, 11);
		r.put(1, 12);
		r.put(11, 111);
		r.put(11, 112);
		r.put(11, 113);
		r.put(12, 121);
		r.put(12, 122);
		r.put(122, 1221);
		r.put(1221, 12211);
		r.put(12211, 122111);
		r.put(12211, 122112);
		
		c = r.pruneSinks();
		assertEquals(11, c);
		assertEquals(1, r.size());
		assertTrue( Arrays.equals(r.get(1), new int[] { 1 }) );
		assertTrue( Arrays.equals(r.get(11), new int[] { }) );

		//tree with looped leaf --------------------------
		r = newIntRelation(10);
		
		r.put(1, 11); 
		r.put(1, 12); //kept
		r.put(11, 111);
		r.put(11, 112); 
		r.put(11, 113);
		r.put(12, 121);
		r.put(12, 122); //kept
		r.put(122, 1221); //kept
		r.put(1221, 12211); //kept
		r.put(12211, 122111); 
		r.put(12211, 122112); //kept
		r.put(122112, 122112); //loop!
		
		c = r.pruneSinks();
		assertEquals(6, c);
		assertEquals(6, r.size());
		assertTrue( Arrays.equals(r.get(11), new int[] { }) );
		assertTrue( Arrays.equals(r.get(121), new int[] { }) );
		assertTrue( Arrays.equals(r.get(1), new int[] { 12 }) );
		assertTrue( Arrays.equals(r.get(12), new int[] { 122 }) );
		assertTrue( Arrays.equals(r.get(122), new int[] { 1221 }) );
		assertTrue( Arrays.equals(r.get(1221), new int[] { 12211 }) );
		assertTrue( Arrays.equals(r.get(12211), new int[] { 122112 }) );
		assertTrue( Arrays.equals(r.get(122112), new int[] { 122112 }) );
				
		//circle --------------------------
		r = newIntRelation(10);
		
		r.put(1, 2); 
		r.put(2, 3); 
		r.put(3, 4); 
		r.put(4, 1); 
		
		c = r.pruneSinks();
		assertEquals(0, c);
		assertEquals(4, r.size());
		assertTrue( Arrays.equals(r.get(1), new int[] { 2 }) );
		assertTrue( Arrays.equals(r.get(4), new int[] { 1 }) );
		
		//pseudo-tree with inner cycle and cross-links --------------------------
		r = newIntRelation(10);
		
		r.put(1, 11); 
		r.put(1, 12); //kept
		r.put(11, 111); 
		r.put(11, 112); 
		r.put(11, 113); 
		r.put(12, 121); 
		r.put(12, 122);  //live
		r.put(122, 11); //cross-link 
		r.put(122, 1221); //live 
		r.put(122, 1222); 
		r.put(1221, 12211); 
		r.put(1221, 12); //backlink!
		r.put(1222, 113); //cross-link 
		r.put(12211, 122111); 
		r.put(12211, 122112); 
		
		c = r.pruneSinks();
		assertEquals(11, c);
		assertEquals(4, r.size());
		assertTrue( Arrays.equals(r.get(1), new int[] { 12 }) );
		assertTrue( Arrays.equals(r.get(11), new int[] { }) );
		assertTrue( Arrays.equals(r.get(113), new int[] { }) );
		assertTrue( Arrays.equals(r.get(1222), new int[] { }) );
		
		assertTrue( Arrays.equals(r.get(12), new int[] { 122 }) );
		assertTrue( Arrays.equals(r.get(122), new int[] { 1221 }) );
		assertTrue( Arrays.equals(r.get(1221), new int[] { 12 }) );
	}
	
}
