package de.brightbyte.data;

import java.io.IOException;
import java.io.Serializable;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.RandomAccess;

import de.brightbyte.io.ConsoleIO;
import de.brightbyte.io.Output;

public class IntList extends AbstractList<Integer> implements Serializable, Cloneable, RandomAccess {
	
	protected int[] data;
	protected int size;
	
	public IntList() {
		this(10);
	}

	public IntList(int capacity) {
		ensureCapacity(capacity);
	}

	public IntList(Collection<? extends Integer> col) {
		this(col.size());
		addAll(col);
	}

	public IntList(IntList list) {
		this(list.size());
		addAll(list);
	}

	public IntList(int[] a) {
		this(a.length);
		addAll(0, a);
	}

	public IntList(int[] a, int ofs, int len) {
		this(len);
		addAll(0, a, ofs, len);
	}

	@Override
	public boolean addAll(int index, Collection<? extends Integer> c) {
		ensureCapacity(size + c.size()); //grow in one go
		return super.addAll(index, c);
	}

	public boolean addAll(int index, int[] a) {
		return addAll(index, a, 0, a.length);
	}

	public boolean addAll(int index, int[] a, int ofs, int len) {
		if (len > a.length - ofs) len = a.length - ofs;
		if (len<=0) return false;
		
		shift(index, len);
		System.arraycopy(a, ofs, data, index, len);
		return true;
	}

	public void ensureCapacity(int cap) {
		if (data==null) {
			data = new int[cap];
		}
		else if (data.length<cap) {
			long c = (long)size*3/2;
			if (c<cap) c = cap;
			if (c>Integer.MAX_VALUE) c = Integer.MAX_VALUE;
			
			int[] s = new int[(int)c];
			System.arraycopy(data, 0, s, 0, size);
			data = s;
		}
	}
	
	protected void shift(int pos, int ofs) {
		if (pos<0) throw new IndexOutOfBoundsException("index out of range: "+pos+" < 0");
		if (pos>size) throw new IndexOutOfBoundsException("index out of range: "+pos+" >= "+size);
		
		if (size+ofs>=data.length) ensureCapacity(size + ofs);

		if (pos>=size) {
			size += ofs;
			return;
		}
		
		int from = ofs < 0 ? pos - ofs : pos ;
		int to   = ofs < 0 ? pos : pos + ofs ;
		
		System.arraycopy(data, from, data, to, size - from);
		size += ofs;
	}
	
	@Override
	public int indexOf(Object value) {
		if (!(value instanceof Integer)) return -1;
		return indexOfInt((Integer)value);
	}
	
	public int indexOfInt(int value) {
		for (int i = 0; i<size; i++) {
			if (data[i] == value) return i;
		}
		
		return -1;
	}
	
	@Override
	public int lastIndexOf(Object value) {
		if (!(value instanceof Integer)) return -1;
		return lastIndexOfInt((Integer)value);
	}
	
	public int lastIndexOfInt(int value) {
		for (int i = size-1; i<=0; i--) {
			if (data[i] == value) return i;
		}
		
		return -1;
	}
	
	@Override
	public Integer remove(int index) {
		return removeInt(index);
	}
	
	public int removeInt(int index) {
		int v = getInt(index);
		shift(index, -1);
		return v;
	}
	
	@Override
	protected void removeRange(int fromIndex, int toIndex) {
		if (toIndex<fromIndex) throw new IndexOutOfBoundsException("toIndex "+toIndex+" <= fromIndex "+fromIndex);
		
		if (toIndex>=size) size = fromIndex;
		else shift(fromIndex, toIndex - fromIndex);
	}
	
	@Override
	public void clear() {
		size= 0;
	}
	
	@Override
	public void add(int index, Integer value) {
		addInt(index, value);
	}
	
	public void addInt(int value) {
		addInt(size, value);
	}
	
	public void addInt(int index, int value) {
		shift(index, 1);
		setInt(size-1, value);
	}

	@Override
	public Integer set(int index, Integer value) {		shift(index, 1);
	setInt(size-1, value);

		return setInt(index, value);
	}

	public int setInt(int index, int value) {
		if (index<0) throw new NoSuchElementException("index out of range: "+index+" < 0");
		if (index>=size) throw new NoSuchElementException("index out of range: "+index+" >= "+size);
		
		int old = data[index];
		data[index] = value;
		return old;
	}

	@Override
	public Integer get(int index) {
		return getInt(index);
	}

	public int getInt(int index) {
		if (index<0) throw new NoSuchElementException("index out of range: "+index+" < 0");
		if (index>=size) throw new NoSuchElementException("index out of range: "+index+" >= "+size);
		return data[index];
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public IntList clone() {
		try {
			IntList c = (IntList) super.clone();
			
			c.data = new int[data.length];
			System.arraycopy(data, 0, c.data, 0, size);
			
			return c;
		} catch (CloneNotSupportedException e) {
			throw new Error("CloneNotSupportedException while cloning cloneable!", e);
		}
	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = super.hashCode();
		result = PRIME * result + Arrays.hashCode(data);
		result = PRIME * result + size;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (null == obj)
			return false;

		if (this == obj)
			return true;

		if (obj instanceof List) {
			if (obj instanceof IntList) {
				final IntList other = (IntList) obj;
				if (size != other.size)
					return false;
				if (!Arrays.equals(data, other.data))
					return false;
				return true;
			}
			else {
				return super.equals(obj);
			}
		}
		else {
			return false;
		}
	}
	
	 private void readObject(java.io.ObjectInputStream stream)
     	throws IOException, ClassNotFoundException {
		 		 
		 size = stream.readInt();
		 for (int i=0; i<size; i++) {
			 data[i] = stream.readInt();
		 }
	 }
	 
	 private void writeObject(java.io.ObjectOutputStream stream)
     	throws IOException {
		 
		 stream.writeInt(size);
		 for (int i=0; i<size; i++) {
			 stream.write(data[i]);
		 }
	 }
	
	 protected static void benchmarkIntListIndexOf(int runs, IntList list) {
		 int x = list.size() * 2 / 3;
		 for (int i=0; i<runs; i++) {
			 list.indexOfInt(x);
		 }
	 }
	 
	 protected static void benchmarkListIndexOf(int runs, List list) {
		 int x = list.size() * 2 / 3;
		 for (int i=0; i<runs; i++) {
			 list.indexOf(x);
		 }
	 }
	 
	 
	public int[] toIntArray() {
		return toIntArray(null);
	}

	public int[] toIntArray(int[] a) {
		if (a==null || a.length<size) a = new int[size];
		System.arraycopy(data, 0, a, 0, size);
		return a;
	}
	
	public void sort() {
		Arrays.sort(data);
	}

	protected static void benchmark(int runs, int depth, Output out) {
		 long t;
		 
		 t = System.currentTimeMillis();
		 IntList ilist = new IntList(10);
		 for (int i=0; i<depth; i++) {
			 ilist.add(i);
		 }
		 out.println("IntList setup("+depth+"): "+(System.currentTimeMillis() - t));

		 t = System.currentTimeMillis();
		 ArrayList<Integer> alist = new ArrayList(10);
		 for (int i=0; i<depth; i++) {
			 alist.add(i);
		 }
		 out.println("ArrayList setup("+depth+"): "+(System.currentTimeMillis() - t));
		 
		 //--------------------------------------------------------------------------
		 
		 t = System.currentTimeMillis();
		 benchmarkIntListIndexOf(runs, ilist);
		 out.println("Known IntList indexOf x "+runs+": "+(System.currentTimeMillis() - t));
		 
		 t = System.currentTimeMillis();
		 benchmarkListIndexOf(runs, ilist);
		 out.println("IntList indexOf x "+runs+": "+(System.currentTimeMillis() - t));
		 
		 t = System.currentTimeMillis();
		 benchmarkListIndexOf(runs, alist);
		 out.println("ArrayList indexOf x "+runs+": "+(System.currentTimeMillis() - t));
	 }
	 
	 public static void main(String[] args) {
		 benchmark(100000, 100000, ConsoleIO.output);
		 ConsoleIO.output.println("-----------------------------");
		 benchmark(100000, 100000, ConsoleIO.output);
		 ConsoleIO.output.println("-----------------------------");
		 benchmark(100000, 100000, ConsoleIO.output);
		 ConsoleIO.output.println("-----------------------------");
	 }
}
