package de.brightbyte.data;

import java.util.Arrays;

public class IntBiRelationTest extends IntRelationTest {
	
	@Override
	protected IntBiRelation newIntRelation(int capacity) {
		return new IntBiRelation(capacity);
	}

	public void testPruneSinksIntArray() {
		//simple tree --------------------------
		IntBiRelation r = newIntRelation(10);

		r.put(1, 11);
		r.put(1, 12);
		r.put(11, 111);
		r.put(11, 112);
		r.put(11, 113);
		r.put(12, 121);
		r.put(12, 122);
		r.put(122, 1221);
		r.put(1221, 12211);
		r.put(12211, 122111);
		r.put(12211, 122112);
		
		int c = r.pruneSinks( new int[] {111, 112, 113, 122112} );
		assertEquals(5, c);
		assertEquals(6, r.size());
		assertEquals(1, r.get(1).length);
		assertEquals(0, r.get(11).length);
		assertEquals(1, r.get(12211).length);
				
		//circle --------------------------
		r = newIntRelation(10);
		
		r.put(1, 2); 
		r.put(2, 3); 
		r.put(3, 4); 
		r.put(4, 1); 
		
		c = r.pruneSinks( new int[] {2, 4});
		assertEquals(0, c);
		assertEquals(4, r.size());
		assertTrue( Arrays.equals(r.get(1), new int[] { 2 }) );
		assertTrue( Arrays.equals(r.get(4), new int[] { 1 }) );
		
		//pseudo-tree with inner cycle and cross-links --------------------------
		r = newIntRelation(10);
		
		r.put(1, 11); 
		r.put(1, 12); //kept
		r.put(11, 111); 
		r.put(11, 112); 
		r.put(11, 113); 
		r.put(12, 121); 
		r.put(12, 122);  //live
		r.put(122, 11); //cross-link 
		r.put(122, 1221); //live 
		r.put(122, 1222); 
		r.put(1221, 12211); 
		r.put(1221, 12); //backlink!
		r.put(1222, 113); //cross-link 
		r.put(12211, 122111); 
		r.put(12211, 122112); 
		
		c = r.pruneSinks( new int[] {112, 113, 122112} );
		assertEquals(5, c);
		assertEquals(10, r.size());
		assertTrue( Arrays.equals(r.get(1), new int[] { 12, 11 }) || Arrays.equals(r.get(1), new int[] { 11, 12 }) );
		assertTrue( Arrays.equals(r.get(11), new int[] { 111 }) );
		assertTrue( Arrays.equals(r.get(113), new int[] { }) );
		assertTrue( Arrays.equals(r.get(1222), new int[] { }) );
		
		assertTrue( Arrays.equals(r.get(12), new int[] { 122, 121 }) || Arrays.equals(r.get(12), new int[] { 121, 122 }) );
		assertTrue( Arrays.equals(r.get(122), new int[] { 1221, 11 }) || Arrays.equals(r.get(122), new int[] { 11, 1221 }) );
		assertTrue( Arrays.equals(r.get(1221), new int[] { 12, 12211 }) || Arrays.equals(r.get(1221), new int[] { 12211, 12 }));
	}
	
	@Override
	public void testPutAll() {
		super.testPutAll();
		
		IntBiRelation r = newIntRelation(5);
		IntBiRelation s = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		
		s.put(7, 14);
		s.put(3, 8);
		s.put(4, 7);
		s.put(2, 14);

		
		r.putAll(s);
		assertEquals(8, r.size());
		assertTrue( Arrays.equals(r.getKeys(2), new int[] { 1 }) );		
		assertTrue( Arrays.equals(r.getKeys(4), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.getKeys(7), new int[] { 3, 4 }) || Arrays.equals(r.getKeys(7), new int[] { 4, 3 }) );		
		assertTrue( Arrays.equals(r.getKeys(8), new int[] { 3, 5 }) || Arrays.equals(r.getKeys(8), new int[] { 5, 3 }) );		
		assertTrue( Arrays.equals(r.getKeys(14), new int[] { 2, 7 }) || Arrays.equals(r.getKeys(14), new int[] { 7, 2 }) );		
	}
	
	public void testCountKeys() {
		IntBiRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		r.put(7, 2);
		
		assertEquals(0, r.count(100));

		assertEquals(2, r.countKeys(2));
		assertEquals(1, r.countKeys(7));
		assertEquals(1, r.countKeys(8));

		r.remove(3);
		assertEquals(0, r.countKeys(3));

		r.remove(1);
		assertEquals(1, r.countKeys(2));
	}

	@Override
	public void testFill() {		
		super.testFill();
		
		IntBiRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		
		r.fill(5, 40);
		assertEquals( 2, r.countKeys(40) );
		assertTrue( Arrays.equals(r.getKeys(40), new int[] { 5, 5 }) );
		
		r.fill(3, 90);
		assertEquals( 1, r.countKeys(90) );
		assertTrue( Arrays.equals(r.getKeys(90), new int[] { 3 }) );
		
		r.fill(9, 70);
		assertEquals( 0, r.countKeys(70) );
		assertTrue( Arrays.equals(r.get(70), new int[] { }) );
	}

	@Override
	public void testPutIntInt() {
		super.testPutIntInt();
		
		IntBiRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(3, 8);
		
		assertTrue( Arrays.equals(r.getKeys(8), new int[] { 3, 5 }) || Arrays.equals(r.getKeys(8), new int[] { 5, 3 }) );		
		assertTrue( Arrays.equals(r.getKeys(7), new int[] { 3 }) );		
	}

	@Override
	public void testPutIntIntArrayIntInt() {
		super.testPutIntIntArrayIntInt();
		
		IntBiRelation r = newIntRelation(3);
		
		r.put(5, new int[] {2,4,8,16}, 1, 2);
		
		assertEquals(2, r.size());
		assertTrue( Arrays.equals(r.getKeys(8), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.getKeys(5), new int[] { }) );
	}

	@Override
	public void testRemoveInt() {
		IntBiRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(3, 8);
		
		r.remove(3);
		assertTrue( Arrays.equals(r.getKeys(8), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.getKeys(7), new int[] { }) );
		assertTrue( Arrays.equals(r.getKeys(4), new int[] { 5 }) );
	}

	@Override
	public void testRemoveIntArray() {
		super.testRemoveIntArray();

		IntBiRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		r.put(1, 2);
		r.put(1, 8);
		r.put(7, 5);
		
		r.remove( new int[] {3, 1} );
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.getKeys(8), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.getKeys(7), new int[] { }) );		
		assertTrue( Arrays.equals(r.getKeys(4), new int[] { 5 }) );		
		assertTrue( Arrays.equals(r.getKeys(2), new int[] { }) );		
	}

	@Override
	public void testRemoveIntInt() {
		super.testRemoveIntInt();
		
		IntBiRelation r = newIntRelation(3);
		
		r.put(1, 4);
		r.put(2, 5);
		r.put(3, 6);
		r.put(2, 55);
		r.put(5, 4);
		
		r.remove(1, 4);
		assertTrue( Arrays.equals(r.getKeys(4), new int[] { 5 }) );
		assertTrue( Arrays.equals(r.getKeys(6), new int[] { 3 }) );
	}

	@Override
	public void testRemoveValue() {
		super.testRemoveValue();
		
		IntBiRelation r = newIntRelation(3);
		
		r.put(4, 1);
		r.put(5, 2);
		r.put(6, 3);
		r.put(55, 2);
		
		r.removeValue(100);
		assertEquals(4, r.size());
		
		r.removeValue(1);
		assertEquals(3, r.size());
		assertEquals(0, r.get(1).length);
		assertTrue( Arrays.equals(r.getKeys(2), new int[] { 5, 55 }) || Arrays.equals(r.getKeys(2), new int[] { 55, 5 }));
		assertTrue( Arrays.equals(r.getKeys(3), new int[] { 6 }) );
		
		r.removeValue(2);
		assertEquals(1, r.size());
		assertEquals(0, r.get(2).length);
		assertTrue( Arrays.equals(r.getKeys(3), new int[] { 6 }) );
	}

	@Override
	public void testReplace() {
		super.testReplace();
		
		IntBiRelation r = newIntRelation(5);
		
		r.put(5, 4);
		r.put(5, 8);
		r.put(3, 7);
		
		r.replace(5, 4, 40);
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.getKeys(40), new int[] {5}) );
		
		r.replace(3, 9, 90);
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.getKeys(90), new int[] {}) );
		
		r.replace(9, 7, 70);
		assertEquals( 3, r.size() );
		assertTrue( Arrays.equals(r.getKeys(70), new int[] {}) );
	}	
	
}
