package de.brightbyte.data;

import java.util.Arrays;
import java.util.Iterator;

import junit.framework.TestCase;

public class ChunkyBitSetTest extends TestCase {
	protected static final int LOW = 23;
	protected static final int MED = 12345 * 8;
	protected static final int HI = 100001 * 17;
	
	public void testGet() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		assertEquals(false, a.get(LOW));
		
		a.set(LOW, true);
		assertEquals(true, a.get(LOW));
		
		assertEquals(false, a.get(MED));
		
		a.set(MED, true);
		assertEquals(true, a.get(MED));
	}

	public void testContainsAny() {
		ChunkyBitSet a = new ChunkyBitSet();
		int[] x = new int[] { 0, LOW, MED, MED + 5};
		
		assertEquals(false, a.containsAny(x, 0, 0));
		assertEquals(false, a.containsAny(x, 0, 4));
		
		a.set(LOW, true);
		assertEquals(false, a.containsAny(x, 0, 0));
		assertEquals(false, a.containsAny(x, 1, 0));
		assertEquals(false, a.containsAny(x, 2, 2));		
		assertEquals(true,  a.containsAny(x, 0, 4));		
		assertEquals(true,  a.containsAny(x, 1, 1));		
		
		a.set(MED, true);
		assertEquals(false, a.containsAny(x, 0, 0));
		assertEquals(false, a.containsAny(x, 1, 0));
		assertEquals(true,  a.containsAny(x, 2, 2));		
		assertEquals(true,  a.containsAny(x, 0, 4));		
		assertEquals(true,  a.containsAny(x, 1, 1));		
		assertEquals(true,  a.containsAny(x, 2, 1));		
	}

	public void testSet() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		assertEquals(false, a.set(LOW, false));
		assertEquals(false, a.get(LOW));
		
		assertEquals(false, a.set(LOW, true));
		assertEquals(true, a.get(LOW));
		
		assertEquals(true, a.set(LOW, false));
		assertEquals(false, a.get(LOW));
		
		
		assertEquals(false, a.set(MED, false));
		assertEquals(false, a.get(MED));
		
		assertEquals(false, a.set(MED, true));
		assertEquals(true, a.get(MED));
		
		assertEquals(true, a.set(MED, false));
		assertEquals(false, a.get(MED));
		
		//---------
		a.set(31, false);
		assertEquals(false, a.get(31));

		a.set(32, false);
		assertEquals(false, a.get(32));

		a.set(33, true);
		assertEquals(true,  a.get(33));
		
		//---------
		
		a.set(31, false);
		a.set(32, false);
		a.set(33, true);
		
		assertEquals(false, a.get(31));
		assertEquals(false, a.get(32));
		assertEquals(true,  a.get(33));
		
		//---------
		a.set(63, false);
		assertEquals(false, a.get(63));

		a.set(64, false);
		assertEquals(false, a.get(64));

		a.set(65, true);
		assertEquals(true,  a.get(65));
		
		//---------
		
		a.set(62, true);
		a.set(63, false);
		a.set(64, false);
		a.set(65, true);
		
		assertEquals(true,  a.get(62));
		assertEquals(false, a.get(63));
		assertEquals(false, a.get(64));
		assertEquals(true,  a.get(65));
		
		//---------
		for (int i=LOW; i<=HI; i++) {
			a.set(i, (i%3==0));
		}

		for (int i=LOW; i<=HI; i++) {
			assertEquals("index: "+i, i%3==0, a.get(i));
		}
	}

	public void testAdd() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		assertEquals(true, a.add(LOW));
		assertEquals(true, a.get(LOW));
		
		assertEquals(false, a.add(LOW));
		assertEquals(true,  a.get(LOW));
		
		//---------
		assertEquals(true, a.add(MED));
		assertEquals(true, a.get(MED));
		
		assertEquals(false, a.add(MED));
		assertEquals(true,  a.get(MED));
	}

	public void testRemove() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		assertEquals(false, a.remove(LOW));
		assertEquals(false, a.get(LOW));
		
		a.add(LOW);
		assertEquals(true, a.remove(LOW));
		assertEquals(false, a.get(LOW));
		
		assertEquals(false, a.remove(MED));
		assertEquals(false, a.get(MED));
		
		a.add(MED);
		assertEquals(true, a.remove(MED));
		assertEquals(false, a.get(MED));
	}

	public void testSize() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		a.add(LOW);
		assertEquals(1, a.size());
		
		a.add(MED);
		assertEquals(2, a.size());
		
		a.add(HI);
		assertEquals(3, a.size());

		a.add(MED);
		assertEquals(3, a.size());
		
		//---------------------------

		a.remove(LOW + 6);
		assertEquals(3, a.size());
		
		a.remove(LOW);
		assertEquals(2, a.size());
		
		a.remove(LOW);
		assertEquals(2, a.size());
		
		a.remove(MED);
		assertEquals(1, a.size());
		
		a.remove(HI);
		assertEquals(0, a.size());
	}

	public void testAddAll() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		a.add(LOW);
		
		int n = a.addAll( new int[] {LOW, MED, HI});
		assertEquals(2, n);
		assertEquals(3, a.size());
	}

	public void testAddAllChunkyBitSet() {
		ChunkyBitSet a = new ChunkyBitSet();
		ChunkyBitSet b = new ChunkyBitSet();
		
		a.add(LOW);
		a.add(MED);
		a.add(MED+1);
		
		b.add(MED-1);
		b.add(MED);
		
		b.add(HI);
		b.add(HI+1);

		a.addAll(b);
		
		assertEquals(6, a.size());
		
		assertTrue(a.get(LOW));
		assertTrue(a.get(MED));
		assertTrue(a.get(MED+1));
		assertTrue(a.get(MED-1));
		assertTrue(a.get(HI));
		assertTrue(a.get(HI+1));
		
		assertFalse(a.get(HI-1));
		assertFalse(a.get(LOW+1));
	}

	public void testSetAll() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		a.setAll( new int[] {LOW, MED}, true);
		assertEquals(2, a.size());
		
		assertEquals(true,  a.get(LOW));
		assertEquals(true,  a.get(MED));
		assertEquals(false, a.get(HI));
		
		a.setAll( new int[] {LOW, HI}, false);
		assertEquals(1, a.size());
		
		assertEquals(false, a.get(LOW));
		assertEquals(true,  a.get(MED));
		assertEquals(false, a.get(HI));
	}

	public void testRemoveAll() {
		ChunkyBitSet a = new ChunkyBitSet();
		a.addAll( new int[] {LOW, MED, HI});
		
		int n = a.removeAll( new int[] {MED, HI});
		assertEquals(2, n);
		assertEquals(1, a.size());
		
		n = a.removeAll( new int[] {LOW, MED});
		assertEquals(1, n);
		assertEquals(0, a.size());
	}

	public void testToIntArray() {
		ChunkyBitSet a = new ChunkyBitSet();
		assertTrue(Arrays.equals(a.toIntArray(), new int[] {}));
		
		a.add(MED);
		assertTrue(Arrays.equals(a.toIntArray(), new int[] {MED}));

		a.add(LOW);
		assertTrue(Arrays.equals(a.toIntArray(), new int[] {LOW, MED}));

		a.add(HI);
		assertTrue(Arrays.equals(a.toIntArray(), new int[] {LOW, MED, HI}));
	}

	public void testClear() {
		ChunkyBitSet a = new ChunkyBitSet();
		a.addAll( new int[] {LOW, MED, HI});
		
		a.clear();
		assertEquals(true, a.isEmpty());
		assertEquals(0, a.size());
		
		assertEquals(false, a.get(LOW));
		assertEquals(false, a.get(MED));
		assertEquals(false, a.get(HI));
	}

	public void testFlip() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		a.flip(LOW);
		assertEquals(true, a.get(LOW));
		
		a.flip(LOW);
		assertEquals(false, a.get(LOW));
		
		a.flip(MED);
		assertEquals(true, a.get(MED));
		
		a.flip(MED);
		assertEquals(false, a.get(MED));
	}

	public void testHashCode() {
		ChunkyBitSet a = new ChunkyBitSet();
		int h0 = a.hashCode();
		
		a.set(LOW, false);
		assertEquals(h0, a.hashCode());
		
		a.set(HI, false);
		assertEquals(h0, a.hashCode());
		
		a.add(HI);
		a.remove(HI);
		assertEquals(h0, a.hashCode());
		
		//-------------------
		a.add(LOW);
		int h1 = a.hashCode();
		
		a.set(LOW, true);
		assertEquals(h1, a.hashCode());
		
		a.set(LOW+20, false);
		assertEquals(h1, a.hashCode());
		
		a.set(HI, false);
		assertEquals(h1, a.hashCode());
		
		a.add(HI);
		a.remove(HI);
		assertEquals(h1, a.hashCode());
	}

	public void testEqualsObject() {
		ChunkyBitSet a = new ChunkyBitSet();
		ChunkyBitSet b = new ChunkyBitSet();
		
		assertEquals(true, a.equals(b));
		assertEquals(true, b.equals(a));
		
		a.add(LOW);
		b.add(LOW);
		
		assertEquals(true, a.equals(b));
		assertEquals(true, b.equals(a));
		
		a.add(MED);
		b.flip(MED);
		
		assertEquals(true, a.equals(b));
		assertEquals(true, b.equals(a));
		
		a.add(HI);
		a.remove(HI);
		
		assertEquals(true, a.equals(b));
		assertEquals(true, b.equals(a));
	}

	public void testIterator() {
		ChunkyBitSet a = new ChunkyBitSet();
		
		a.add(LOW);
		
		a.addAll( new int[] {0, MED, MED+1, HI, LOW});
		Iterator<Integer> it = a.iterator();
		
		assertEquals(true, it.hasNext());
		assertEquals(0, (int)it.next());
		
		assertEquals(true, it.hasNext());
		assertEquals(LOW, (int)it.next());
		
		assertEquals(true, it.hasNext());
		assertEquals(MED, (int)it.next());
		
		assertEquals(true, it.hasNext());
		assertEquals(MED+1, (int)it.next());
		
		assertEquals(true, it.hasNext());
		assertEquals(HI, (int)it.next());
		
		assertEquals(false, it.hasNext());
		assertEquals(false, it.hasNext());
	}

}
