package de.brightbyte.axis;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class DefaultAxisManager implements AxisManager {
	
	
	protected Map<String, Class> axisTypes = new HashMap<String, Class>();
	protected Map<String, AxisProvider<?, ?>> axisProviders = new HashMap<String, AxisProvider<?, ?>>();
	
	public static String getAxisName(String axis) {
		int idx = axis.indexOf(';');
		if (idx>0) {
			axis = axis.substring(0, idx);
		}
		
		return axis.trim().toLowerCase();
	}

	public static Map<String, Object> getAxisOptions(String axis) {
		int idx = axis.indexOf(';');
		if (idx<0) return Collections.emptyMap();
		
		String s = axis.substring(idx+1).trim();
		if (s.length()==0) return Collections.emptyMap();

		String[] ss = s.split(" *; *");
		Map<String, Object> options = new HashMap<String, Object>(ss.length);
		
		for(String o : ss) {
			Object v;
			
			o = o.trim();
			if (o.length()==0) continue;
			
			idx = o.indexOf('=');
			
			if (idx<0) v = Boolean.TRUE;
			else {
				v = parseOptionValue(o.substring(idx+1).trim());
				o = o.substring(0, idx).trim();
			}
			
			options.put(o, v);
		}
		
		return options;
	}

	protected static Object parseOptionValue(String s) {
		if (s==null) return null;
		s = s.trim();
		
		try {
			return Integer.parseInt(s);
		}
		catch (NumberFormatException ex){ /* ignore */ }
		
		try {
			return Float.parseFloat(s);
		}
		catch (NumberFormatException ex){ /* ignore */ }
		
		if (s.equalsIgnoreCase("true")) return Boolean.TRUE;
		if (s.equalsIgnoreCase("false")) return Boolean.FALSE;
		
		return s;
	}

	public <T, O> T getAxis(String axis, O obj)
			throws UnsupportedAxisException {
		
		//if the object is a wrapper, look inside first (recursively)
		if (obj instanceof AxisWrapper<?>) {
			try {
				Object wobj = ((AxisWrapper<?>)obj).getWrappedObject();
				return this.<T, Object>getAxis(axis, wobj);
			}
			catch (UnsupportedAxisException ex) { /* ignore */ }
		}		
		
		AxisProvider<T, O> provider = getAxisProvider(axis, obj);
		Map<String, Object> options = getAxisOptions(axis);
		
		return provider.getAxis(obj, options);
	}

	public Class<?> getAxisType(String axis) throws UnsupportedAxisException {
		Class<?> t = findAxisType(axis);
		if (t==null) throw new UnsupportedAxisException(axis);
		return t;
	}

	protected Class<?> findAxisType(String axis) {
		axis = getAxisName(axis);
		return axisTypes.get(axis);
	}

	public <T, O> AxisProvider<T, O> getAxisProvider(String axis, O obj) throws UnsupportedAxisException {
		AxisProvider<T, O> provider = findAxisProvider(axis, obj);
		if (provider==null) throw new UnsupportedAxisException(axis);
		
		return provider;
	}

	@SuppressWarnings("unchecked")
	protected <T, O> AxisProvider<T, O> findAxisProvider(String axis, O obj) {
		axis = getAxisName(axis);
		
		Class<?> type = findAxisType(axis);
		if (type==null) return null;
		
		AxisProvider<?, ?> provider = axisProviders.get(axis);
		return (AxisProvider<T, O>)provider; //NOTE: unsafe case. Should be ok due to checks during registration
	}

	public boolean hasAxis(String axis, Object obj) {
		return findAxisProvider(axis, obj) != null;
	}

	public void declareAxis(String axis, Class type) {
		Class t = axisTypes.get(axis);
		if (t!=null) {
			if (t!=type) throw new IllegalArgumentException("axis "+axis+" already registered with type "+t+", conflict with type "+type);
		}
		else {
			axisTypes.put(axis, type);
		}
	}

	public <T, O> void registerProvider(AxisProvider<T, O> provider) throws UnsupportedAxisException {
		String axis = provider.getAxisName(); 
		Class<T> type = provider.getAxisType();
		
		Class<?> t = getAxisType(axis); //TODO: automatically register axis type??
		if (!t.isAssignableFrom(type)) throw new IllegalArgumentException("axis "+axis+" is defined to have type "+t+", but provider gives "+type);
		
		axisProviders.put(axis, provider);
	}

}
