package de.brightbyte.application;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.brightbyte.io.IOUtil;

public class VersionInfo {
	
	protected static final DateFormat dateTimePattern = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss ZZ");

	protected static String getFileInfo(URL u, boolean terse) throws IOException {
		URLConnection conn = u.openConnection();
		
		long time = conn.getLastModified();
		long size = conn.getContentLength();

		StringBuilder buff = new StringBuilder();
		
		String stamp = null;
		
		if (time>=0) {
			buff.append("T");
			buff.append(time);
			
			if (size>=0) {
				buff.append("Z");
				buff.append(size);
			}

			stamp = buff.toString();
			buff.setLength(0);
		}
		
		if (terse) {
			return stamp;
		}
		
		buff.append("Time: ");
		
		if (time<0) buff.append("<unknown>");
		else {
			buff.append(time);
			buff.append(" (");
			buff.append(dateTimePattern.format(new Date(time))); //TODO: pattern, TZ
			buff.append(")");
		}
		
		buff.append("\n");
		
		buff.append("Size: ");

		if (size<0) buff.append("<unknown>");
		else {
			buff.append(size);
			buff.append(" (");
			buff.append(size / 1024); //TODO: pattern, TZ
			buff.append("KB)");
		}
		
		buff.append("\n");
		
		buff.append("Stamp: ");

		if (stamp==null) buff.append("<unknown>");
		else buff.append(stamp);
		
		buff.append("\n");
		
		return buff.toString();
	}
	
	protected static void appendFileInfo(URL u, String name, boolean slurp, StringBuilder buff) {
		buff.append(name);
		buff.append(": ");
		
		if (u == null) {
			buff.append("<not found>");
		}
		else {
			try {
				buff.append(u.toExternalForm());
				buff.append(":\n\t");
				
				String v = slurp ? IOUtil.slurp(u) : getFileInfo(u, false);
				v = v.trim().replaceAll("(\r\n|\r|\n)", "\n\t");
				
				buff.append(v);
			}
			catch (IOException ex) {
				buff.append("<error: ");
				buff.append(ex.toString());
				buff.append(">");
			}
		}
		
	}
	
	protected static final String[] manifestAttributes = new String[] {
		"Bundle-Name", 
		"Bundle-Version",
		"Implementation-Version",
		"SHA1-Digest",
	};
	
	protected static void appendManifestInfo(String name, Attributes attributes, StringBuilder buff) {
		buff.append(name);
		buff.append(": ");
		
		if (attributes == null) {
			buff.append("<not found>");
		}
		else {
			for (int i = 0; i < manifestAttributes.length; i++) {
				buff.append("\n\t");

				String n = manifestAttributes[i];
				String v = attributes.getValue(n);
				
				buff.append(n);
				buff.append(": ");
				
				if (v==null || v.trim().length()==0) v = "<not found>";
				buff.append(v);
			}
		}
	}
	
	public static String getVersionStamp(Class cls) {
		//try version file
		URL versionFile = getVersionFile(cls);
		if (versionFile!=null) {
			try {
				String v = IOUtil.slurp(versionFile);
				return v.trim();
			} catch (IOException e) {
				//ignore
			}
		}
		
		//try manifest
		Manifest mf = getManifest(cls);
		if (mf!=null) {
			String v = mf.getMainAttributes().getValue("Bundle-Version");
			if (v!=null && v.trim().length()>0) return v;
		}
		
		//try jar file specs
		URL jarFile = getJarFile(cls);
		if (jarFile!=null) {
			try {
				String v = getFileInfo(jarFile, true);
				if (v!=null) return "J(" + v.trim() +")";
			} catch (IOException e) {
				//ignore
			}
		}
		
		//try class file specs
		URL classFile = getClassFile(cls);
		if (classFile!=null) {
			try {
				String v = getFileInfo(classFile, true);
				if (v!=null) return "C(" + v.trim() + ")";
			} catch (IOException e) {
				//ignore
			}
		}
		
		//this is very strange and shouldn't happen.
		return "<unknown>";
	}
	
	public static String getVersionInfo(Class cls) {
		URL versionFile = getVersionFile(cls);
		URL classFile = getClassFile(cls);
		URL jarFile = getJarFile(cls);
		Manifest mani = getManifest(cls);
		
		StringBuilder buff = new StringBuilder();

		buff.append("VERSION INFO for "+cls.getName());

		buff.append("\n\n");
		buff.append("VERSION STAMP: ");
		buff.append( getVersionStamp(cls) );

		buff.append("\n\n");
		appendFileInfo(versionFile, "VERSION FILE", true, buff);

		buff.append("\n\n");
		
		if (mani==null) buff.append("MANIFEST INFO: <not found>");
		else {
			appendManifestInfo("MANIFEST INFO", mani.getMainAttributes(), buff);

			//buff.append("\n\n");
			//String n = cls.getCanonicalName();
			//n = n.replaceAll("\\.", "/");
			//appendManifestInfo("CLASS MANIFEST INFO", mani.getAttributes( n + ".class" ), buff);
		}

		buff.append("\n\n");
		appendFileInfo(jarFile, "JAR FILE", false, buff);
		
		buff.append("\n\n");
		appendFileInfo(classFile, "CLASS FILE", false, buff);
		
		buff.append("\n\n");
		
		return buff.toString();
	}

	public static Manifest getManifest(Class cls) {
		try {
			URL u = getManifestFile(cls);
			if (u==null) return null;
			
			InputStream in = u.openStream();
			Manifest m = new Manifest(in);
			in.close();
			
			return m;
		} catch (IOException e) {
			//throw new RuntimeException("failed to load manifest", e);
			return null; //no manifest.
		}
	}

	public static URL getManifestFile(Class cls) {
		URL jar = getJarFile(cls);
		if (jar == null) return null;
		
		try {
			String path = "jar:" + jar.toExternalForm() + "!/META-INF/MANIFEST.MF";
			URL u = new URL(path);

			return u; //TODO: check if exists
		} catch (IOException e) {
			//throw new RuntimeException("failed to load manifest", e);
			return null; //no manifest.
		}
	}

	public static URL getClassFile(Class cls) {
		if (cls.isAnonymousClass()) throw new IllegalArgumentException(cls.getName()+" is anonymous");
		if (cls.isPrimitive()) throw new IllegalArgumentException(cls.getName()+" is primitive");
		if (cls.isArray()) throw new IllegalArgumentException(cls.getName()+" is an array class");
		
		String n = cls.getSimpleName();
		if (n==null) throw new IllegalArgumentException("can't get simple name for "+cls.getName());
		
		URL rc = cls.getResource(n+".class");
		if (rc==null) throw new IllegalArgumentException("can't find class file for "+cls.getName());

		return rc;
	}
	
	public static URL getJarFile(Class cls) {
		URL rc = getClassFile(cls);
		if (!rc.getProtocol().equals("jar")) return null; //not loaded from a jar file
		
		String jarpath = rc.toExternalForm();
		
		Matcher m = Pattern.compile("^jar:(.*?)!/.*").matcher(jarpath);
		if (!m.matches()) throw new RuntimeException("bad jar path: "+jarpath);
		
		String jurl = m.group(1);
		
		try {
			return new URL(jurl);
		} catch (MalformedURLException e) {
			throw new RuntimeException("bad jar file URL: "+jurl, e);
		}
	}

	public static URL getVersionFile(Class cls) {
		String n = cls.getCanonicalName();
		n = n.replaceAll("\\.", "/");
		return getVersionFile(cls.getClassLoader(), n);
	}

	public static URL getVersionFile(ClassLoader cl, String path) {
		URL u = null;
		
		if (u==null) u = cl.getResource(path + ".version");
		if (u==null) u = cl.getResource(path + "/version");
		
		if (u==null) {
			String p = path.replaceAll("[/\\.\\$][\\w\\d]+$", "");
			if (p.length() < path.length()) u = getVersionFile(cl, p);
		}
		
		return u;
	}
	
	public static void main(String[] args) throws ClassNotFoundException {
		String c = args.length > 0 ? args[0] : null;
		
		Class cls = c == null ? VersionInfo.class : Class.forName(c);
		
		String info = getVersionInfo(cls);
		
		System.out.println(info);
	}

}
