package de.brightbyte.application;

import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.brightbyte.io.Output;
import de.brightbyte.io.PrintStreamOutput;

public class Arguments {
	private List<String> parameters = new ArrayList<String>();
	private Map<String, Object> options = new HashMap<String, Object>();
	
	private Map<String, String> help = new HashMap<String, String>();
	private Map<String, String> aliases = new HashMap<String, String>();
	private Map<String, Constructor> constructors = new HashMap<String, Constructor>();
	private Set<String> greedy = new HashSet<String>();

	public Arguments() {
		super();
	}
	
	public void alias(String alias, String realname) {
		this.aliases.put(alias, realname);
	}
	
	public void declare(String name, String abbrev, boolean greedy, Class type, String help) {
		if (abbrev!=null) this.aliases.put(abbrev, name);
		if (greedy) this.greedy.add(name);
		
		if (type!=null) {
			try {
				Constructor c = type.getConstructor(new Class[] { String.class });
				constructors.put(name, c);
			} catch (NoSuchMethodException e) {
				throw new IllegalArgumentException("no suitable constructor found in "+type, e);
			}
		}
		
		if (help==null) help = "";
		declareHelp(name, help);
	}
	
	public void declareHelp(String name, String help) {
		if (help==null) this.help.remove(name);
		else this.help.put(name, help);
	}
	
	public void parse(String[] argv) {
		//TODO: optionally enforce know
		
		String currentName = null;
		for (int i = 0; i < argv.length; i++) {
			String a = argv[i];
			
			if (currentName!=null) {
				setOption(currentName, construct(currentName, a));
				currentName = null;
			}
			else if (a.length()>2 && a.charAt(0)=='-') {
				String s;
				if (a.length()>3 && a.charAt(1)=='-') {
					s = a.substring(2);
				}
				else {
					s = a.substring(1);
				}
				
				int idx = s.indexOf('=');
				if (idx>0) {
					String n = s.substring(0, idx);
					String v = s.substring(idx+1);
					
					setOption(n, construct(n, v));
				}
				else {
					s = unalias(s);
					if (isGreedy(s)) {
						currentName = s;
					}
					else {
						setOption(s, Boolean.TRUE);
					}
				}
			}
			else {
				parameters.add(a);
			}
		}
	}
	
	protected boolean isGreedy(String name) {
		name = unalias(name);
		return greedy.contains(name);
	}

	protected Object construct(String name, String v) {
		Constructor c = constructors.get(name);
		if (c==null) return v;
		
		try {
			return c.newInstance(new Object[] { v });
		} catch (IllegalArgumentException e) {
			throw e;
		} catch (InstantiationException e) {
			throw new RuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new IllegalArgumentException("failed to construct "+c.getDeclaringClass().getName()+" from \""+v+"\"", e.getTargetException());
		}
	}

	protected String unalias(String s) {
		String a = aliases.get(s);
		return a == null ? s : a;
	}

	public Map<String, Object> getOptions() {
		return options;
	}
	
	@SuppressWarnings("unchecked")
	public <T>T getOption(String name, T def) {
		name = unalias(name);
		T v = (T)options.get(name);
		if (v==null) v = def;
		return v;
	}
	
	public int getIntOption(String name, int def) {
		Object v = getOption(name, null);
		if (v==null) v = def;
		else if (v instanceof String) {
			v = construct(name, (String)v);
			if (v instanceof String) v = new Integer((String)v);
		}
		else if (!(v instanceof Number)) throw new IllegalArgumentException("option "+name+" has type "+v.getClass().getName()+", cannot convert to int.");
		
		return ((Number)v).intValue();
	}

	public String getStringOption(String name, String def) {
		Object v = getOption(name, null);
		if (v==null) v = def;
		else if (!(v instanceof Number)) v = v.toString();
		
		return (String)v;
	}

	public List<String> getParameters() {
		return parameters;
	}
	
	public String getParameter(int i) {
		return parameters.get(i);
	}
	
	public int getParameterCount() {
		return parameters.size();
	}
	
	public Iterable<String> parameters() {
		return parameters;
	}

	public boolean isSet(String name) {
		name = unalias(name);
		return options.containsKey(name);
	}
	
	@Override
	public String toString() {
		StringBuilder s = new StringBuilder();
		
		for (Map.Entry<String, Object> e : options.entrySet()) {
			Object v= e.getValue();
			
			s.append("--");
			s.append(e.getKey());
			s.append('=');
			if (v instanceof String) {
				s.append('"');
				s.append(v);
				s.append('"');
			}
			else if (v instanceof Boolean) ; //ignore
			else {
				s.append(v);
			}
			s.append(' ');
		}
		
		for (String p : parameters) {
			s.append(p);
			s.append(' ');
		}
		
		return s.toString();
	}
	
	public void printHelp(PrintStream out) {
		printHelp(new PrintStreamOutput(out));
	}
	
	public void printHelp(Output out) {
		List<String> opt = new ArrayList<String>(help.keySet());
		Collections.sort(opt);
		
		for (String o : opt) {
			String h = help.get(o);
			
			String n = o;
			if (n.matches("^[\\w\\d].*")) n = "--" + n;
			for (Map.Entry<String, String> e: aliases.entrySet()) {
				if (e.getValue().equals(o)) {
					n += " -"+e.getKey();
				}
			}
			
			h = wrap(h, "\t\t", 64);
			
			out.println("  "+n+":\t"+h);
		}
		
	}
	
	protected static String wrap(String t, String indent, int width) {
		String[] ss = t.split("\\s*(\r\n|[\r\n])\\s*");
		StringBuilder buffer = new StringBuilder();
		String newline = System.getProperty("line.separator");
		boolean first = true;
		
		for (String s: ss) {
			s = s.replaceAll("\\s+", " ");
			
			int i = 0;
			while (i < s.length()) {
				int j = i + width;
				
				if (j>=s.length()) j = s.length();
				else {
					while (j > i && s.charAt(j)!=' ') j--;
					if (j==i) j = i+width;
				}
		
				if (first) first = false;
				else {
					buffer.append(newline);
					buffer.append(indent);
				}
				
				buffer.append(s.substring(i, j).trim());
				
				i = j;
			}
		}
		
		return buffer.toString();
	}

	public void setOption(String k, Object v) {
		k = unalias(k);
		options.put(k, v);
	}
}
