package de.brightbyte.application;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import de.brightbyte.io.ConsoleIO;
import de.brightbyte.io.Output;
import de.brightbyte.util.PersistenceException;

public class AgendaTest extends TestCase {
	
	protected Output out = ConsoleIO.output;
	
	protected static class TestAgendaLog extends Agenda.TransientPersistor {
		//...
	}
	
	protected Agenda newAgenda(Agenda.Persistor log) throws PersistenceException {
		Agenda agenda = new Agenda( log );
		agenda.setLogger(out, out);
		return agenda;
	}

	public void testBeginTask() throws PersistenceException {
		Agenda.Persistor log;
		Agenda agenda;

		log = new TestAgendaLog();
		agenda = newAgenda( log );
		assertEquals("fresh 0", true, agenda.beginTask("APP", "app"));
		assertEquals("fresh 1", true, agenda.beginTask("TEST", "one")); agenda.endTask("TEST", "one");
		assertEquals("fresh 2", true, agenda.beginTask("TEST", "two"));
		
		//----
		agenda = newAgenda( log );
		assertEquals("last unfinished 0", true, agenda.beginTask("APP", "app"));
		assertEquals("last unfinished 1", false, agenda.beginTask("TEST", "one")); 
		assertEquals("last unfinished 2", true, agenda.beginTask("TEST", "two"));
		
		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two");
		
		//----
		agenda = newAgenda( log );
		assertEquals("last incomplete 0", true, agenda.beginTask("APP", "app"));
		assertEquals("last incomplete 1", false, agenda.beginTask("TEST", "one")); 
		assertEquals("last incomplete 2", true, agenda.beginTask("TEST", "two"));
		
		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two");
		agenda.endTask("TEST", "two", "ok");
		
		//----
		agenda = newAgenda( log );
		assertEquals("last complete 0", true, agenda.beginTask("APP", "app"));
		assertEquals("last complete 1", false, agenda.beginTask("TEST", "one")); 
		assertEquals("last complete 2", false, agenda.beginTask("TEST", "two")); 
		assertEquals("last complete 3", true, agenda.beginTask("TEST", "three"));
		
		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two"); agenda.endTask("TEST", "two");
		agenda.endTask("APP", "app", "ok");
		
		//----
		agenda = newAgenda( log );
		assertEquals("last finished 1.0", true, agenda.beginTask("APP", "app"));
		assertEquals("last finished 1.1", true, agenda.beginTask("TEST", "one")); agenda.endTask("TEST", "one");
		assertEquals("last finished 1.2", true, agenda.beginTask("TEST", "two")); 
		
		//----
		agenda = newAgenda( log );
		assertEquals("last finished 2.0", true, agenda.beginTask("APP", "app"));
		assertEquals("last finished 2.1", false, agenda.beginTask("TEST", "one")); 
		assertEquals("last finished 2.2", true, agenda.beginTask("TEST", "two")); agenda.endTask("TEST", "two");
		
		//----
		agenda = newAgenda( log );
		assertEquals("last finished 3.0", true, agenda.beginTask("APP", "app"));
		assertEquals("last finished 3.1", false, agenda.beginTask("TEST", "one")); 
		assertEquals("last finished 3.2", false, agenda.beginTask("TEST", "two")); 
		assertEquals("last finished 3.3", true, agenda.beginTask("TEST", "three")); agenda.endTask("TEST", "three");
	}

	public void testGetCurrentRecord() throws PersistenceException {
		Agenda.Persistor log;
		Agenda agenda;

		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two");
		
		//---
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		assertEquals(Agenda.State.COMPLETE, agenda.getCurrentRecord().state);

		agenda.beginTask("TEST", "two");
		assertEquals(Agenda.State.STARTED, agenda.getCurrentRecord().state);

		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two");
		
		//---
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		assertEquals(Agenda.State.COMPLETE, agenda.getCurrentRecord().state);
		agenda.beginTask("TEST", "two"); agenda.endTask("TEST", "two");
		assertEquals(Agenda.State.STARTED, agenda.getCurrentRecord().state);
		agenda.beginTask("TEST", "three");
		
		//---
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		assertEquals(Agenda.State.SKIPPED, agenda.getCurrentRecord().state);
		agenda.beginTask("TEST", "two");
		assertEquals(Agenda.State.COMPLETE, agenda.getCurrentRecord().state);
		agenda.beginTask("TEST", "three");  agenda.endTask("TEST", "three");
		assertEquals(Agenda.State.STARTED, agenda.getCurrentRecord().state); //XXX: qwirky :( getCurrentRecordy returns a dirt record from the last run() 

		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		agenda.endTask("TEST", "one", "ok");
		
		//---
		agenda = newAgenda( log );

		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		assertEquals(Agenda.State.COMPLETE, agenda.getCurrentRecord().state);

		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		agenda.endTask("TEST", "one", "ok");
		
		//---
		agenda = newAgenda( log );

		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); 
		assertEquals(Agenda.State.COMPLETE, agenda.getCurrentRecord().state);
	}

	public void testGetParameters() throws PersistenceException {
		Agenda.Persistor log;
		Agenda agenda;
		
		Map<String, Object> q = new HashMap<String, Object>();
		q.put("foo", 1);
		q.put("bar", "test this");
		
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two", q);
		
		//---
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");

		agenda.beginTask("TEST", "one"); 
		assertEquals(Collections.EMPTY_MAP, agenda.getCurrentRecord().parameters);

		agenda.beginTask("TEST", "two", q); agenda.endTask("TEST", "two");
		assertEquals(q, agenda.getCurrentRecord().parameters);

		agenda.beginTask("TEST", "xxx");
		assertEquals(Collections.EMPTY_MAP, agenda.getCurrentRecord().parameters);
	}

	public void testCanContinue() throws PersistenceException {
		Agenda.Persistor log;
		Agenda agenda;

		log = new TestAgendaLog();
		agenda = newAgenda( log );

		assertEquals("fresh:continue", false, agenda.canContinue("app"));
		assertEquals("fresh:finished", false, agenda.wasFinished("app"));
		
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.beginTask("TEST", "two");
		
		//---
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");

		assertEquals("unfinished:continue", true, agenda.canContinue("app"));
		assertEquals("unfinished:finished", false, agenda.wasFinished("app"));
		
		//=====================================================================
		log = new TestAgendaLog();
		agenda = newAgenda( log );
		agenda.beginTask("APP", "app");
		agenda.beginTask("TEST", "one"); agenda.endTask("TEST", "one");
		agenda.endTask("APP", "app", "ok");
		
		//---
		agenda = newAgenda( log );
		assertEquals("finished:continue", false, agenda.canContinue("app"));
		assertEquals("finished:finished", true, agenda.wasFinished("app"));
		
		assertEquals("other:continue", false, agenda.canContinue("foo"));
		assertEquals("other:finished", false, agenda.wasFinished("foo"));
		
	}

	//FIXME: check nesting more closely
	//FIXME: check isTaskDirty!
}
