package de.brightbyte.application;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import de.brightbyte.io.Output;
import de.brightbyte.text.DurationFormat;
import de.brightbyte.util.PersistenceException;
import de.brightbyte.util.StructuredDataCodec;

/**
 * <p>An agenda keeps track of which tasks have been started and completed, in order to
 * allow resuming execution after a crash.</p> 
 * <p>A process controlled by an agenda is devided into structure of nested tasks.
 * Tasks are purely logical steps of execution, they are not modelled directly;
 * however, they often correspond to method invocations.
 * If a process fails, it can be resumed where it left of: the Agenda remembers 
 * the last task that was being executed, and its parameters.
 * Tasks can be nested, and a main entry point (aka application or program) should
 * have a root task. The root task is the anchor point for determining where and how
 * execution should and can be resumed.
 * </p>
 */
public class Agenda {
	
	/**
	 * States a task might be in 
	 */
	public static enum State {
		/** the task was started, and was not completed, nor was any progress reported for it **/
		STARTED,

		/** the task was completed **/
		COMPLETE,

		/** the task was skipped **/
		SKIPPED,

		/** the task has failed **/
		FAILED;

		/** returns a State instance given an ordinal number **/
		public static State valueOf(int ordinal) {
			State[] states = State.values();
			return states[ordinal];
		}
	}
	
	/**
	 * An Record represents a task and it's state. It is the entitiy that is  
	 * stored by the Agenda.Log.
	 */
	public static class Record {
		public final int level;
		public final int start;
		public final int end;
		public final long timestamp;
		public final long duration;
		public final String context;
		public final String task;
		public final State state;
		public final Map<String, Object> parameters;
		public final boolean complex;
		public final String result;
		
		public Record(int level, int start, int end, 
						long timestamp, long duration, 
						String context, String task, State state, boolean complex,
						Map<String, Object> parameters,
						String result) {
			
			if (parameters==null) parameters = Collections.emptyMap();

			this.level = level;
			this.start = start;
			this.end = end <= 0 ? start : end;
			this.timestamp = timestamp;
			this.duration = duration;
			this.context = context;
			this.task = task;
			this.state = state;
			this.complex = complex;
			this.parameters = parameters;
			this.result = result;
		}

		@Override
		public String toString() {
			return String.format("#%d#%d-%d: %s %s (%s): %s", level, start, end, task, state, parameters, result);
		}
		
		public boolean isDone() {
			return state == State.COMPLETE || state == State.SKIPPED;
		}
		
		public boolean isTerminated() {
			return state == State.COMPLETE || state == State.SKIPPED || state == State.FAILED;
		}
		
		public boolean isDirty() {
			return state == State.STARTED || state == State.FAILED;
		}
	}
	
	/**
	 * A Agenda.Monitor may be used to track activity on an agenda from the outside; 
	 * It can be registered with an Agenda using the Agenda.setMonitor() method.
	 */
	public static interface Monitor {
		/**
		 * Notfies the monitor of the start of a task
		 * 
		 * @param rec the start record of the agenda entry, as returned by logStart
		 */
		public void beginTask(Record rec);
		
		/**
		 * Notfies the monitor of the end of a task
		 * 
		 * @param rec the start record of the agenda entry, as returned by logStart
		 * @param end the last id of this agenda entry, equal to start, or belonging to a sub-task's entry
		 * @param duration milliseconds elapsed since the task was started
		 * @param state the terminal state (COMPLETE, SKIPPED or FAILED)
		 * @param result a message describing the outcome
		 */
		public void endTask(Record rec, int end, long duration, State state, String result);

		/**
		 * Notfies the monitor that a task was skiped
		 * 
		 * @param rec the start record of the agenda entry, as returned by logStart
		 */
		public void skipTask(Record rec);
		
	}
	
	/**
	 * An Agenda.Persistor implementation is responsible for storing and retrieving information
	 * about the execution status of tasks. Generally, it would use some persistent storage
	 * mechanism, such as a file or a database.
	 */
	public static interface Persistor {
		
		/**
		 * Logs the start of a task
		 * 
		 * @param level the nesting level of the new task
		 * @param context the name of the context (method) the task was called from (may be null)
		 * @param task the name of the task - should be unique within a program
		 * @param parameters the parameters for this task, or null
		 * @param complex truw if the task is complex and may have sub-tasks 
		 * @throws PersistenceException if the state can not be logged
		 */
		public Record logStart(int level, String context, String task, Map<String, Object> parameters, boolean complex) throws PersistenceException;
		
		/**
		 * Logs completion of a task
		 * 
		 * @param start the start id of the agenda entry, as returned by logStart
		 * @param end the last id of this agenda entry, equal to start, or belonging to a sub-task's entry
		 * @param duration milliseconds elapsed since the task was started
		 * @param state the terminal state (COMPLETE, SKIPPED or FAILED)
		 * @param result a message describing the outcome
		 * @throws PersistenceException if the state can not be logged
		 */
		public void logTerminated(int start, int end, long duration, State state, String result) throws PersistenceException;
		
		/**
		 * Returns the root record stored  by the previous run of the agenda.
		 * The state of this record indicates if the previous run was completed. 
		 */
		public Record getLastRootRecord() throws PersistenceException;

		/**
		 * Returns the list of records generated by the last run.
		 * May be used to skip to the right point from which to continue execution. 
		 */
		public List<Record> getRecordLog() throws PersistenceException;

		/**
		 * Perpares the persistor for operation. Could be implemented to set up
		 * any required database tables, etc. 
		 */
		public void prepare() throws PersistenceException;
	}
	
	public static class TransientPersistor implements Persistor {

		protected List<Record> records = new ArrayList<Record>();
		
		protected int getRootIndex() {
			int i = records.size()-1;
			while (i>0) {
				Record r = records.get(i);
				if (r.level==0) break;
				i--;
			}
			
			return i;
		}
		
		public List<Record> getRecordLog() throws PersistenceException {
			if (records.isEmpty()) return null;
			List<Record> log = new ArrayList<Record>();

			int i = getRootIndex();
			//if (records.get(i).state != State.STARTED) return null; //complete 
			
			int skip = -1;
			while (i<records.size()) {
				Record rec = records.get(i);
				
				if (skip<i) {
					if (!rec.isDirty()) {
						skip = i;
					}
					
					log.add(rec);
				}
				
				i++;
			}
			
			return log; 
		}
		
		public Record getLastRootRecord() {
			return records.isEmpty() ? null : records.get(getRootIndex());
		}

		public void logTerminated(int start, int end, long duration, State state, String result) {
			if (state == State.STARTED) throw new IllegalArgumentException("bad state for logTerminated: "+state);
			
			Record record = records.get(start);			
			
			if (start != record.start) throw new IllegalArgumentException("id mismatches current record");
			record = new Record(record.level, record.start, end, 
					record.timestamp, duration, 
					record.context, record.task, state, record.complex, record.parameters,
					result);
			
			records.set(start, record);
		}

		public Record logStart(int level, String context, String task, Map<String, Object> parameters, boolean complex) {
			int id = records.size();
			Record record = new Record(level, id, id, System.currentTimeMillis(), 0, context, task, Agenda.State.STARTED, complex, parameters, null);
			records.add(record);
			return record;
		}

		public void prepare() throws PersistenceException {
			//noop
		}
		
	}
	
	private boolean loaded = false;
	//private boolean force = false;
	
	private int position;
	private Monitor monitor;
	private Persistor persistor;
	private List<Record> log = null;		
	private List<Record> stack = new ArrayList<Record>();
	
	private int counter = 0;
	private int end = 0;
	private Record current = null;
	private Record last = null;
	private Record lastRoot = null;
	
	private Output tracer;
	private Output logger;

	
	/**
	 * Creates a new Agenda using the given log. If the Log
	 */
	public Agenda(Persistor log)  throws PersistenceException {
		this.persistor = log;
	}
	
	public void setMonitor(Monitor monitor) {
		if (this.monitor!=null && this.monitor!=monitor) throw new IllegalStateException("monitor already set");
		
		this.monitor = monitor;
	}
	
	public void setLogger(Output logger, Output tracer) {
		this.logger = logger;
		this.tracer = tracer;
	}
	
	protected void trace(String msg) {
		if (tracer!=null) {
			tracer.println("Agenda trace: "+msg);
		}
	}

	protected void log(String state, String context, String task, String msg) {
		log("Agenda task "+state+"#"+stack.size()+": "+context+"::"+task+": "+msg);
	}
	
	protected void log(String msg) {
		if (logger!=null) {
			logger.println(msg);
		}
		if (tracer!=null && logger!=tracer) {
			tracer.println(msg);
		}
	}

	/*
	protected boolean isPastLast() throws PersistenceException {
		if (!loaded) getLastRecord(); //load
		return pastLast;
	}
	*/
	
	protected void load() throws PersistenceException {
		if (loaded) return;
		
		persistor.prepare();
		lastRoot = persistor.getLastRootRecord();
		
		if (lastRoot!=null && !lastRoot.isDone()) {
			log = persistor.getRecordLog();
		} 
		
		loaded = true;
	}
	
	/*
	public Record getLastRecord() throws PersistenceException {
		if (loaded) return lastRec;
		loaded = true;
		
		this.lastRec = persistor.getLastRecord();
		
		if (lastRec!=null && lastRec.task.equals("*")) {
			lastRec = null; //previous run was marked finished, so there is nothing to continue.
			finished = true;
		}
		
		return lastRec;
	}
	*/
	
	/**
	 * Force this Agenda to behave as if the last state of the previous run was 
	 * characterized by the given stack of tasks. 
	 * May be useful for debugging or manual recovery. Use with care.
	 * 
	 * @param tasks the names of the tasks to consider as last state
	 * @param parameters the parameters for this task, or null
	 * @throws PersistenceException 
	 */
	/*public void forceLastTask(String[] tasks) throws PersistenceException {
		load();
		
		List<Record> lg = new ArrayList<Record>();
		if (log!=null) lg.addAll(log.subList(0, position));
		log = lg;
		
		loaded = true;
		finished = false;
		force = true;
		
		trace("forceLastTask("+StringUtils.join("/", tasks)+")");

		Record rec = log.isEmpty() || position == 0 ? null : log.get(position-1);
		int level = rec == null ? 0 : rec.level;
		for (String t: tasks) {
			rec = new Record(level, log.size(), log.size(), 0, 0, t, State.STARTED, null);
			log.add(rec);
			level++;
		}
	}*/
	
	/**
	 * Resets the Agenda, so it does not try to continue where the last execution left off,
	 * but executes each task. If shouldRun was called already on this Agenda instance, 
	 * an IllegalStateException is thrown by this method.
	 */
	public void reset() {
		if (counter>0) throw new IllegalStateException("agenda was already started, can not be reset now.");
		
		current = null;
		last = null;
		log = null;
		stack.clear();
		position = 0;
		loaded = true;
		trace("reset");
	}

	/**
	 * Determines if the given task should be executed during this execution
	 * of the Agenda, or if it should be skipped, because it was completed
	 * by a previous execution of the Agenda. This will first log the previous
	 * task as COMPLETE, and then the given new task as STARTED.
	 * 
	 * @param context the context this method is called from (usually the name of the calling method).
	 *                used only for scope-checking of non-complex tasks. If null, no checks are performed.
	 * @param task the name of the task - should be unique within a program
	 * @param parameters the parameters for this task, or null
	 * @param subtasks determines if this tasks may have subtasks and will
	 *        be reported as complete explicitly.
	 * 
	 * @return true if the task should be run, false if it should be skipped becuase 
	 * it was completed during the previous execution of this agenda. 
	 * @throws PersistenceException
	 */
	protected boolean beginTask(String context, String task, Map<String, Object> parameters, boolean complex)  throws PersistenceException {
		load();
		
		if (current!=null && !current.complex) {
			if (context!=null && current.context!=null) {
				if (!context.equals(current.context)) throw new IllegalStateException("last task was "+current.task+" in context "+current.context+", can not auto-close from context "+context);
			}
			
			endTask(context, current.task, null, false);
		}

		if (log==null) {
			int level = this.stack.size(); 
			if (counter>0 && level==0) {
				throw new IllegalStateException("can not have two root tasks in the same agenda.");
			}
		}
		
		counter ++;
		
		if (parameters==null) parameters = Collections.emptyMap();
		
		if (log!=null && position>=log.size()) log = null;
			
		if (log!=null) {
			Record rec = log.get(position);
			
			if (!rec.task.equals(task)) throw new IllegalArgumentException("bad task: "+task+" requested, but found "+rec.task+" in the log");
			if (!rec.context.equals(context)) throw new IllegalArgumentException("bad context: "+task+" requested, but found "+rec.context+" in the log");
			
			//NOTE: checking the params is nice, but sometimes, there's not enough info to re-create all params while skipping!
			//if (!rec.parameters.equals(parameters)) throw new IllegalArgumentException("bad parameters for task "+task+": "+parameters+" requested, but found "+rec.parameters+" in the log");
			
			//NOTE: parameters ending with "_" will be maintained from the past run. //FIXME: doc this!
			if (parameters!=null) {
				for (Map.Entry<String, Object> e: rec.parameters.entrySet()) {
					String k = e.getKey();
					if (k.endsWith("_")) {
						parameters.put(k, e.getValue());
					}
				}
			}
			
			position ++;
			if (position>=log.size()) log = null;
			
			last = rec;
			if (rec.isDone()) { //record matches, skip task
				
				rec = persistor.logStart(this.stack.size(), context, task, parameters, complex);
				persistor.logTerminated(rec.start, rec.end, System.currentTimeMillis() - rec.timestamp, State.SKIPPED, null);
				
				log("SKIPPED", context, task, "skipped");
				if (monitor!=null) monitor.skipTask(rec);
				
				return false;
			}
		}
		else {
			last = null;
		}

		if (current!=null && context!=null && current.context!=null) {
			if (context.equals(current.context)) throw new IllegalStateException("current complex task is "+current.task+" in context "+current.context+", can not add a subtask from the same context!");
		}

		this.current = persistor.logStart(this.stack.size(), context, task, parameters, complex);
		if (monitor!=null) monitor.beginTask(this.current);

		this.stack.add(this.current);
		
		end = this.current.start;

		log("BEGIN", context, task, "("+parameters+")");
		return true;
	}
	
	/** 
	 * shorthand for shouldRun(task, parameters), true);
	 */
	public boolean beginTask(String context, String task, Map<String, Object> parameters)  throws PersistenceException {
		return beginTask(context, task, parameters, true);
	}
	
	/** 
	 * shorthand for shouldRun(task, decodeParameters(parameters, true), true);
	 */
	public boolean beginTask(String context, String task, String parameters)  throws PersistenceException {
		return beginTask(context, task, decodeParameters(parameters, true), true);
	}
	
	/** 
	 * shorthand for shouldRun(task, (Map<String, Object>)null, true);
	 */
	public boolean beginTask(String context, String task)  throws PersistenceException {
		return beginTask(context, task, (Map<String, Object>)null, true);
	}
	/** 
	 * shorthand for shouldRun(task, parameters), false);
	 */
	@Deprecated
	public boolean beginPrimitiveTask(String context, String task, Map<String, Object> parameters)  throws PersistenceException {
		return beginTask(context, task, parameters, false);
	}
	
	
	/** 
	 * shorthand for shouldRun(task, decodeParameters(parameters, true), false);
	 */
	@Deprecated
	public boolean beginPrimitiveTask(String context, String task, String parameters)  throws PersistenceException {
		return beginTask(context, task, decodeParameters(parameters, true), false);
	}
	
	/** 
	 * shorthand for shouldRun(task, (Map<String, Object>)null, false);
	 */
	@Deprecated
	public boolean beginPrimitiveTask(String context, String task)  throws PersistenceException {
		return beginTask(context, task, (Map<String, Object>)null, false);
	}
	
	/**
	 * shorthand for endTask(context, task, "ok");
	 */
	public void endTask(String context, String task)  throws PersistenceException {
		endTask(context, task, "ok");
	}
	
	/**
	 * shorthand for endTask(context, task, result, false);
	 */
	public void endTask(String context, String task, String result)  throws PersistenceException {
		endTask(context, task, result, false);
	}

	/**
	 * shorthand for endTask(context, task, result, true);
	 */
	@Deprecated
	public void endTasks(String context, String task, String result)  throws PersistenceException {
		endTask(context, task, result, true);
	}
	
	/**
	 * Logs the current task as COMPLETED.
	 * 
	 * @param context the context this method is called from (usually the name of the calling method).
	 *                used only for scope-checking of non-complex tasks. If null, no checks are performed.
	 * @param task the name of the task - should be unique within a program
	 * @param result message describing the result of running the task, for logging/debugging. may be null
	 * @param lenient true if open sub-tasks should be closed automatically
	 * 
	 * @throws PersistenceException
	 */
	protected void endTask(String context, String task, String result, boolean lenient)  throws PersistenceException {
		 if (current==null) throw new IllegalArgumentException("no current task, can't end task "+task);
		 
		 int j = stack.size() -1;
		 Record r = stack.get(j);
		 
		 if (!task.equals(r.task)) {
			 if (!lenient) {
				 if (!r.complex) {
					 Record p = stack.get(j-1);
					 if (!task.equals(p.task)) throw new IllegalArgumentException("parent of current task is "+p.task+", can't end task "+task);

					 endTask(r.context, r.task, null, false); //recurse, but only one level max.
					 
					 j--;
					 r = stack.get(j);
				 }
				 else {
					 throw new IllegalArgumentException("current task is "+r.task+" in context "+r.context+", can't end task "+task);
				 }
			 }
			 else {
				 while (j>=0) {
					 r = stack.get(j);
					 if (r.task.equals(task)) break;
					 
					 endTask(r.context, r.task, null, false); //recurse, but only one level max.
					 j--;
				 }
				 
				 if (j<0) throw new IllegalArgumentException("can't log completion of "+task+" in context "+r.context+": not found on stack!");
			 }
		 }
		 
		 if (context!=null && r.context!=null && !context.equals(r.context)) {
			 throw new IllegalArgumentException("can't end task "+task+": given context is "+context+", expected "+r.context);
		 }
		 
		 long duration = System.currentTimeMillis() - r.timestamp;
		 log("__END", context, task, result+" ("+DurationFormat.instance.format(duration)+")"); //NOTE: use __ to align with START. makes logs easier to read.
		 
		 persistor.logTerminated(r.start, end, duration, State.COMPLETE, result);
		 if (monitor!=null) monitor.endTask(r, end, duration, State.COMPLETE, result);
		 
		 stack.remove(stack.size()-1);
		 current = stack.isEmpty() ? null : stack.get(stack.size()-1);
	}
	
	public int getCurrentLevel() throws PersistenceException {
		Record rec = getCurrentRecord();
		return rec == null ? null : rec.level + 1;
	}
	
	/**
	 * Returns the state of the current task was left with during the last execution of this Agenda.
	 * If the current task was the last task logged during the previous execution, this method will
	 * return the actual state that task was left in (i.e. STARTED or PROGRESS if it was the
	 * task that failed, or COMPLETED if it was the last task that finished before failure).
	 * If the current execution is not continuing a previous execution, or has passed the point 
	 * of tasks completed during the previous execution, this method returns null. Otherwise,
	 * that is, if this execution of the Agenda is continuing a previous execution, and has not yet 
	 * passed the point  of the last task run during the previous execution, then this method
	 * returns COMPLETED.
	 * @throws PersistenceException 
	 */
	/*public State getState() throws PersistenceException {
		Record last = getLastRecord();
		
		if (last!=null && last.task.equals(current)) {
			return last.state;
		}
		else if (pastLast) {
			return null;
		}
		else {
			return State.COMPLETE;
		}
	}*/
	
	/**
	 * Returns the record for the current task. this may taken from the log of the 
	 * previous run: if the previous call to beginTask() returned false, or if
	 * calling isTaskDirty() would be returning true, the record returned is taken from
	 * the log of the previous run. Otherwise, it's a fresh record for the
	 * current task, created by the previous call to beginTask(). If isTaskDirty() returns true,
	 * this method may be used to optain information about the state of the previous execution
	 * of the task, in order to reset the any dirty state.
	 * @throws PersistenceException 
	 */
	public Record getCurrentRecord() throws PersistenceException {
		load();

		if (last==null) {
			return current;
		}
		else {
			return last;		
		}
	}
	
	public boolean isTaskDirty() throws PersistenceException {
		if (last==null) return false;
		else return last.isDirty();
	}
	
	/**
	 * Returns the record for the root task of the previous run.
	 * Returns null if there was no previous run.
	 * The result is undefined after the first call to a shouldRun method
	 * during the current run of the agenda.
	 * @throws PersistenceException 
	 */
	public Record getLastRootRecord() throws PersistenceException {
		load();

		return lastRoot;
	}
	
	/**
	 * Returns the record for the last task started in previous run.
	 * Returns null if there was no previous run.
	 * The result is undefined after the first call to a shouldRun method
	 * during the current run of the agenda.
	 * @throws PersistenceException 
	 */
	public Record getLastRecord() throws PersistenceException {
		load();

		return (log==null || log.isEmpty()) ? null : log.get(log.size()-1);
	}
	
	/**
	 * Returns true if there was a previous execution of the given root task and it 
	 * was not finished. That is, it returns true if the next execution shoud
	 * continue the previous execution. 
	 * @param rootTask the name of the root task of the current agenda 
	 * @throws PersistenceException 
	 */
	public boolean canContinue(String rootTask) throws PersistenceException {
		Record root = getLastRootRecord();
		if (root==null) return false;
		else return root.isDirty() && root.task.equals(rootTask);
	}
	
	/**
	 * Returns true if there was a previous execution of this agenda, and it 
	 * was complete. That is, it returns true
	 * if the next execution shoud not try to continue the previous execution,
	 * but should either redo everything, or do nothign at all.
	 * @param rootTask the name of the root task of the current agenda 
	 * @throws PersistenceException 
	 */
	public boolean wasFinished(String rootTask) throws PersistenceException {
		load();
		Record rec = getLastRootRecord();
		if (rec==null || !rec.task.equals(rootTask)) return false;
		else return rec.isDone();
	}
	
	/**
	 * Logs a special task, indicating that this agenda was executed completely.
	 * Subsequent executions should start again with the first task. 
	 */
	/*public void logFinished() throws PersistenceException {
		if (current!=null) { //mark pervious task as finished
			logComplete(current);
		}

		trace("logFinished");
		persistor.logComplete("*", null);	
	}*/

	/**
	 * Converts a string-encoded parameter map into a Map. Inverse of encodeParameters(Map).
	 * Usefull for implementations of Agenda.Log. If the lenient flag is set, 
	 * some heuristics are used to fix not-quite-wellformed input. That us usfull for
	 * manuyll supplied map representations.
	 */
	@SuppressWarnings("unchecked")
	public static Map<String, Object> decodeParameters(String s, boolean lenient) {
		if (s==null || s.trim().length()==0) return Collections.emptyMap();
		
		if (lenient) {
			s = s.trim();
			if (s.charAt(0)!='{') s = "{"+s+"}";
		}
		
		Object obj = StructuredDataCodec.instance.decodeValue(s);
		if (!(obj instanceof Map)) throw new IllegalArgumentException("bad parameter string: "+s);
		return (Map<String, Object>)obj;
	}

	/**
	 * Converts a parameter Map into a String. Inverse of decodeParameters(String).
	 * Usefull for implementations of Agenda.Log. 
	 */
	public static String encodeParameters(Map<String, Object> m) {
		if (m==null || m.size()==0) return "";

		return StructuredDataCodec.instance.encodeValue(m);
	}

	/**
	 * Wraps the given string in a way suitable for use in a parameter string. 
	 * @return
	 */
	public static String quote(String s) {
		return StructuredDataCodec.instance.quote(s);
	}

}
