package de.brightbyte.abstraction;

import java.text.Format;
import java.text.ParseException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import de.brightbyte.data.NaturalComparator;

public class GenericPropertyInfo<T, V> implements PropertyInfo<T, V> {
	private Map<String, Object> features = new HashMap<String, Object>();
	
	public GenericPropertyInfo(String name, Class<V> type, PropertyAccessor<T, V> accessor) {
		features.put("name", name);
		features.put("type", type);
		features.put("accessor", accessor);
		
		if (Comparable.class.isAssignableFrom(type)) {
			setComparator((Comparator<V>)NaturalComparator.instance);
		}
	}

	public Object getFeature(String feature) {
		return getFeature(feature, null);
	}
	
	public void setFeature(String feature, Object v) {
		if (feature.equals("name")) throw new IllegalArgumentException("can't set the core feature 'name'");
		if (feature.equals("type")) throw new IllegalArgumentException("can't set the core feature 'type'");
		if (feature.equals("accessor")) throw new IllegalArgumentException("can't set the core feature 'accessor'");
		features.put(feature, v);
	}

	public Object getFeature(String feature, Object def) {
		Object v = features.get(feature);
		return v==null ? def : v;
	}

	public String getFeatureString(String feature, String def) {
		Object v = features.get(feature);
		return v==null ? def : v.toString();
	}

	public int getFeatureInt(String feature, int def) {
		Object v = features.get(feature);
		return v==null ? def : ((Number)v).intValue();
	}

	public Comparator<V> getComparator() {
		return (Comparator<V>)getFeature("comparator", null);
	}

	public void setComparator(Comparator<V> comparator) {
		setFeature("comparator", comparator);
	}

	public String getLabel() {
		return getFeatureString("label", getName());
	}
	
	public void setLabel(String label) {
		setFeature("label", label);
	}

	public String getName() {
		String name = getFeatureString("name", null); 
		if (name==null) throw new RuntimeException("missing core feature 'name'");
		return name;
	}

	public Class<V> getType() {
		Class<V> type = (Class<V>)getFeature("type", null); 
		if (type==null) throw new RuntimeException("missing core feature 'type'");
		return type;
	}

	public V getValue(T record) {
		return getAccessor().getValue(record);
	}
	
	public void setValue(T record, V value) {
		getAccessor().setValue(record, value);
	}

	public boolean isMutable() {
		return getAccessor().isMutable();
	}

	public boolean isValid(V value) {
		ValidationResult vr = new ValidationResult();
		return validate(value, vr);
	}

	public boolean isWellformed(String s) {
		try {
			tryParse(s);
			return true;
		} catch (ParseException e) {
			return false;
		}
	}

	public V parse(String s) {
		try {
			return tryParse(s);
		} catch (ParseException e) {
			return null;
		}
	}

	protected V tryParse(String s) throws ParseException {
		Format f = getFormat();
		if (f==null) {
			if (getType()==String.class) return (V)s;
			else throw new ParseException("no parser defined", -1);
		}
		else {
			return (V)f.parseObject(s);
		}
	}

	public String format(V value) {
		Format f = getFormat();
		if (f==null) return value.toString();
		else return f.format(value);
	}

	public boolean validate(V value, ValidationResult result) {
		Validator<V> validator = getValidator();
		if (validator!=null) validator.validate(value, result);
		
		return !result.hasProblems();
	}
	
	public Validator<V> getValidator() {
		return (Validator<V>)getFeature("validator", null);
	}
	
	public void setValidator(Validator<V> validator) {
		setFeature("validator", validator);
	}
	
	public Format getFormat() {
		return (Format)getFeature("format", null);
	}
	
	public void setFormat(Format format) {
		setFeature("formar", format);
	}
	
	public PropertyAccessor<T, V> getAccessor() {
		PropertyAccessor<T, V> acc = (PropertyAccessor<T, V>)getFeature("accessor", null);
		if (acc==null) throw new RuntimeException("missing core feature 'accessor'");
		return acc;
	}
	
}
