package de.brightbyte.abstraction;

import java.beans.BeanInfo;
import java.beans.EventSetDescriptor;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.MethodDescriptor;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import de.brightbyte.util.BeanUtils;

/**
 * Provides access to properties, methods and event sets for a class of objects, using
 * information from the classe's BeanInfo
 */
public class BeanTypeAbstractor<T> implements ExtendedAbstractor<T> {
	
	protected static Map<Class, BeanTypeAbstractor> abstractors = new HashMap<Class, BeanTypeAbstractor>();
	
	protected static class FilteredPropertyChangeListener implements PropertyChangeListener {
		protected PropertyChangeListener listener;
		protected String property;
		
		public FilteredPropertyChangeListener(PropertyChangeListener listener, String property) {
			if (listener==null) throw new NullPointerException();
			if (property==null) throw new NullPointerException();
			
			this.listener = listener;
			this.property = Introspector.decapitalize(property);
		}

		public void propertyChange(PropertyChangeEvent evt) {
			if (evt.getPropertyName().equals(property)) listener.propertyChange(evt);			
		}

		@Override
		public int hashCode() {
			final int PRIME = 31;
			int result = 1;
			result = PRIME * result + ((listener == null) ? 0 : listener.hashCode());
			result = PRIME * result + ((property == null) ? 0 : property.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final FilteredPropertyChangeListener other = (FilteredPropertyChangeListener) obj;
			if (listener == null) {
				if (other.listener != null)
					return false;
			} else if (!listener.equals(other.listener))
				return false;
			if (property == null) {
				if (other.property != null)
					return false;
			} else if (!property.equals(other.property))
				return false;
			return true;
		}

		
	}
	
	@SuppressWarnings("unchecked")
	public static <T>BeanTypeAbstractor<T> getAbstractor(Class<T> cls) throws IntrospectionException {
		BeanTypeAbstractor<T> abstractor = (BeanTypeAbstractor<T>)abstractors.get(cls);
		
		if (abstractor==null) {
			abstractor = new BeanTypeAbstractor<T>(cls);
			abstractors.put(cls, abstractor);
		}
		
		return abstractor;
	}
	
	public static void purgeAbstractors() {
		abstractors.clear();
	}

	protected Class<T> targetType;
	protected BeanInfo info;

	protected Map<String, PropertyDescriptor> properites;
	protected Map<String, MethodDescriptor> methods;
	protected Map<String, EventSetDescriptor> eventSets;

	@SuppressWarnings("unchecked")
	public BeanTypeAbstractor(Class<T> targetType) throws IntrospectionException {
		this(targetType, Introspector.getBeanInfo(targetType));
	}
	
	public BeanTypeAbstractor(Class<T> targetType, BeanInfo info) {
		if (targetType==null) throw new NullPointerException();
		if (info==null) throw new NullPointerException();
		
		this.targetType = targetType; 
		this.info = info;
		
		properites = new HashMap<String, PropertyDescriptor>();
		methods =    new HashMap<String, MethodDescriptor>();
		eventSets =  new HashMap<String, EventSetDescriptor>();

		BeanInfo[] additional = info.getAdditionalBeanInfo();
		
		if (additional!=null) {
			for (BeanInfo b: additional) {
				BeanUtils.fillFeatureMap(b.getPropertyDescriptors(), properites);
				BeanUtils.fillFeatureMap(b.getMethodDescriptors(),   methods);
				BeanUtils.fillFeatureMap(b.getEventSetDescriptors(), eventSets);		
			}
		}
		
		BeanUtils.fillFeatureMap(info.getPropertyDescriptors(), properites);
		BeanUtils.fillFeatureMap(info.getMethodDescriptors(),   methods);
		BeanUtils.fillFeatureMap(info.getEventSetDescriptors(), eventSets);		
	}

	public Class<T> getTargetType() {
		return targetType;
	}


	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((info == null) ? 0 : info.hashCode());
		result = PRIME * result + ((targetType == null) ? 0 : targetType.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final BeanTypeAbstractor other = (BeanTypeAbstractor) obj;
		if (info == null) {
			if (other.info != null)
				return false;
		} else if (!info.equals(other.info))
			return false;
		if (targetType == null) {
			if (other.targetType != null)
				return false;
		} else if (!targetType.equals(other.targetType))
			return false;
		return true;
	}

	public PropertyDescriptor getPropertyDescriptor(String name) {
		name = Introspector.decapitalize(name);
		return properites.get(name);
	}

	public MethodDescriptor getMethodDescriptor(String name) {
		name = Introspector.decapitalize(name);
		return methods.get(name);
	}

	public EventSetDescriptor getEventSetDescriptor(String name) {
		name = Introspector.decapitalize(name);
		return eventSets.get(name);
	}

	public void addListener(T object, String event, PropertyChangeListener li) throws IllegalArgumentException {
		EventSetDescriptor esd = getEventSetDescriptor(event);
		if (esd==null) throw new IllegalArgumentException("event set "+event+" not known in "+targetType);
		
		Method m = esd.getAddListenerMethod();
		if (m==null) throw new IllegalArgumentException("event set "+event+" in "+targetType+" has no method to add listeners");

		callSafe(object, m, new Object[] { li } );
	}

	public void removeListener(T object, String event, PropertyChangeListener li) throws IllegalArgumentException {
		EventSetDescriptor esd = getEventSetDescriptor(event);
		if (esd==null) throw new IllegalArgumentException("event set "+event+" not known in "+targetType);
		
		Method m = esd.getRemoveListenerMethod();
		if (m==null) throw new IllegalArgumentException("event set "+event+" in "+targetType+" has no method to remove listeners");

		callSafe(object, m, new Object[] { li } );
	}

	public void addPropertyChangeListener(T object, final String name, final PropertyChangeListener li) throws IllegalArgumentException {
		PropertyDescriptor pd = getPropertyDescriptor(name);
		if (pd==null) throw new IllegalArgumentException("property "+name+" not known in "+targetType);
		if (!pd.isBound()) throw new IllegalArgumentException("property "+name+" of "+targetType+" does not trigger events");
		
		try  {
			Method m = BeanUtils.getMethod(targetType, "addPropertyChangeListener", new Class[] { String.class, PropertyChangeListener.class });
			callSafe(object, m, new Object[] { name, li });
		}
		catch (NoSuchMethodException ex) {
			PropertyChangeListener fli = new FilteredPropertyChangeListener(li, name);
			addListener(object, "propertyChange", fli);
		}
	}

	public void removePropertyChangeListener(T object, final String name, final PropertyChangeListener li) throws IllegalArgumentException {
		PropertyDescriptor pd = getPropertyDescriptor(name);
		if (pd==null) throw new IllegalArgumentException("property "+name+" not known in "+targetType);
		if (!pd.isBound()) throw new IllegalArgumentException("property "+name+" of "+targetType+" does not trigger events");
		
		try  {
			Method m = BeanUtils.getMethod(targetType,"removePropertyChangeListener", new Class[] { String.class, PropertyChangeListener.class });
			callSafe(object, m, new Object[] { name, li });
		}
		catch (NoSuchMethodException ex) {
			//XXX: does this work?!
			PropertyChangeListener fli = new FilteredPropertyChangeListener(li, name);
			removeListener(object, "propertyChange", fli);
		}
	}

	protected Object callSafe(T object, Method m, Object[] args) throws IllegalArgumentException {
		if (!targetType.isInstance(object)) throw new IllegalArgumentException("tagert object type "+targetType+" is incompatible with "+targetType);
		
		try {
			return m.invoke(object, args);
		} catch (IllegalAccessException e) {
			throw new RuntimeException("failed to access method "+m+" from "+targetType, e);
		} catch (InvocationTargetException e) {
			Throwable exx = e.getCause();
			if (exx instanceof Error) throw (Error)exx;
			if (exx instanceof RuntimeException) throw (RuntimeException)exx;
			
			throw new RuntimeException("odd exception when calling method "+m+" from "+targetType);
		}
	}
	
	public Object call(T object, String name, Object[] args) throws Exception {
		if (!targetType.isInstance(object)) throw new IllegalArgumentException("tagert object type "+targetType+" is incompatible with "+targetType);
		
		//FIXME: overloading - find best match?!
		MethodDescriptor md = getMethodDescriptor(name);
		if (md==null) throw new IllegalArgumentException("method "+name+" not known in "+targetType);
		
		Method m = md.getMethod();

		try {
			return m.invoke(object, args);
		} catch (IllegalAccessException e) {
			throw new RuntimeException("failed to access method "+m+" from "+targetType, e);
		} catch (InvocationTargetException e) {
			Throwable exx = e.getCause();
			if (exx instanceof Error) throw (Error)exx;
			if (exx instanceof Exception) throw (Exception)exx;
			
			throw new Error("odd exception when calling method "+m+" from "+targetType);
		}
	}

	public Object getProperty(T object, String name) throws IllegalArgumentException {
		PropertyDescriptor pd = getPropertyDescriptor(name);
		if (pd==null) throw new IllegalArgumentException("property "+name+" not known in "+targetType);
		
		Method m = pd.getReadMethod();
		if (m==null) throw new IllegalArgumentException("property "+name+" in "+targetType+" is not readable");

		return callSafe(object, m, null);
	}

	public void setProperty(T object, String name, Object v) throws IllegalArgumentException {
		PropertyDescriptor pd = getPropertyDescriptor(name);
		if (pd==null) throw new IllegalArgumentException("property "+name+" not known in "+targetType);
		
		Method m = pd.getWriteMethod();
		if (m==null) throw new IllegalArgumentException("property "+name+" in "+targetType+" is not writable");

		callSafe(object, m, new Object[] {v});
	}

	public boolean isPropertyMutable(String name) {
		PropertyDescriptor pd = getPropertyDescriptor(name);
		if (pd==null) throw new IllegalArgumentException("property "+name+" not known in "+targetType);

		Method m = pd.getWriteMethod();
		return m != null;
	}

	public Class getPropertyType(String name) {
		PropertyDescriptor pd = getPropertyDescriptor(name);
		if (pd==null) throw new IllegalArgumentException("property "+name+" not known in "+targetType);
		
		return pd.getPropertyType();
	}

	public boolean hasProperty(T obj, String name) {
		return getPropertyDescriptor(name)!=null;
	}

}
