package de.brightbyte.abstraction;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import de.brightbyte.util.BeanUtils;

public class BeanPropertyAbstractor implements Abstractor<Object> {

	public static final BeanPropertyAbstractor instance = new BeanPropertyAbstractor(false);
	public static final BeanPropertyAbstractor lenientInstance = new BeanPropertyAbstractor(true);
	
	protected boolean lenient;
	
	public BeanPropertyAbstractor(boolean lenient) {
		this.lenient = lenient;
	}

	protected Object callSafe(Object object, Method m, Object[] args) throws IllegalArgumentException {
		Class targetType = object.getClass();
		if (!targetType.isInstance(object)) throw new IllegalArgumentException("tagert object type "+targetType+" is incompatible with "+targetType);
		
		try {
			return m.invoke(object, args);
		} catch (IllegalAccessException e) {
			throw new RuntimeException("failed to access method "+m+" from "+targetType, e);
		} catch (InvocationTargetException e) {
			Throwable exx = e.getCause();
			if (exx instanceof Error) throw (Error)exx;
			if (exx instanceof RuntimeException) throw (RuntimeException)exx;
			
			throw new RuntimeException("odd exception when calling method "+m.getName()+" from "+targetType, e);
		}
	}
	
	public Object getProperty(Object obj, String name) throws IllegalArgumentException {
		Method m = getReadMethod(obj.getClass(), name, lenient);
		if (m==null) {
			if (lenient) return null;
			else throw new IllegalArgumentException("property "+name+" in "+obj.getClass()+" is not readable");
		}

		return callSafe(obj, m, null);
	}

	protected static Method getReadMethod(Class cls, String name, boolean lenient) {
		String n = "get" + name.substring(0, 1).toUpperCase() + name.substring(1);
		Method m;
		
		try {
			m = BeanUtils.getMethod(cls, n);
		} catch (NoSuchMethodException e) {
			try {
				m = BeanUtils.getMethod(cls, name);
			} catch (NoSuchMethodException ex) {
				if (lenient) return null;
				else throw new IllegalArgumentException("no such method in "+cls+": "+n, ex);
			}
		}
		
		if (m.getReturnType()==Void.TYPE) throw new IllegalArgumentException("not a getter: "+m);
		return m;
	}

	protected static Method getWriteMethod(Class cls, String name, Class type) {
		String n = "set" + name.substring(0, 1).toUpperCase() + name.substring(1);
		Method m = null; 
		try {
			m = BeanUtils.getMethod(cls,n, type);
		} catch (NoSuchMethodException e) {
			//TODO: move this into BeanUtils.getMatchingMethod!
			Method[] mm = cls.getMethods();
			for(Method mth: mm) {
				if (mth.getName().equals(n)) {
					Class[] p = mth.getParameterTypes();
					if (p.length==1 && (type==null || p[0].isAssignableFrom(type))) {
						m = BeanUtils.getDefinedMethod(mth);
						break;
					}
				}
			}
			
			if (m==null) throw new IllegalArgumentException("no such method in "+cls+": "+n);
		}
		
		//if (m.getReturnType()!=Void.TYPE) throw new IllegalArgumentException("not a setter: "+m);
		return m;
	}

	public Class getPropertyType(String property) {
		return Object.class;
	}

	public Class<Object> getTargetType() {
		return Object.class;
	}

	public boolean isPropertyMutable(String name) {
		return true;
	}

	public void setProperty(Object obj, String name, Object v) throws IllegalArgumentException {
		Method m = getWriteMethod(obj.getClass(), name, v==null?null:v.getClass());
		if (m==null) throw new IllegalArgumentException("property "+name+" in "+obj.getClass()+" is not writable");

		callSafe(obj, m, new Object[] { v });
	}

	public boolean hasProperty(Object obj, String name) {
		return getReadMethod(obj.getClass(), name, true)!=null;
	}
	
}
