package de.brightbyte.abstraction;

import java.util.List;

import de.brightbyte.data.filter.Filter;

/**
 * A Filter implementation that applies another filter to all properties of a record,
 * and returns true of the filter matches any of the properties. This implements a
 * "full text / all field" type search and is used by SearchData implementations.
 */
public class AnyPropertyFilter<T> implements Filter<T> { 
	protected List<PropertyInfo> properties;
	protected Filter filter;
	protected boolean matchAsString = false;

	public AnyPropertyFilter(List<PropertyInfo> properties, Filter filter) {
		if (properties == null) throw new NullPointerException("properties must not be null");
		if (filter == null) throw new NullPointerException("filter must not be null");
		
		this.properties = properties;
		this.filter = filter;
	}

	public boolean matches(T rec) {
		for (PropertyInfo info: properties) {
			Object v = info.getValue(rec);

			v = convert(v);
			
			if (filter.matches(v)) return true;
		}
		
		return false;
	}
	
	protected Object convert(Object v) {
		if (matchAsString) v = v == null ? null : v.toString();
		return v;
	}

	public boolean isMatchAsString() {
		return matchAsString;
	}

	public void setMatchAsString(boolean matchAsString) {
		this.matchAsString = matchAsString;
	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((filter == null) ? 0 : filter.hashCode());
		result = PRIME * result + (matchAsString ? 1231 : 1237);
		result = PRIME * result + ((properties == null) ? 0 : properties.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final AnyPropertyFilter other = (AnyPropertyFilter) obj;
		if (filter == null) {
			if (other.filter != null)
				return false;
		} else if (!filter.equals(other.filter))
			return false;
		if (matchAsString != other.matchAsString)
			return false;
		if (properties == null) {
			if (other.properties != null)
				return false;
		} else if (!properties.equals(other.properties))
			return false;
		return true;
	}
	
	
}
