package de.brightbyte.rdf.sesame;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.openrdf.model.Graph;
import org.openrdf.model.GraphException;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.BNodeImpl;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.ValueFactoryImpl;
import org.openrdf.rio.RdfDocumentWriter;
import org.openrdf.rio.n3.N3Writer;
import org.openrdf.rio.ntriples.NTriplesWriter;
import org.openrdf.rio.rdfxml.RdfXmlWriter;
import org.openrdf.rio.turtle.TurtleWriter;
import org.openrdf.sesame.sail.StatementIterator;

import de.brightbyte.rdf.AbstractPlatform;
import de.brightbyte.rdf.RdfException;
import de.brightbyte.rdf.RdfNamespace;
import de.brightbyte.rdf.RdfPlatforms;
import de.brightbyte.util.PersistenceException;

public class SesamePlatform extends AbstractPlatform<Value, Resource, Resource, RdfDocumentWriter> {

	public static class SPI implements RdfPlatforms.SPI<SesamePlatform> {

		public SesamePlatform newPlatform() {
			return new SesamePlatform();
		}
	}
	
	protected ValueFactory factory;
	protected Graph graph;

	public SesamePlatform() {
		this( new ValueFactoryImpl(), null );
	}
		
	public SesamePlatform(ValueFactory factory, Graph graph) {
		super();
		this.graph = graph;
		this.factory = factory;
	}

	public String[] getSupportedFormats() {
		return getSupportedFormats("file");
	}

	public String[] getSupportedFormats(String location) {
		//NEW: diverse rdf stores (sail and other)
		if (location.equals("file")) return  new String[] { "N3", "Turtle", "NTriples", "XML" };
		else return new String[] { };
	}

	public Resource newAbout(Resource resource) throws RdfException {
		if (resource instanceof URIImpl) {
			if (graph==null) return new AboutURI((URIImpl)resource);
			else return new AboutURI((URIImpl)resource, graph);
		}
		else if (resource instanceof BNodeImpl) {
			if (graph==null) return new AboutBNode((BNodeImpl)resource);
			else return new AboutBNode((BNodeImpl)resource, graph);
		}
		else {
			return resource; //XXX: we don't know this works... no way to check if there's a backing store associated!
		}
	}

	public Value newLiteral(String value, String language) throws RdfException {
		return factory.createLiteral(value, language);
	}

	public Value newLiteral(String value, Resource datatype) throws RdfException {
		if (!(datatype instanceof URI)) throw new RdfException("datatype must be a URI");
		return factory.createLiteral(value, (URI)datatype);
	}

	public Resource newNode() throws RdfException {
		return factory.createBNode();
	}

	public Resource newResource(String namespace, String name) throws RdfException {
		return factory.createURI(namespace, name);
	}

	public RdfDocumentWriter newWriter(String location, String format) throws RdfException, PersistenceException {
		Pattern p = Pattern.compile("^(.*):(.*)$");
		Matcher m = p.matcher(location); 
		if (!m.matches()) throw new IllegalArgumentException("malformed location: "+location);
		
		String schema = m.group(1);
		String path = m.group(2);
		
		try {
			if (schema.equals("file")) {
				OutputStream out = new FileOutputStream(path);
				return newWriter(new BufferedOutputStream(out), format);
			}
			else {
				throw new IllegalArgumentException("bad location schema: "+schema);
			}
		} catch (IOException e) {
			throw new PersistenceException(e);
		} 
	}

	public RdfDocumentWriter newWriter(OutputStream out, String format) throws RdfException, PersistenceException {
		RdfDocumentWriter w;
		
		if (format.equalsIgnoreCase("N3")) w = new N3Writer(out);
		else if (format.equalsIgnoreCase("turtle")) w = new TurtleWriter(out);
		else if (format.equalsIgnoreCase("NTriples")) w = new NTriplesWriter(out);
		else if (format.equalsIgnoreCase("XML")) w = new RdfXmlWriter(out);
		else throw new IllegalArgumentException("unsupported format: "+format);
		
		configureWriter(w);
		return w;
	}

	public RdfDocumentWriter newWriter(Writer out, String format) throws RdfException, PersistenceException {
		RdfDocumentWriter w;
		
		if (format.equalsIgnoreCase("N3")) w = new N3Writer(out);
		else if (format.equalsIgnoreCase("turtle")) w = new TurtleWriter(out);
		else if (format.equalsIgnoreCase("NTriples")) w = new NTriplesWriter(out);
		else if (format.equalsIgnoreCase("XML")) w = new RdfXmlWriter(out);
		else throw new IllegalArgumentException("unsupported format: "+format);
		
		configureWriter(w);
		return w;
	}

	protected void configureWriter(RdfDocumentWriter w) throws PersistenceException {
		try {
			for (RdfNamespace<Value, Resource> ns: getNamespaces()) {
				w.setNamespace(ns.getPrefix(), ns.getNamespace());
			}
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	public void setProperty(Resource subject, Resource predicate, Value object) throws RdfException {
		if (!(predicate instanceof URI)) throw new RdfException("predicate must be a URI");
		
		try {
			subject.addProperty((URI)predicate, object);
		} catch (GraphException e) {
			throw new RdfException(e);
		}
	}

	public void writeFoot(RdfDocumentWriter writer) throws RdfException, PersistenceException {
		try {
			writer.endDocument();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	public void writeHead(RdfDocumentWriter writer) throws RdfException, PersistenceException {
		try {
			writer.startDocument();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	public void writeResource(RdfDocumentWriter writer, Resource about) throws RdfException, PersistenceException {
		try {
			StatementIterator it = about.getSubjectStatements();
			while (it.hasNext()) {
				Statement st = it.next();
				writeStatement(writer, st.getSubject(), st.getPredicate(), st.getObject());
			}
		} catch (GraphException e) {
			throw new RdfException(e);
		}
	}

	public void writeStatement(RdfDocumentWriter writer, Resource subject, Resource predicate, Value object) throws RdfException, PersistenceException {
		if (!(predicate instanceof URI)) throw new RdfException("predicate must be a URI");
		try {
			writer.writeStatement(subject, (URI)predicate, object);
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	public void closeWriter(RdfDocumentWriter writer) throws RdfException, PersistenceException {
		//noop
	}

	public void flushWriter(RdfDocumentWriter writer) throws RdfException, PersistenceException {
		//noop
	}

}
