package de.brightbyte.rdf.aardfark;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import de.brightbyte.io.Output;
import de.brightbyte.rdf.RdfException;
import de.brightbyte.rdf.RdfNamespace;
import de.brightbyte.rdf.vocab.XSDatatypes;
import de.brightbyte.util.PersistenceException;

public class N3Sink extends RdfStreamSink {
	
	protected boolean turtle = true;
	protected XSDatatypes xsd;

	public N3Sink(AardfarkPlatform context, File f) throws UnsupportedEncodingException, FileNotFoundException, RdfException {
		super(context, f, "UTF-8");
		init(context);
	}

	public N3Sink(AardfarkPlatform context, Output output) throws RdfException{
		super(context, output);
		init(context);
	}

	public N3Sink(AardfarkPlatform context, OutputStream out) throws UnsupportedEncodingException, RdfException {
		super(context, out, "UTF-8");
		init(context);
	}

	public N3Sink(AardfarkPlatform context, Writer out) throws RdfException{
		super(context, out);
		init(context);
	}
	
	@SuppressWarnings("unchecked")
	private void init(AardfarkPlatform context) throws RdfException {
		xsd = context.loadNamespace(XSDatatypes.class, false);
	}

	public N3Sink setTurtleMode(boolean turtle) {
		this.turtle = turtle;
		return this;
	}
	
	public void prepare() throws RdfException, PersistenceException {
		if (turtle) {
			for (RdfNamespace<RdfValue, RdfResource> vocab: context.getNamespaces()) {
				if (vocab.getPrefix().equals("")) writeln("@base <"+vocab.getNamespace()+"> .");
				//NOTE: always declare base also as empty prefix!  
				
				writeln("@prefix "+vocab.getPrefix()+": <"+vocab.getNamespace()+"> .");
			}
		}
	}
	
	public void putStatement(RdfResource subject, RdfResource predicate, RdfValue object) throws RdfException, PersistenceException {
		String s = getN3Value(subject)+" "+getN3Value(predicate)+" "+getN3Value(object)+" .";
		writeln(s);
	}

	@Override
	public void putAbout(RdfResource subject, Map<RdfResource, RdfValue> properties) throws RdfException, PersistenceException {
		if (!turtle) {
			super.putAbout(subject, properties);
		}
		else {
			List<RdfPropertyInstance> props = new ArrayList<RdfPropertyInstance>(properties.size());
			for (Map.Entry<RdfResource, RdfValue> prop : properties.entrySet()) {
				RdfPropertyInstance p = new RdfPropertyInstance(prop.getKey(), prop.getValue());
				props.add(p);
			}
			
			putAbout(subject, props);
		}
	}

	@Override
	public void putAbout(RdfResource subject, List<RdfPropertyInstance> properties) throws RdfException, PersistenceException {
		if (!turtle) {
			super.putAbout(subject, properties);
		}
		else {
			boolean first = true;
			for (RdfPropertyInstance prop : properties) {
				if (first) {
					write(getN3Value(subject));
					write(" ");
					first = false;
				}
				else {
					write(" ;\n\t");
				}
				
				write(getN3Value(prop.getPredicate()));
				write(" ");
				write(getN3Value(prop.getObject()));
			}
			writeln(" .");
		}
	}
	
	public String getN3Value(RdfValue rc) throws RdfException, PersistenceException {
		if (rc instanceof RdfLiteral) {
			RdfResource type = ((RdfLiteral)rc).getType();
			String v = ((RdfLiteral)rc).getValue();
			
			if (turtle && xsd !=null && type!=null) {
				//TODO: validate each type!
				if (type.equals(xsd._boolean)) return v; 
				else if (type.equals(xsd._byte) 
						|| type.equals(xsd._int) 
						|| type.equals(xsd._long) 
						|| type.equals(xsd._short)
						|| type.equals(xsd.integer) ) return v; 
				else if (type.equals(xsd._double) 
						|| type.equals(xsd._float)
						|| type.equals(xsd.decimal)) return v; 
			}
			
			String s ="\""+escapeN3(v, false, true)+"\"";
			
			String lang = ((RdfLiteral)rc).getLanguage();
			if (lang!=null) s += "@" + lang;
			if (type!=null) s += "^^" + getN3Value(type);
			return s;
		}
		else if (rc instanceof RdfNode) {
			return "<_:"+((RdfNode)rc).getId()+">";
		}
		else if (rc instanceof RdfReference) {
			String qname = turtle ? context.getQName(((RdfReference)rc)) : null;
			if (qname!=null) return escapeN3(qname, false, false);
			else {
				String u = context.getURI((RdfReference)rc, turtle);
				u = escapeN3(u, true, false);
				return "<"+u+">";
			}
		}
		else throw new IllegalArgumentException("unknown type of RDF resource: "+rc.getClass().getName());
	}
	
	/**
	 * reference: http://www.dajobe.org/2004/01/turtle/#sec-strings and http://www.w3.org/2001/sw/RDFCore/ntriples/#sec-issues
	 * @param s
	 * @return
	 */
	public String escapeN3(String s, boolean inURI, boolean inString) {
		StringBuilder b = null;
		
		int c = s.length();
		for (int i=0; i<c; i++) {
			char ch = s.charAt(i);
			char esc = '\0';
			int cv = (int)ch;
			
			if (ch=='\\') esc = '\\';
			else if (ch=='\t' && turtle) esc = 't';
			else if (ch=='\r' && turtle) esc = 'r';
			else if (ch=='\n' && turtle) esc = 'n';
			else if (ch=='\"' && inString && turtle) esc = '"';
			else if (ch=='>' && inURI && turtle) esc = '>';
			else if (cv<32 || (!turtle && cv>127)) {
				if (b==null) b = new StringBuilder(s.substring(0, i));
				
				b.append("\\u");
				int m = 0xF000;
				int o = 12;
				
				while(m>0) {
					int v = (m & cv) >>> o;
					
					char vc;
					if (v < 10) vc = (char)((int)'0' + v); 
					else if (v < 16) vc = (char)((int)'A' + (v-10));
					else throw new RuntimeException("oops!");
					
					b.append(vc);

					m = m >>> 4;
					o -= 4;
				}
			}
			else if (b!=null) {
				b.append(ch);
			}
			
			if (esc!='\0') {
				if (b==null) b = new StringBuilder(s.substring(0, i));
				b.append('\\');
				b.append(esc);
			}
		}
		
		if (b==null) return s;
		else return b.toString();
	}

	public String getBaseURI() {
		return context.getBaseURI();
	}
	
}
