package de.brightbyte.rdf.aardfark;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.brightbyte.net.UrlUtils;
import de.brightbyte.rdf.AbstractPlatform;
import de.brightbyte.rdf.RdfException;
import de.brightbyte.rdf.RdfNamespace;
import de.brightbyte.rdf.RdfPlatforms;
import de.brightbyte.util.PersistenceException;

public class AardfarkPlatform extends AbstractPlatform<RdfValue, RdfResource, RdfAbout, RdfSink> implements RdfContext {
	public static class SPI implements RdfPlatforms.SPI<AardfarkPlatform> {

		public AardfarkPlatform newPlatform() {
			return new AardfarkPlatform();
		}
	}

	public String[] getSupportedFormats() {
		return getSupportedFormats("file");
	}

	public String[] getSupportedFormats(String location) {
		if (location.equals("file")) return  new String[] { "N3", "Turtle", "NTriples" };
		else return new String[] { };
	}

	public RdfValue newLiteral(String value, String language) {
		return new RdfLiteral(value, language);
	}

	public RdfValue newLiteral(String value, RdfResource datatype) {
		return new RdfLiteral(value, datatype);
	}

	public RdfSink newWriter(String location, String format) throws PersistenceException, RdfException {
		Pattern p = Pattern.compile("^(.*):(.*)$");
		Matcher m = p.matcher(location); 
		if (!m.matches()) throw new IllegalArgumentException("malformed location: "+location);
		
		String schema = m.group(1);
		String path = m.group(2);
		
		try {
			if (schema.equals("file")) {
				OutputStream out = new FileOutputStream(path);
				return newWriter(new BufferedOutputStream(out), format);
			}
			else {
				throw new IllegalArgumentException("bad location schema: "+schema);
			}
		} catch (IOException e) {
			throw new PersistenceException(e);
		} 
	}

	public RdfSink newWriter(OutputStream out, String format) throws RdfException, PersistenceException {
		try {
			if (format.equalsIgnoreCase("ntriples")) return new N3Sink(this, out).setTurtleMode(false);
			else if (format.equalsIgnoreCase("turtle") || format.equalsIgnoreCase("n3")) return new N3Sink(this, out).setTurtleMode(true);
			else throw new IllegalArgumentException("unsupported format: "+format);
		} catch (UnsupportedEncodingException e) {
			throw new PersistenceException(e);
		}
	}

	public RdfSink newWriter(Writer out, String format) throws RdfException {
		if (format.equalsIgnoreCase("ntriples")) return new N3Sink(this, out).setTurtleMode(false);
		else if (format.equalsIgnoreCase("turtle") || format.equalsIgnoreCase("n3")) return new N3Sink(this, out).setTurtleMode(true);
		else throw new IllegalArgumentException("unsupported format: "+format);
	}

	public void setProperty(RdfAbout subject, RdfResource predicate, RdfValue object) {
		subject.setProperty(predicate, object);
	}

	public void writeFoot(RdfSink writer) throws PersistenceException, RdfException {
		writer.finish();
	}

	public void writeHead(RdfSink writer) throws PersistenceException, RdfException {
		writer.prepare();
	}

	public void writeResource(RdfSink writer, RdfAbout about) throws RdfException, PersistenceException {
		writer.putAbout(about);
	}

	public void writeStatement(RdfSink writer, RdfResource subject, RdfResource predicate, RdfValue object) throws RdfException, PersistenceException {
		writer.putStatement(subject, predicate, object);
	}

	public RdfAbout newAbout(RdfResource resource) throws RdfException {
		return new RdfAbout(resource);
	}

	public RdfResource newNode() throws RdfException {
		return new RdfNode();
	}

	public RdfResource newResource(String namespace, String name) throws RdfException {
		return new RdfReference(this, namespace, name);
	}
	
	/*
	public String getQName() {
		if (namespace==null || name==null || !isPlainName(name)) return null;
		else return namespace+":"+name; 
	}
	*/

	public String getQName(RdfReference reference) {
		RdfNamespace<RdfValue, RdfResource> v = getNamespace(reference.getNamespace());
		if (v==null) return null;
		
		String n = reference.getLocalName();
		if (!isPlainName(n)) return null;
		
		return v.getPrefix()+":"+n;
	}

	public static final String criticalChars = UrlUtils.criticalChars+"<>";

	public String getURI(RdfReference reference, boolean iri) throws RdfException {
		String n = reference.getLocalName();
		
		if (!iri) n = UrlUtils.encode(n);
		else n = UrlUtils.encodeCritical(n, criticalChars); 
		
		return reference.getNamespace()+n; 
	}

	public void closeWriter(RdfSink writer) throws RdfException, PersistenceException {
		writer.close();
	}

	public void flushWriter(RdfSink writer) throws RdfException, PersistenceException {
		writer.flush();
	}
	
}
