package de.brightbyte.rdf;

import java.io.StringWriter;

import de.brightbyte.rdf.RdfPlatform;
import de.brightbyte.util.PersistenceException;
import junit.framework.TestCase;

public abstract class PlatformTestBase<V, R extends V, A, W> extends TestCase {

	protected String type;
	
	public PlatformTestBase(String type) {
		this.type = type;
	}
	
	public void testTurtle() throws RdfException, PersistenceException {
		RdfPlatform<V, R, A, W> platform = RdfPlatforms.newPlatform(type);

		//Following the example at <http://www.w3.org/2000/10/swap/test/meet/blue.n3>
		//discussed at <http://www.w3.org/2000/Talks/0906-xmlweb-tbl/slide6-0.html>
		//found at <http://www.w3.org/2000/10/swap/Examples.html>
		
		GenericNamespace<V, R> p = platform.addNamespace("http://www.example.org/personal_details#", "p");
		GenericNamespace<V, R> m = platform.addNamespace("http://www.example.org/meeting_organization#", "m");
		
		R GivenName = p.getResource("GivenName");
		R hasEmail =  p.getResource("hasEmail");
		R attending = m.getResource("attending");
		R homePage =  m.getResource("homePage");
		
		StringWriter buff = new StringWriter();
		W writer = platform.newWriter(buff, "turtle");
		platform.writeHead(writer);		
		
		R fred = platform.newResource("http://www.example.org/people#", "fred");
		R m1 =   platform.newResource("http://meetings.example.com/cal#", "m1");
		
		A aboutFred = platform.newAbout(fred);
		platform.setLiteralProperty(aboutFred, GivenName, "Fred", (String)null);
		platform.setReferenceProperty(aboutFred, hasEmail, "mailto:", "fred@example.com"); //XXX: mail URIs can't be assembly simply by appending to the host name!
		platform.setProperty(aboutFred, attending, m1);
		
		platform.writeResource(writer, aboutFred);

		platform.writeStatement(writer, m1, homePage, platform.newResource("http://meetings.example.com", "/m1/hp"));
		platform.writeFoot(writer);
		
		String s = buff.toString();
		String exp = 
			"@prefix m:  <http://www.example.org/meeting_organization#> ."+
			"@prefix p:  <http://www.example.org/personal_details#> ."+
			""+
			"<http://www.example.org/people#fred>"+
			"		p:GivenName  	\"Fred\";"+
			"		p:hasEmail 		<mailto:fred@example.com>;"+
			"		m:attending 	<http://meetings.example.com/cal#m1> ."+
			""+
			"<http://meetings.example.com/cal#m1>"+
			"		m:homePage 		<http://meetings.example.com/m1/hp> .";
			
		exp = exp.replaceAll("[ \t\r\n]+", " ").replaceAll("[ \t\r\n]+([;.@^<>])", "$1").trim();
		s   =   s.replaceAll("[ \t\r\n]+", " ").replaceAll("[ \t\r\n]+([;.@^<>])", "$1").trim();
		
		assertEquals(exp, s);
	}

}
