package de.brightbyte.db.testing;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import junit.framework.AssertionFailedError;
import de.brightbyte.db.DatabaseAccess;
import de.brightbyte.db.DatabaseUtil;
import de.brightbyte.util.Counter;
import de.brightbyte.util.StructuredDataCodec;

public class DatabaseTestUtils {
	public static class TestCase {
		protected String name;
		protected String sql;
		protected List<Object[]> data; 
		
		public TestCase(String name, String sql, List<Object[]> data) {
			super();
			this.name = name;
			this.sql = sql;
			this.data = data;
		}

		public void run(DatabaseAccess access) throws SQLException {
			String q = sql.replace("/*prefix*/", access.getTablePrefix());
			
			ResultSet rs = access.executeQuery(name, q);
			ResultSetMetaData meta = rs.getMetaData();
			
			int i = 0;
			while (rs.next()) {
				i++;

				Object[] actual = decode(rs, meta);
				
				if (i>data.size()) {
					fail(name+": unexpected data: row "+i+", found "+Arrays.toString(actual));
				}
				
				Object[] expected = data.get(i-1);
				
				if (!Arrays.equals(expected, actual)) {
					fail(name+": data mismatch in row "+i+": expected "+Arrays.toString(expected)+", found "+Arrays.toString(actual));
				}
			}
			
			if (i<data.size()) {
				fail(name+": missing data: row "+(i+1));
			}
		}
		
		public String getName() {
			return name;
		}
		
		@Override
		public String toString() {
			return "Test case "+getName();
		}
	}

	private static final StructuredDataCodec codec = new StructuredDataCodec();

	protected static Object[] decode(String[] row, ResultSetMetaData meta) throws SQLException {
		Object[] a = new Object[row.length];
		
		for (int i=0; i<a.length; i++) {
			if (DatabaseUtil.isNumeric(meta, i+1)) {
				a[i] = DatabaseUtil.asDouble(row[i]);
			}
			else {
				a[i] = DatabaseUtil.asString(row[i]);
			}
		}
		
		return a;
	}

	protected static Object[] decode(ResultSet rs, ResultSetMetaData meta) throws SQLException {
		int cc = meta.getColumnCount();
		Object[] a = new Object[cc];
		
		for (int i=0; i<cc; i++) {
			Object obj = rs.getObject(i+1);
			
			if (DatabaseUtil.isInteger(meta, i+1)) {
				a[i] = DatabaseUtil.asInt(obj);
			}
			else if (DatabaseUtil.isNumeric(meta, i+1)) {
				a[i] = DatabaseUtil.asDouble(obj);
			}
			else {
				a[i] = DatabaseUtil.asString(obj);
			}
		}
		
		return a;
	}
	
	protected static void fail(String msg) {
		throw new AssertionFailedError(msg);
	}
	
	public static List<TestCase> loadTestCases(Class<?> context, String name) throws IOException {
		URL u = context.getResource(name);
		if (u==null) throw new FileNotFoundException(name+" in the context of "+context);
		
		return loadTestCases(u);
	}
	
	public static List<TestCase> loadTestCases(URL u) throws IOException {
		BufferedReader in = new BufferedReader(new InputStreamReader(u.openStream(), "UTF-8"));
		List<TestCase> tests = new ArrayList<TestCase>();
		
		String name = u.getPath().replaceAll("/?(.*/)*(.*?)(\\.[^./]*)?$", "$2");
		
		try {
			TestCase t;
			Counter line = new Counter();
			
			while ((t = loadTestCase(name, in, u.toString(), line)) != null) {
				tests.add(t);
			}
			
			return tests;
		}
		finally {
			in.close();
		}
	}
	
	public static TestCase loadTestCase(String group, BufferedReader in, String filename, Counter line) throws IOException {
		String s;
		String name = null;
		while ((s = in.readLine()) != null) {
			if (s.trim().length()==0) continue;
			if (s.matches("\\s*---+\\s*")) return null;
			if (s.matches("\\s*(--|#|%).*")) continue;
			
			name = s.trim();
			break;
		}

		if (name==null) return null;
		
		StringBuilder sql = new StringBuilder();
		List<Object[]> data = new ArrayList<Object[]>();
		
		while ((s = in.readLine()) != null) {
			line.inc();
			
			s = s.trim();
			if (s.trim().length()==0) continue;
			if (s.matches("\\s*---+\\s*")) break;
			if (s.matches("\\s*(--|#|%).*")) continue;
			
			sql.append(s);
			sql.append(' ');
			
			if (sql.length()>0 && s.length()==0) break;
		}
		
		while ((s = in.readLine()) != null) {
			line.inc();
			
			if (s.trim().trim().length()==0) continue;
			if (s.matches("\\s*---+\\s*")) break;
			if (s.matches("\\s*(--|#|%).*")) continue;
			
			try {
				List<Object> v = (List<Object>)codec.decodeValue("["+s+"]");
				Object[] d = (Object[]) v.toArray(new Object[v.size()]);
				data.add(d);
			}
			catch (IllegalArgumentException ex) {
				throw (IOException)new IOException("Malformed data in "+filename+" at line "+line).initCause(ex);
			}
		}
		
		return new TestCase(group+": "+name, sql.toString(), data);
	}
}
