package de.brightbyte.db.file;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.SQLException;

import de.brightbyte.db.Inserter;
import de.brightbyte.util.PersistenceException;

public class TsvDump extends AbstractDump<BufferedReader, Writer> {
	
	public static abstract class Codec {
		protected String name;

		public Codec(String name) {
			super();
			this.name = name;
		}
		
		public abstract void decodeField(String v, Inserter drain)
			throws IOException, SQLException;
		
		public abstract void writeField(ResultSet source, Writer drain)
			throws IOException, SQLException;
	}
	
	public static class StringCodec extends Codec {
		private String encoding;

		public StringCodec(String name, String enc) {
			super(name);
			this.encoding = enc;
		}

		@Override
		public void decodeField(String v, Inserter drain) throws IOException, SQLException {
			drain.updateString(name, v);
		}

		@Override
		public void writeField(ResultSet source, Writer drain) throws IOException, SQLException {
			String s;
			if (encoding == null) s = source.getString(name);
			else s = new String(source.getBytes(name));
				
			s = s.replace('\t', ' ');
			drain.write(s);
		}
	}
	
	public static class IntegerCodec extends Codec {
		public IntegerCodec(String name) {
			super(name);
		}

		@Override
		public void decodeField(String v, Inserter drain) throws IOException, SQLException {
			int i = Integer.parseInt(v);
			drain.updateInt(name, i);
		}

		@Override
		public void writeField(ResultSet source, Writer drain) throws IOException, SQLException {
			int i = source.getInt(name);
			drain.write(String.valueOf(i));
		}
	}
	
	public static class DoubleCodec extends Codec {
		public DoubleCodec(String name) {
			super(name);
		}

		@Override
		public void decodeField(String v, Inserter drain) throws IOException, SQLException {
			double d = Double.parseDouble(v);
			drain.updateDouble(name, d);
		}

		@Override
		public void writeField(ResultSet source, Writer drain) throws IOException, SQLException {
			double d = source.getDouble(name);
			drain.write(String.valueOf(d));
		}
	}

	protected Codec[] columnCodecs;
	protected int bufferSize = 16 * 1024;
	protected String enc = "UTF-8";
	
	public TsvDump(String name, Codec[] columnCodecs) {
		super(name);
		this.columnCodecs = columnCodecs;
	}

	@Override
	protected boolean readRow(BufferedReader source, Inserter drain) throws IOException, SQLException {
		String s = source.readLine();
		if (s==null) return false;
		
		String[] ss = s.split("\t", columnCodecs.length); //FIXME: this is rather slow!
		int i = 0;
		for (Codec c: columnCodecs) {
			c.decodeField(ss[i++], drain);
		}
		
		drain.updateRow();
		return true;
	}
	

	@Override
	protected boolean writeRow(ResultSet source, Writer drain) throws IOException, SQLException {
		if (!source.next()) return false;
		
		boolean first = true;
		for (Codec c: columnCodecs) {
			if (first) first = false;
			else drain.write('\t');
			
			c.writeField(source, drain);
		}
		
		drain.write('\n'); 
		
		return true;
	}

	@Override
	protected Writer openDrain(File f) throws PersistenceException {
		try {
			return new BufferedWriter( new OutputStreamWriter( new FileOutputStream(f), enc ), bufferSize );
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	@Override
	protected BufferedReader openSource(File f) throws PersistenceException {
		try {
			return new BufferedReader( new InputStreamReader( new FileInputStream(f), enc ), bufferSize );
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	@Override
	protected void closeDrain(Writer drain) throws PersistenceException {
		try {
			drain.close();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	@Override
	protected void closeSource(BufferedReader source) throws PersistenceException {
		try {
			source.close();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}
	
}
