package de.brightbyte.db.file;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;

import de.brightbyte.db.Inserter;
import de.brightbyte.util.PersistenceException;

public class BinaryDump extends AbstractDump<DataInputStream, DataOutputStream> {
	
	public static abstract class Codec {
		protected String name;

		public Codec(String name) {
			super();
			this.name = name;
		}
		
		public abstract void readField(DataInputStream source, Inserter drain)
			throws IOException, SQLException;
		
		public abstract void writeField(ResultSet source, DataOutputStream drain)
			throws IOException, SQLException;
	}
	
	public static class StringCodec extends Codec {
		String encoding;
		
		public StringCodec(String name, String enc) {
			super(name);
			this.encoding = enc;
		}

		@Override
		public void readField(DataInputStream source, Inserter drain) throws IOException, SQLException {
			String s = source.readUTF();
			drain.updateString(name, s);
		}

		@Override
		public void writeField(ResultSet source, DataOutputStream drain) throws IOException, SQLException {
			String s;
			
			if (encoding==null) s = source.getString(name);
			else s = new String(source.getBytes(name), encoding);
				
			drain.writeUTF(s);
		}
	}
	
	public static class IntegerCodec extends Codec {
		public IntegerCodec(String name) {
			super(name);
		}

		@Override
		public void readField(DataInputStream source, Inserter drain) throws IOException, SQLException {
			int i = source.readInt();
			drain.updateInt(name, i);
		}

		@Override
		public void writeField(ResultSet source, DataOutputStream drain) throws IOException, SQLException {
			int i = source.getInt(name);
			drain.writeInt(i);
		}
	}
	
	public static class DoubleCodec extends Codec {
		public DoubleCodec(String name) {
			super(name);
		}

		@Override
		public void readField(DataInputStream source, Inserter drain) throws IOException, SQLException {
			double d = source.readDouble();
			drain.updateDouble(name, d);
		}

		@Override
		public void writeField(ResultSet source, DataOutputStream drain) throws IOException, SQLException {
			double d = source.getDouble(name);
			drain.writeDouble(d);
		}
	}

	protected Codec[] columnCodecs;
	protected int bufferSize = 16 * 1024;
	
	public BinaryDump(String name, Codec[] columnCodecs) {
		super(name);
		this.columnCodecs = columnCodecs;
	}

	@Override
	protected boolean readRow(DataInputStream source, Inserter drain) throws IOException, SQLException {
		int code = source.read();
		if (code<0) return false; //Implicit EOF
		if (code==4) return false; //Explicit EOF (ASCII EOT)
		if (code!=30) throw new IOException("bad record separator code: "+code); //expect ASCII Record Separator
		
		for (Codec c: columnCodecs) {
			c.readField(source, drain);
		}
		
		drain.updateRow();
		return true;
	}
	

	@Override
	protected boolean writeRow(ResultSet source, DataOutputStream drain) throws IOException, SQLException {
		if (!source.next()) return false;
		
		drain.writeByte(30); //ASCII record separator

		for (Codec c: columnCodecs) {
			c.writeField(source, drain);
		}		
		
		return true;
	}

	@Override
	protected DataOutputStream openDrain(File f) throws PersistenceException {
		try {
			return new DataOutputStream( new BufferedOutputStream( new FileOutputStream(f), bufferSize ) );
		} catch (FileNotFoundException e) {
			throw new PersistenceException(e);
		}
	}

	@Override
	protected DataInputStream openSource(File f) throws PersistenceException {
		try {
			return new DataInputStream( new BufferedInputStream( new FileInputStream(f), bufferSize ) );
		} catch (FileNotFoundException e) {
			throw new PersistenceException(e);
		}
	}

	@Override
	protected void closeDrain(DataOutputStream drain) throws PersistenceException {
		try {
			drain.close();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}

	@Override
	protected void closeSource(DataInputStream source) throws PersistenceException {
		try {
			source.close();
		} catch (IOException e) {
			throw new PersistenceException(e);
		}
	}
	
}
