package de.brightbyte.db.file;

import java.io.File;
import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;

import de.brightbyte.db.Inserter;
import de.brightbyte.job.Progress;
import de.brightbyte.job.Progress.Listener;
import de.brightbyte.util.PersistenceException;

public abstract class AbstractDump<S, D> implements DumpReader, DumpWriter, Progress.Source  {
	
	protected String name;
	protected long total = -1;
	protected Progress.State progress;

	public AbstractDump(String name) {
		super();
		this.name = name;
	}

	public void read(File f, Inserter drain) throws PersistenceException {
		S source = openSource(f);
		read(source, drain);
		closeSource(source);
	}

	public void write(ResultSet source, File f) throws PersistenceException {
		D drain = openDrain(f);
		write(source, drain);
		closeDrain(drain);
	}
	
	protected abstract S openSource(File f) throws PersistenceException;
	protected abstract D openDrain(File f) throws PersistenceException;

	protected abstract void closeSource(S source) throws PersistenceException;
	protected abstract void closeDrain(D drain) throws PersistenceException;

	protected abstract boolean readRow(S source, Inserter drain) throws IOException, SQLException;

	public void read(S source, Inserter drain) throws PersistenceException {
		if (progress!=null) {
			progress.reset();
			progress.fireStart(total);
		}
		
		try {
			while (readRow(source, drain)) {
				if (progress!=null) progress.fireStep(1);
			}
			
			if (progress!=null) progress.fireDone(total);
		} catch (SQLException e) {
			if (progress!=null) progress.fireFailed(e);
			throw new PersistenceException(e);			
		} catch (IOException e) {
			if (progress!=null) progress.fireFailed(e);
			throw new PersistenceException(e);			
		} catch (RuntimeException e) {
			if (progress!=null) progress.fireFailed(e);
			throw e;			
		} finally {
			if (progress!=null && !progress.isTerminated()) progress.fireFailed(null);
		}
	}

	protected abstract boolean writeRow(ResultSet source, D drain) throws IOException, SQLException;

	public void write(ResultSet source, D drain) throws PersistenceException {
		if (progress!=null) {
			progress.reset();
			progress.fireStart(total);
		}
		
		try {
			while (writeRow(source, drain)) {
				if (progress!=null) progress.fireStep(1);
			}
			
			if (progress!=null) progress.fireDone(total);
		} catch (SQLException e) {
			if (progress!=null) progress.fireFailed(e);
			throw new PersistenceException(e);			
		} catch (IOException e) {
			if (progress!=null) progress.fireFailed(e);
			throw new PersistenceException(e);			
		} catch (RuntimeException e) {
			if (progress!=null) progress.fireFailed(e);
			throw e;			
		} finally {
			if (progress!=null && !progress.isTerminated()) progress.fireFailed(null);
		}
	}
	
	public void addProgressListener(Listener li) {
		if (progress==null) progress = new Progress.State(this, name);
		progress.addProgressListener(li);
	}

	public void removeProgressListener(Listener li) {
		if (progress!=null) progress.removeProgressListener(li);
	}

}