package de.brightbyte.db;

import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public abstract class DatabaseTable {
	protected DatabaseAccess access;
	protected String name;
	protected String qname;
	protected String attributes;
	
	protected Map<String, DatabaseField> fields;
	protected Map<String, DatabaseKey> keys;
	
	//protected Inserter inserter;
	protected InserterFactory inserterFactory;
	
	protected boolean knowsAutomaticField = false;
	protected String automaticField = null;
	protected int bufferWidth = -1;
	protected int bufferLength = 128;

	public DatabaseTable(DatabaseAccess access, String name, String attributes/*, int insertBufferSize*/) {
		this.access  = access;
		this.name = name;
		this.attributes = attributes;
		//this.insertBufferSize = insertBufferSize;
		
		String prefix = access.getTablePrefix();
		if (prefix!=null) this.qname = prefix + name;
		else this.qname = name;
		
		fields = new HashMap<String, DatabaseField>();
		keys = new HashMap<String, DatabaseKey>();
		
		//if (insertBufferSize>0) insertBuffer = new ArrayList<String>(insertBuffer);
	}
	
	public String getSQLName() {
		return qname; 
	}
	
	public String getCreateStatement(boolean opt) {
		StringBuilder sql = new StringBuilder();
		sql.append("CREATE TABLE ");
		if (opt) sql.append(" IF NOT EXISTS ");
		sql.append(access.quoteName(qname));
		sql.append(" ( ");
		
		boolean first = true;
		for (DatabaseField field : fields.values()) {
			if (first) first = false;
			else sql.append(", ");
			
			sql.append(field.getSQL());
		}
		
		for (DatabaseKey key : keys.values()) {
			if (first) first = false;
			else sql.append(", ");
			
			sql.append(key.getSQL());
		}
		
		sql.append(" ) ");
		if (attributes!=null) sql.append(attributes);
		sql.append(";");
		
		return sql.toString();
	}
	
	public String getDropStatement(boolean opt) {
		StringBuilder sql = new StringBuilder();
		sql.append("DROP TABLE ");
		if (opt) sql.append(" IF EXISTS ");
		sql.append(access.quoteName(qname));
		sql.append(";");
		return sql.toString();
	}
	
	public String getTruncateStatement() {
		return "TRUNCATE TABLE " + access.quoteName(qname);
	}
	
	/*
	public String makeInsertStatement(Map<String, Object> ) {
		StringBuilder sql = new StringBuilder();
		sql.append("DROP TABLE ");
		sql.append(qname);
		if (opt) sql.append(" IF EXISTS ");
		sql.append(";");
		return sql.toString();
	}
	*/
	
	public void addField(DatabaseField field) {
		fields.put(field.getName(), field);
		
		KeyType t = field.getKeyType();
		if (t!=null) {
			DatabaseKey k = new DatabaseKey(access, t, field.getName(), new String[] { field.getName() });
			addKey(k);
		}
	}
	

	public DatabaseField removeField(String name) {
		return fields.remove(name);
	}

	public DatabaseKey removeKey(String name) {
		return keys.remove(name);
	}

	public void addKey(DatabaseKey k) {
		keys.put(k.getName(), k);
	}
	
	public DatabaseField getField(String name) {
		DatabaseField f = fields.get(name);
		if (f==null) throw new IllegalArgumentException("no such field in table "+this.name+": "+name);
		return f;
	}

	public DatabaseKey getKey(String name) {
		DatabaseKey k = keys.get(name);
		if (k==null) throw new IllegalArgumentException("no such key in table "+this.name+": "+name);
		return k;
	}

	public String getAttributes() {
		return attributes;
	}

	/*
	public int getInsertBufferSize() {
		return insertBufferSize;
	}*/

	public String getName() {
		return name;
	}

	@Override
	public String toString() {
		return "TABLE "+name;
	}

	public Map<String, DatabaseField> getFields() {
		return Collections.unmodifiableMap(fields);
	}

	public DatabaseAccess getAccess() {
		return access;
	}
	
	public Inserter getInserter() throws SQLException {
		Inserter inserter = null;
		
		if (inserterFactory!=null) {
			inserter = inserterFactory.newInserter(this, getAutomaticField(), getBufferLength() * access.getBufferScale(), getBufferWidth());
		}
				
		//XXX: if no factory is set, do what? return null? throw exception? create non-buffering inserter? 
		return inserter;
	}

	public String getAutomaticField() {
		if (knowsAutomaticField) return automaticField;
		knowsAutomaticField = true;
		
		for(DatabaseField f: fields.values()) { //TODO: maybe look for AUTO_INCREMENT attribute instead?
			if (f.getKeyType() == KeyType.PRIMARY) return f.getName();
		}
		
		return null;
	}

	public void setAutomaticField(String automaticField) {
		if (automaticField!=null && !fields.containsKey(automaticField)) throw new IllegalArgumentException("unknown field: "+automaticField);
		this.knowsAutomaticField = true;
		this.automaticField = automaticField;
	}

	public int getBufferLength() {
		return bufferLength;
	}

	public void setBufferLength(int bufferLength) {
		this.bufferLength = bufferLength;
	}

	public int getBufferWidth() {
		if (bufferWidth>0) return bufferWidth;
		
		int w = 3;
		for(DatabaseField f: fields.values()) {
			w+= f.getSize() + 2; 
		}
		
		return w;
	}

	public void setBufferWidth(int bufferWidth) {
		this.bufferWidth = bufferWidth;
	}

	public InserterFactory getInserterFactory() {
		return inserterFactory;
	}

	public void setInserterFactory(InserterFactory inserterFactory) {
		this.inserterFactory = inserterFactory;
	}	
	
	public void setBufferDimensions(int length, int width) {
		setBufferLength(length);
		setBufferWidth(width);
	}

	/*
	public void flushInserter() throws SQLException {
		if (inserter!=null) inserter.flush();
	}
	*/
}
