package de.brightbyte.db;

import static de.brightbyte.util.LogLevels.LOG_INFO;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.sql.DataSource;

import de.brightbyte.application.Agenda;
import de.brightbyte.io.LeveledOutput;
import de.brightbyte.io.LogOutput;
import de.brightbyte.io.Output;
import de.brightbyte.job.BlockingJobQueue;
import de.brightbyte.job.FinishedFuture;
import de.brightbyte.util.PersistenceException;
import de.brightbyte.util.Processor;
import de.brightbyte.util.StringUtils;

public class DatabaseSchema implements DatabaseAccess {
	
	public interface DialectHandler {

		public String mangleType(String type);
		//FIXME: do all the quoting here!
	}
	
	protected String prefix;
	protected String defaultTableAttributes;

	protected Map<String, DatabaseTable> tables = new HashMap<String, DatabaseTable>();
	
	protected DataSource dataSource;
	protected Connection connection;
	
	//XXX: make the dialect handle this!
	protected String nameQuoteLeft = "`";
	protected String nameQuoteRight = "`";
	protected String nameQuoteRightEscaped = null;
	protected String nameEscape = null;
	protected String nameEscapeEscaped = null;
	
	protected String stringQuoteLeft = "'";
	protected String stringQuoteRight = "'";
	protected String stringQuoteRightEscaped = "\\'";
	protected String stringEscape = "\\";
	protected String stringEscapeEscaped = "\\\\";
	
	protected String dialect;
	protected DialectHandler dialectHandler;

	private LeveledOutput logOut = new LogOutput();
	
	private boolean traceSQL = false;
	private int slowSQLThreashold = 0;
	private int explainSQLThreashold = 0;
	
	private BlockingJobQueue executor;
	private int bufferScale = 16;
	
	private int maxStatementSize = -1; //MySQL's default: a bit below 16M	bytes (!)
	
	protected int logLevel  = LOG_INFO;
	
	public DatabaseSchema(String prefix, DataSource connectionInfo, int queueCapacity) {
		this(prefix, connectionInfo, null, queueCapacity);
	}
	
	public DatabaseSchema(String prefix, Connection connection, int queueCapacity) {
		this(prefix, null, connection, queueCapacity);
	}
	
	private DatabaseSchema(String prefix, DataSource connectionInfo, Connection connection, int queueCapacity) {
		if (connection==null && connectionInfo==null) throw new IllegalArgumentException("either a connection or a connectionInfo must be provided");
			
		this.prefix = prefix;
		this.dataSource = connectionInfo;
		this.connection = connection;
		this.executor = queueCapacity<=0 ? null : new BlockingJobQueue(queueCapacity, true); //NOTE: daemon mode! join explicitly!
	}
	
	public void setNameQuotes(char left, char right, char esc) {
		nameQuoteLeft = left=='\0' ? null : ""+left;
		nameQuoteRight = right=='\0' ? null : ""+right;
		nameQuoteRightEscaped = esc=='\0' ? null : ""+esc+right;
		nameEscape = esc=='\0' ? null : ""+esc;
		nameEscapeEscaped = esc=='\0' ? null : ""+esc+esc;
	}
	
	public void setStringQuotes(char left, char right, char esc) {
		stringQuoteLeft = ""+left;
		stringQuoteRight = ""+right;
		stringQuoteRightEscaped = ""+esc+right; //XXX: could also be right+right
		stringEscape = ""+esc;
		stringEscapeEscaped = ""+esc+esc;
	}
	
	public String getDefaultTableAttributes() {
		return defaultTableAttributes;
	}

	public void setDefaultTableAttributes(String defaultTableAttributes) {
		this.defaultTableAttributes = defaultTableAttributes;
	}

	public DataSource getConnectionInfo() {
		return dataSource;
	}

	public DatabaseTable getTable(String name) {
		DatabaseTable t = tables.get(name);
		if (t==null) throw new IllegalArgumentException("no such table: "+name);
		return t;
	}
	
	public Inserter getInserter(String name) throws SQLException {
		return getTable(name).getInserter();
	}
	
	public void addTable(DatabaseTable table) {
		tables.put(table.getName(), table);
	}
	
	public void addField(String table, DatabaseField field) {
		getTable(table).addField(field);
	}

	public void addKey(String table, DatabaseKey key) {
		getTable(table).addKey(key);
	}

	public void addReference(String table, String field, String toTable, String toField, boolean required, KeyType keyType, String refAction) {
		DatabaseField f = getTable(toTable).getField(toField);
		if (!f.isUnique()) throw new IllegalArgumentException("field "+toField+" in table "+toTable+" is not unique");
		
		ReferenceField r = new ReferenceField(this, field, f.getType(), null, required, keyType, toTable, toField, refAction);
		getTable(table).addField(r);
	}

	public String getTablePrefix() {
		return prefix;
	}

	public String encodeValue(Object x) { 
		if (x==null) return "NULL";
		else if (x.getClass()==String.class) return quoteString((String)x);
		else if (x instanceof Date) return quoteString(x.toString()); //FIXME: undefine format...! 
		else if (x instanceof Number) {
			Number n = (Number)x;
			//TODO: handle NaN/INF
			return n.toString(); 
		}
		else {
			throw new IllegalArgumentException("unknown type for SQL literal: "+x.getClass());
		}
	}
	
	public String encodeSet(Object[] values) {
		return encodeSet(Arrays.asList(values));
	}

	public String encodeSet(int[] values) {
		StringBuilder s = new StringBuilder();
		s.append("(");
		
		boolean first = true;
		for (Object v: values) {
			if (first) first = false;
			else s.append(", ");
				
			s.append( encodeValue(v) );
		}
		
		s.append(")");
		return s.toString();
	}

	public String encodeSet(Collection<?> values) {
		StringBuilder s = new StringBuilder();
		s.append("(");
		
		boolean first = true;
		for (Object v: values) {
			if (first) first = false;
			else s.append(", ");
				
			s.append( encodeValue(v) );
		}
		
		s.append(")");
		return s.toString();
	}

	public String quoteString(String s) {
		return quoteString(s, stringQuoteLeft, stringQuoteRight, stringQuoteRightEscaped, stringEscape, stringEscapeEscaped);
	}
	
	private String quoteString(String s, String left, String right, String rightEscaped, String escape, String escapeEscaped) {
		if (rightEscaped!=null) {
			s = s.replace(escape, escapeEscaped);
			s = s.replace(right, rightEscaped);
		}
		else {
			if (s.indexOf(right)>=0) throw new IllegalArgumentException("encountered bad character: "+right);
		}
		
		return left+s+right; //TODO: use connection info
	}
	
	public String quoteName(String s) { 
		return quoteString(s, nameQuoteLeft, nameQuoteRight, nameQuoteRightEscaped, nameEscape, nameEscapeEscaped);
	}
	
	public String quoteQualifiedName(String s) { 
		String[] ss = s.split("\\.");
		for (int i=0; i<ss.length; i++) {
			ss[i] = quoteName(ss[i]);
		}
		
		return StringUtils.join(".", ss);
	}
	
	public void createTables(boolean opt) throws SQLException {
		Statement st = connection.createStatement();
		
		try {
			for (DatabaseTable table : tables.values()) {
				info("creating table "+table.getName());
				String sql = table.getCreateStatement(opt);
				trace("SQL: "+sql);
				st.executeUpdate(sql);
			}
		}
		finally {
			st.close();
		}
	}
	
	public void dropTables(boolean opt) throws SQLException {
		Statement st = connection.createStatement();
		
		try {
			for (DatabaseTable table : tables.values()) {
				info("dropping table "+table.getName());
				String sql = table.getDropStatement(opt);
				trace("SQL: "+sql);
				st.executeUpdate(sql);
			}
		}
		finally {
			st.close();
		}
	}
	
	public void createTable(String name, boolean opt) throws SQLException {
		DatabaseTable table = getTable(name);
		Statement st = connection.createStatement();
		
		try {
			info("creating table "+table.getSQLName());
			String sql = table.getCreateStatement(opt);
			trace("SQL: "+sql);
			st.executeUpdate(sql);
		}
		finally {
			st.close();
		}
	}
	
	public void dropTable(String name, boolean opt) throws SQLException {
		DatabaseTable table = getTable(name);
		Statement st = connection.createStatement();
		
		try {
			info("dropping table "+table.getSQLName());
			String sql = table.getDropStatement(opt);
			trace("SQL: "+sql);
			st.executeUpdate(sql);
		}
		finally {
			st.close();
		}
	}
	
	public void truncateTable(String table, boolean opt) throws SQLException {
		if (opt && !tableExists(table)) return;
		
		DatabaseTable t = getTable(table);
		Statement st = connection.createStatement();
		try {
			info("truncating table "+t.getName());
			String sql = t.getTruncateStatement();
			trace("SQL: "+sql);
			st.executeUpdate(sql);
		}
		finally {
			st.close();
		}
	}
	
	protected void forEachTable(String sql, boolean optional) throws SQLException {
		Statement st = connection.createStatement();
		
		try {
			for (DatabaseTable table : tables.values()) {
				if (optional && !tableExists(table.getName())) continue;
				
				String s = sql.replace("{table}", table.getSQLName());
				trace("SQL: "+s);
				st.executeUpdate(s);
			}
		}
		finally {
			st.close();
		}
	} 
	
	public void disableKeys(String table) throws SQLException {
		debug("disabling keys on "+table);
		execute("disableKeys", "ALTER TABLE "+quoteName(getSQLTableName(table))+" ENABLE KEYS;");		
	}
	
	public void enableKeys(String table) throws SQLException {
		debug("enabling keys on "+table);
		execute("enableKeys", "ALTER TABLE "+quoteName(getSQLTableName(table))+" DISABLE KEYS;");		
	}
	
	public void disableKeys() throws SQLException {
		info("disabling keys on all tables");
		forEachTable("ALTER TABLE {table} DISABLE KEYS;", true);		
	}
	
	public boolean tableExists(String t) throws SQLException {
		String n = getSQLTableName(t, false);
		n = n.replaceAll("_", "\\\\_");
		String sql = "SHOW TABLES LIKE "+this.quoteString(n);
		return this.executeSingleRowQuery("tabeExists", sql) != null;		
	}
	
	public void enableKeys() throws SQLException {
		info("enabling keys on all tables");
		forEachTable("ALTER TABLE {table} ENABLE KEYS;", true);		
	}
	
	public void optimizeIndexes() throws SQLException {
		info("optimizing indexes");
		forEachTable("ANALYZE TABLE {table};", true);		
	}
	
	public void optimizeTables() throws SQLException {
		info("optimizing tables");
		forEachTable("OPTIMIZE TABLE {table};", true);		
	}
	
	public String getDialect() {
		return dialect;
	}
	
	public void open() throws SQLException {
		if (connection==null) {
			if (dataSource==null) throw new IllegalStateException("no connection info given, can't open connection");
			connection = dataSource.getConnection();
			
			DatabaseMetaData meta = connection.getMetaData();
			dialect = meta.getDatabaseProductName();
			
			String q = meta.getIdentifierQuoteString();
			
			if (q==null || q.equals("") || q.equals(" ")) {
				setNameQuotes('\0', '\0', '\0');
			}
			else if (q.length()==1) {
				setNameQuotes(q.charAt(0), q.charAt(0), '\0');
			}
			else {				
				setNameQuotes(q.charAt(0), q.charAt(1), '\0');
			}
		}
	}
	
	public Connection getConnection() throws SQLException {
		if (connection==null) open();
		return connection;
	}
	
	public void joinExecutor(boolean shutdown) throws SQLException, InterruptedException {
		if (executor!=null) {
			if (executor.getQueue().size()>0 && !executor.isTerminated()) {
				if (executor.isShutdown()) {
					executor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
				}
				else {
					//FIXME: only works if no new tasks are added!!!!!!!!!!!
					try {
						executor.submit(new Runnable() {
							public void run() { /* dummy */ }
						}).get();
					} catch (ExecutionException e) {
						throw new Error("dummy method can't throw an exception!", e); 
					}
					
					executor.shutdown();
				}
			}
		}
	}
	
	public void finish() throws SQLException, InterruptedException {
		joinExecutor(true);
	}	
	
	public void flush() throws SQLException, InterruptedException {
		joinExecutor(false);
	}	
	
	public void close() throws SQLException {
		if (executor!=null) executor.shutdownNow(); 
		if (connection!=null) connection.close();
		connection = null;
	}

	public PreparedStatement prepareStatement(String sql) throws SQLException {
		PreparedStatement st = getConnection().prepareStatement(sql);
		st.setEscapeProcessing(false);
		return st;
	}

	public Statement createStatement() throws SQLException {
		Statement st = getConnection().createStatement();
		st.setEscapeProcessing(false);
		return st;
	}

	public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
		Statement st = getConnection().createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
		st.setEscapeProcessing(false);
		return st;
	}

	public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
		Statement st = getConnection().createStatement(resultSetType, resultSetConcurrency);
		st.setEscapeProcessing(false);
		return st;
	}
	
	protected static final Pattern selectPattern = Pattern.compile("(^|\\s+)select\\s+", Pattern.CASE_INSENSITIVE + Pattern.DOTALL);
	protected static final Pattern insertPattern = Pattern.compile("(^|\\s+)insert\\s+", Pattern.CASE_INSENSITIVE + Pattern.DOTALL);
	protected static final Pattern deletePattern = Pattern.compile("(^|\\s+)delete\\s+from\\s+(.*?\\s+using\\s+)?", Pattern.CASE_INSENSITIVE + Pattern.DOTALL);
	protected static final Pattern updatePattern = Pattern.compile("(^|\\s+)update\\s+", Pattern.CASE_INSENSITIVE + Pattern.DOTALL);
	protected static final Pattern valuePattern = Pattern.compile("(\\s+SET\\s+|\\(.*?\\)\\s+VALUES\\s*\\().*?(\\s+WHERE|$)", Pattern.CASE_INSENSITIVE + Pattern.DOTALL);
	
	//XXX: this is specific to MySQL!
	protected void traceExplain(String name, String sql) {
		if (logOut==null) return;
		
		String q = sql;
		Matcher m = selectPattern.matcher(q);
		if (!m.lookingAt()) {
			if (insertPattern.matcher(q).lookingAt()) {
				if (m.find()) {
					q = q.substring(m.start()); //XXX: crude hack. should usually work though
				}
				else {
					return;
				}
			}
			else if (deletePattern.matcher(q).lookingAt()) {
				q = deletePattern.matcher(q).replaceFirst("SELECT * FROM ");
			}
			else if (updatePattern.matcher(q).lookingAt()) {
				q = updatePattern.matcher(q).replaceFirst("SELECT * FROM ");
				q = valuePattern.matcher(q).replaceFirst("$2");
			}
			else {
				return;
			}
		}
		else {
			return;
		}
		
		q = "EXPLAIN " + q;
		List<? extends Map<String, ? extends Object>> rows;
		
		try {
			Statement st = createStatement();
			ResultSet rs = st.executeQuery(q);
			//ResultSetMetaData meta = rs.getMetaData();
			
			rows = slurpRows(rs);
			
			rs.close();
			st.close();
		}
		catch (SQLException ex) {
			warn("EXPLAIN failed for "+q+"\n\t\toriginal SQL: "+sql, ex);
			return;
		}
		
		//rough estimate
		long weight = 1; 
		for (Map<String, ? extends Object> r: rows) {
			long w = 1;
			Object o = r.get("rows");
			w *= o==null ? 1 : ((Number)o).longValue();
			
			if (r.get("key_len")!=null) {
				o = r.get("key_len");
				if (!(o instanceof Number)) o = Long.parseLong(o.toString());
				int kl = ((Number)o).intValue();
				w *= (kl/64)+1;
			}
			
			String key = (String)r.get("key");
			if (key==null || key.length()==0) {
				w *= 10; //XXX: instead, use ln in the opposite case?....
			}
			
			String extra = (String)r.get("extra");
			if (extra!=null) {
				if (extra.indexOf("using filesort")>=0) w*= 400;
				if (extra.indexOf("optimized away")>=0) w = 1;
			}
			
			weight *= w;
			
			/*
			String type = (String)r.get("type");
			if (type!=null && (type.equals("ALL") || type.equals("index"))) {
				weight *= w;
			}
			else {
				weight += w;
			}
			*/
		}

		//FIXME: threshold needs to be relative to table size
		if (weight > explainSQLThreashold) {
			logOut.println("**********************************************************");
			logOut.println("EXPLAIN yielded weight "+weight+":");
			logOut.println("----------------------------------------------------------");
			for (Map<String, ? extends Object> r: rows) {
				logOut.println(r);
			}
			logOut.println("----------------------------------------------------------");
			logOut.println("SQL: "+sql);
			logOut.println("QUERY: "+q);
			logOut.println("**********************************************************");
		}
	}

	/*
	public static void dumpData(ResultSet r) {
		// TODO Auto-generated method stub
		
	}

	public static void dumpData(List<Map<String, Object>> rows, ResultSetMetaData meta) {
		// TODO Auto-generated method stub
		
	}

	protected static String[] getRowFormats(ResultSetMetaData meta) throws SQLException {
		 int c = meta.getColumnCount();
		 String[] formats = new String[c];
		 for (int i = 1; i<=c; i++) {
			 int t = meta.getColumnType(i);
			 String f;
			 
			 if (t == Types.BIGINT || t == Types.BIT || t == Types.BOOLEAN || t == Types.DECIMAL
					|| t == Types.DOUBLE || t == Types.FLOAT || t == Types.INTEGER
					|| t == Types.NUMERIC || t == Types.REAL || t == Types.SMALLINT || t == Types.TINYINT ) {
				 f = "%0d"+(meta.getPrecision(i))+"."+(meta.getScale(i));
			 }
			 else {
				 f = "%s"+(meta.getColumnDisplaySize(i));
			 }
			 
			 formats[i-1] = f;
		 }
		 
		 return formats;
	}
	*/
	
	public int executeUpdate(String name, String sql) throws SQLException {
		Statement st = createStatement();
		
		if (explainSQLThreashold>0) traceExplain("executeUpdate("+name+")", sql); 
		else if (traceSQL) traceSQL("executeUpdate("+name+")", sql);
		
		sql = quoteComment(name)+" "+sql;
		long t = slowSQLThreashold <= 0 ? 0 : System.currentTimeMillis(); 
		
		try {
			int c = st.executeUpdate(sql);
			
			//FIXME: threshold needs to be relative to table size
			if (slowSQLThreashold>0 && (t = System.currentTimeMillis() - t) >  slowSQLThreashold*1000) {
				traceSQL("SLOW ("+t/1000+" sec): executeUpdate("+name+")", sql);
			}
			
			return c;
		}
		catch (SQLException e) {
			logBadSQL(name, sql);
			throw e;
		}
		finally {
			st.close();
		}
	}
	
	public ResultSet executeBigQuery(String name, String sql) throws SQLException {
		return executeQuery(name, sql, false);
	}

	public ResultSet executeQuery(String name, String sql) throws SQLException {
		return executeQuery(name, sql, true);
	}
	
	public ResultSet executeQuery(String name, String sql, boolean allowPrefetch) throws SQLException {
		Statement st = createStatement();
		if (!allowPrefetch) st.setFetchSize(Integer.MIN_VALUE);
		
		if (explainSQLThreashold>0) traceExplain("executeQuery("+name+")", sql);
		else if (traceSQL) traceSQL("executeQuery("+name+")", sql);

		sql = quoteComment(name)+" "+sql;
		long t = slowSQLThreashold <= 0 ? 0 : System.currentTimeMillis(); 
				
		try {
			ResultSet rs = st.executeQuery(sql);
			
			if (slowSQLThreashold>0 && (t = System.currentTimeMillis() - t) >  slowSQLThreashold*1000) {
				traceSQL("SLOW ("+t/1000+" sec): executeQuery("+name+")", sql);
			}
			
			return rs; //NOTE: statement stays open. would be nice to auto-close it together with the result set
		}
		catch (SQLException ex) {
			logBadSQL(name, sql);
			st.close();
			throw ex;
		}
		catch (RuntimeException ex) {
			st.close();
			throw ex;
		}
	}
	
	public boolean execute(String name, String sql) throws SQLException {
		Statement st = createStatement();
		
		if (explainSQLThreashold>0) traceExplain("execute("+name+")", sql);
		else if (traceSQL) traceSQL("execute("+name+")", sql);

		sql = quoteComment(name)+" "+sql;
		long t = slowSQLThreashold <= 0 ? 0 : System.currentTimeMillis(); 
		
		try {
			boolean ok = st.execute(sql);
			
			if (slowSQLThreashold>0 && (t = System.currentTimeMillis() - t) >  slowSQLThreashold*1000) {
				traceSQL("SLOW ("+t/1000+" sec): execute("+name+")", sql);
			}
			
			return ok;
		}
		catch (SQLException e) {
			logBadSQL(name, sql);
			throw e;
		}
		finally {
			st.close();
		}
	}

	public String quoteComment(String comment) {
		return "/* "+comment+" */";
	}
	
	public Object executeSingleValueQuery(String name, String sql) throws SQLException {
		ResultSet r = executeQuery(name, sql);
		
		try {
			if (!r.next()) return null;
			return r.getObject(1);
		}
		finally {
			r.close();
		}
	}
	
	public static List<? extends Map<String, ? extends Object>> slurpRows(ResultSet r) throws SQLException {
		ResultSetMetaData meta = r.getMetaData();
		List<Map<String, Object>> rows = new ArrayList<Map<String, Object>>();
		while (r.next()) {
			rows.add( rowMap(r, meta) );
		}
		
		return rows;
	}
	
	public static Map<? extends Object, ? extends Map<String, ? extends Object>> slurpRows(ResultSet r, String keyColumn) throws SQLException {
		ResultSetMetaData meta = r.getMetaData();
		Map<Object, Map<String, Object>> rows = new HashMap<Object, Map<String, Object>>();
		while (r.next()) {
			Map<String, Object> row = rowMap(r, meta);
			Object key = row.get(keyColumn);
			rows.put(key, row );
		}
		
		return rows;
	}
	
	public static List<? extends Object> slurpList(ResultSet r, int valueColumn) throws SQLException {
		List<Object> rows = new ArrayList<Object>();
		while (r.next()) {
			Object v = r.getObject(valueColumn);
			rows.add(v);
		}
		
		return rows;
	}
	
	public static List<? extends Object> slurpList(ResultSet r, String valueColumn) throws SQLException {
		List<Object> rows = new ArrayList<Object>();
		while (r.next()) {
			Object v = r.getObject(valueColumn);
			rows.add(v);
		}
		
		return rows;
	}
	
	public static Map<String, Object> rowMap(ResultSet r) throws SQLException {
		ResultSetMetaData meta = r.getMetaData();
		return rowMap(r, meta);
	}
	
	protected static Map<String, Object> rowMap(ResultSet r, ResultSetMetaData meta) throws SQLException {
		int c = meta.getColumnCount();
		Map<String, Object> row = new HashMap<String, Object>(c);
		
		for (int i=1; i<=c; i++) {
			String n = meta.getColumnLabel(i); //NOTE: with MySQL connector/j 5.x, getColumnName does NOT give the expected result!
			Object v = r.getObject(i);
			
			row.put(n, v);
		}
		
		return row;
	}

	public Object fetchValue(String table, String field, Object value, String vfield) throws SQLException {
		DatabaseTable t = getTable(table);
		if (t!=null) table = t.getSQLName();
		else table = quoteName(table);
		
		String sql = "SELECT "+quoteName(vfield)+" FROM "+table+" WHERE "+quoteName(field)+(value == null ? "IS NULL" : " = "+encodeValue(value));
		return executeSingleValueQuery("fetchValue", sql);
	}

	public Map<String, Object> fetchRecord(String table, String field, Object value) throws SQLException {
		DatabaseTable t = getTable(table);
		if (t!=null) table = t.getSQLName();
		else table = quoteName(table);
		
		String sql = "SELECT * FROM "+table+" WHERE "+quoteName(field)+(value == null ? "IS NULL" : " = "+encodeValue(value));
		return executeSingleRowQuery("fetchRecord", sql);
	}
	
	public <T>List<T> executeSingleColumnQuery(String name, String sql) throws SQLException {
		ResultSet r = executeQuery(name, sql);
		List<T> a = new ArrayList<T>();
		
		try {
			while (r.next()) {
				T v = (T)r.getObject(1);
				a.add(v);
			}
		}
		finally {
			r.close();
		}
		
		return a;
	}
	
	public <K,T>Map<K,T> executeSimpleMapQuery(String name, String sql) throws SQLException {
		ResultSet r = executeQuery(name, sql);
		Map<K,T> m = new HashMap<K,T>();
		
		try {
			while (r.next()) {
				K k = (K)r.getObject(1);
				T v = (T)r.getObject(2);
				
				m.put(k, v);
			}
		}
		finally {
			r.close();
		}
		
		return m;
	}
	
	public Map<String, Object> executeSingleRowQuery(String name, String sql) throws SQLException {
		ResultSet r = executeQuery(name, sql);
		
		try {
			if (!r.next()) return null;
			return rowMap(r);
		}
		finally {
			r.close();
		}
	}

	public void logBadSQL(String method, String sql) {
		if (logOut == null) return;
		if (sql.length()>1024) sql = sql.substring(0, 1024) + "...";
		logOut.info("BAD SQL: "+method+": "+sql);
	}

	protected void traceSQL(String method, String sql) {
		if (logOut == null) return;
		if (sql.length()>1024) sql = sql.substring(0, 1024) + "...";
		logOut.info("TRACE SQL: "+method+": "+sql);
	}
	
	public void setLogLevel(int level) {
		this.logLevel = level;
		
		if (logOut == null) return;
		if (!(logOut instanceof LogOutput)) return;
		
		((LogOutput)logOut).setLogLevel(level);
	}
	
	public int getLogLevel() {
		return logLevel;
	}

	public void setLogOutput(Output out) {
		if (!(out instanceof LeveledOutput)) out = new LogOutput(out);
		this.logOut = (LeveledOutput)out;
	}
	
	public LeveledOutput getLogOutput() {
		return this.logOut;
	}

	public void debug(String msg) {
		logOut.debug(msg);
	}

	public void error(String msg, Throwable ex) {
		logOut.error(msg, ex);
	}

	public void info(String msg) {
		logOut.info(msg);
	}

	public void trace(String msg) {
		logOut.trace(msg);
	}

	public void warn(String msg, Throwable ex) {
		logOut.warn(msg, ex);
	}

	public void warn(String msg) {
		logOut.warn(msg);
	}

	public Future<Object> executeLater(DatabaseTask task) throws SQLException {
		if (executor!=null) {
			trace("executeLater: submitting task; queue status is "+executor.getQueue().size()+"/"+executor.getQueueCapacity());
			return executor.submit(task);
		}
		else {
			return new FinishedFuture<Object>(task.call(), null);
		}
	}
	
	public Object executeSynced(DatabaseTask task) throws SQLException {
		try {
			Future<Object> future = this.executeLater(task);
			trace("executeSynced: waiting for task");
			Object r = future.get();
			trace("executeSynced: task complete");
			return r;
		} catch (CancellationException e) {
			throw (SQLException)new SQLException("execution cancelled").initCause(e);
		} catch (InterruptedException e) {
			throw (SQLException)new SQLException("execution interrupted").initCause(e);
		} catch (ExecutionException e) {
			Throwable exx = e.getCause();
			if (exx instanceof SQLException) throw (SQLException)exx;
			if (exx instanceof Error) throw (Error)exx;
			if (exx instanceof RuntimeException) throw (SQLException)exx;
			throw (SQLException)new SQLException().initCause(exx);
		}
	}
	
	public int getTableSize(String table) throws SQLException {
		DatabaseTable t = getTable(table);
		String sql = "SELECT COUNT(*) FROM "+this.quoteName(t.getSQLName());
		int sz = ((Number)executeSingleValueQuery("getTableSize", sql)).intValue();
		return sz;
	}

	public Map<String, Integer> getGroupSizes(String table, String groupby) throws SQLException {
		DatabaseTable t = getTable(table);

		Map<String, Integer> groups = new HashMap<String, Integer>();
		Statement st = createStatement();
		try {
			String sql = "SELECT "+groupby+", count(*) FROM "+this.quoteName(t.getSQLName())+" GROUP BY "+groupby+" ORDER BY "+groupby;
			ResultSet r = st.executeQuery(sql);
			try {
				while (r.next()) {
					String g = r.getString(1);
					Integer n = r.getInt(2);
					
					groups.put(g, n);
				}
			}
			finally {
				r.close();
			}
			
		}
		finally {
			st.close();
		}
		
		return groups;
	}

	public int getBufferScale() {
		return bufferScale;
	}

	public void setBufferScale(int bufferScale) {
		this.bufferScale = bufferScale;
	}

	public int getExplainSQLThreashold() {
		return explainSQLThreashold;
	}

	public void setExplainSQLThreashold(int explainSQLThreashold) {
		this.explainSQLThreashold = explainSQLThreashold;
	}

	public int getSlowSQLThreashold() {
		return slowSQLThreashold;
	}

	public void setSlowSQLThreashold(int slowSQLThreashold) {
		this.slowSQLThreashold = slowSQLThreashold;
	}

	public boolean isTraceSQL() {
		return traceSQL;
	}

	public void setTraceSQL(boolean traceSQL) {
		this.traceSQL = traceSQL;
	}
	
	public String echo(String text, String charset) throws SQLException {
		charset = charset.replaceAll("[-_]", "").toLowerCase();
		
		if (charset.equalsIgnoreCase("binary")) execute("echo: create", "CREATE TEMPORARY TABLE echo_test ( text varbinary(256) ) ");
		else execute("echo: create", "CREATE TEMPORARY TABLE echo_test ( text varchar(256) ) CHARSET = "+charset);
		
		executeUpdate("echo: insert", "INSERT INTO echo_test SET text = "+encodeValue(text));
		String s = (String)executeSingleValueQuery("echo: select", "SELECT text FROM echo_test LIMIT 1");
		execute("echo: drop", "DROP TEMPORARY TABLE echo_test");
		return s;
	}

	public Iterable<DatabaseTable> getTables() {
		return tables.values();
	}

	public String getSQLTableName(String name) {
		return getSQLTableName(name, false);
	}

	public String getSQLTableName(String name, boolean unsafe) {
		if (unsafe) return prefix + name;
		else return getTable(name).getSQLName();
	}

	public String mangleType(String type) {
		if (dialectHandler!=null) type = dialectHandler.mangleType(type);
		return type;
	}

	public DialectHandler getDialectHandler() {
		return dialectHandler;
	}

	public void setDialectHandler(DialectHandler dialectHandler) {
		this.dialectHandler = dialectHandler;
	}
	
	@SuppressWarnings("unchecked")
	public void setDialectHandler(String name) throws InstantiationException, ClassNotFoundException {
		Class c = getClass().getClassLoader().loadClass(name);
		setDialectHandler(c);
	}
	
	public void setDialectHandler(Class<DialectHandler> handlerClass) throws InstantiationException {
		Field f = null;
		int m;
		DialectHandler h = null;
		
		if (!DialectHandler.class.isAssignableFrom(handlerClass)) {
			throw new IllegalArgumentException("not compatible with DialectHandler: "+handlerClass);
		}
		
		try {
			f= handlerClass.getField("instance");
			m = f.getModifiers();
			if (!Modifier.isPublic(m) || !Modifier.isStatic(m) 
					|| !DialectHandler.class.isAssignableFrom(f.getType())) 
						f = null;
		} 
		catch (SecurityException e) { /* ignore */ } 
		catch (NoSuchFieldException e) { /* ignore */ }
		
		try {
			if (f==null) {
				f= handlerClass.getField("singleton");
				m = f.getModifiers();
				if (!Modifier.isPublic(m) || !Modifier.isStatic(m) 
						|| !DialectHandler.class.isAssignableFrom(f.getType())) 
							f = null;
			}
		} 
		catch (SecurityException e) { /* ignore */ } 
		catch (NoSuchFieldException e) { /* ignore */ }
		
		if (f!=null) {
			try {
				h = (DialectHandler)f.get(null);
			} 
			catch (IllegalAccessException e) { /* ignore */ }
		}
		
		if (h==null) {
			try {
				h = handlerClass.newInstance();
			} catch (IllegalAccessException e) {
				throw (InstantiationException)new InstantiationException().initCause(e);
			}
		}
			
		setDialectHandler(h);
	}
	
	public static String qualifyName(String q, String name) {
		if (q==null || name==null) return name;
		else if (name.indexOf('.') >=0) return name;
		else return q + "." + name;
	}
	
	public void setGroupConcatMaxLen(int len) throws SQLException {
		setSessionVariable("group_concat_max_len", len); 
	}

	public void setSqlMode(String mode) throws SQLException {
		setSessionVariable("sql_mode", mode); //TODO: dialect
	}

	public void setSessionVariable(String name, Object value) throws SQLException {
		executeUpdate("setSessionVariable", "set session "+name+" = " + encodeValue(value)); 
	}

	public int getMaxStatementSize() throws SQLException {
		if (maxStatementSize<=0) {
			open();
			String sql = "select @@max_allowed_packet";
			
			Object v = executeSingleValueQuery("getMaxStatementSize", sql);
			if (v==null) maxStatementSize = 16*1024*1024 - 1024;
			else maxStatementSize = DatabaseUtil.asInt(v);
		}
		
		return maxStatementSize;
	}

	public void setMaxStatementSize(int maxStatementSize) {
		if (maxStatementSize>=0 && maxStatementSize<32) throw new IllegalArgumentException("limit too low: "+maxStatementSize);
		
		this.maxStatementSize = maxStatementSize;
	}
	
	public ChunkingInfo getChunkingInfo(ChunkedQuery query, int chunkSize) throws SQLException {
		if (chunkSize<=0) {
			return null;
		}

		DatabaseTable chunkTable = query.getChunkTable();
		String chunkField = query.getChunkField();

		String counterCond = "";
		//String counterCond = where == null ? "" : " WHERE " +where ; //XXX: grrr.... that doesnt work :( 
		String fname = chunkField.replaceAll("^\\w+\\.", ""); //XXX: dirty hack!
		
		Map<String, Object> info = executeSingleRowQuery("executeChunked#count", "select count(*) as c, max("+fname+") as mx, min("+fname+") as mn from "+chunkTable.getSQLName()+counterCond);
		int numIds = info == null || info.get("c")  == null ? 0 : DatabaseUtil.asInt(info.get("c"));
		int maxId =  info == null || info.get("mx") == null ? 0 : DatabaseUtil.asInt(info.get("mx"));
		int minId =  info == null || info.get("mn") == null ? 0 : DatabaseUtil.asInt(info.get("mn"));
		int diffId = (maxId - minId) +1;

		if (diffId<0) throw new RuntimeException("oops! id-diff must not be negative! is: "+diffId+" (minid: "+minId+"; maxid: "+maxId+"; numids: "+numIds+")");

		if (numIds==0) return null; //nothing to do anyway
		long chunkStep = ((chunkSize * (long)diffId) / numIds) +1;
		
		if (chunkStep<=0) throw new RuntimeException("oops! chunkStep must not be null or negative! is: "+chunkStep+" (minid: "+minId+"; maxid: "+maxId+"; numids: "+numIds+")");
		if (chunkStep>diffId) chunkStep = diffId;

		return new ChunkingInfo(minId, maxId, (int)chunkStep);
	}

	public int executeChunkedUpdate(ChunkedQuery query, int chunkSize, Agenda agenda) throws SQLException, PersistenceException {
			DatabaseTable chunkTable = query.getChunkTable();
			
			try {
				ChunkingInfo info = getChunkingInfo(query, chunkSize);

				if (info==null) {
					return query.executeUpdate(0, 0, Integer.MAX_VALUE); //XXX: use Long.MAX_VALUE ?? TEST!
				}
				
				long first = info.minId;
				long end = first + info.chunkStep;
				int i = 0;
				int c = 0;
				
				//XXX: restore minId/maxId from agenda?...
				
				while (first <= info.maxId) {
					i++;
					
					if (agenda==null || agenda.beginTask(query.getContext()+".executeChunked", query.getName()+"#chunk"+i, "field=\""+chunkTable.getName()+"."+query.getChunkField()+"\",first=L"+first+",end=L"+end+",maxId_=L"+info.maxId+",minId_=L"+info.minId)) {
						int n = query.executeUpdate(i, first, end);
						if (agenda!=null) agenda.endTask(query.getContext()+".executeChunked", query.getName()+"#chunk"+i, n+" entries");
						
						c+= n;
					}
					
					first += info.chunkStep;
					end += info.chunkStep;
				}
				
				return c;
			} catch (SQLException e) {
				throw e;
			} catch (PersistenceException e) {
				throw e;
			} catch (Exception e) {
				throw new PersistenceException(e);
			}
	}
	
	public int executeChunkedQuery(ChunkedQuery query, int chunkSize, Agenda agenda, Processor<ResultSet> processor) throws SQLException, PersistenceException {
		DatabaseTable chunkTable = query.getChunkTable();
		
		try {
			ChunkingInfo info = getChunkingInfo(query, chunkSize);
			if (info==null) info = new ChunkingInfo(0, Integer.MAX_VALUE, Integer.MAX_VALUE);

			long first = info.minId;
			long end = first + info.chunkStep;
			int i = 0;
			int c = 0;
			
			//XXX: restore minId/maxId from agenda?...
			
			while (first <= info.maxId) {
				i++;
				
				if (agenda==null || agenda.beginTask(query.getContext()+".executeChunked", query.getName()+"#chunk"+i, "field=\""+chunkTable.getName()+"."+query.getChunkField()+"\",first=L"+first+",end=L"+end+",maxId_=L"+info.maxId+",minId_=L"+info.minId)) {
					int n = 0;
					
					ResultSet rs = query.executeQuery(i, first, end);
					try {
						processor.process(rs);
						n += rs.getRow(); //FIXME: always 0 ?!
						if (agenda!=null) agenda.endTask(query.getContext()+".executeChunked", query.getName()+"#chunk"+i, n+" entries");
					}
					finally {
						rs.close();
					}
					
					c+= n;
				}
				
				first += info.chunkStep;
				end += info.chunkStep;
			}

			return c;
		} catch (SQLException e) {
			throw e;
		} catch (PersistenceException e) {
			throw e;
		} catch (Exception e) {
			throw new PersistenceException(e);
		}
		
	}

}
