package de.brightbyte.db;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DatabaseField {
	protected DatabaseAccess access;
	protected String name;
	protected String type;
	protected boolean required;
	protected String attributes;
	protected String sql;
	protected KeyType keyType;
	private int size;
	
	protected static final Pattern twoArgTypePattern = Pattern.compile("^\\s*\\w+\\s*\\(\\s*(\\d+)\\s*,\\s*(\\d+)\\s*\\)\\s*$");
	protected static final Pattern oneArgTypePattern = Pattern.compile("^\\s*\\w+\\s*\\(\\s*(\\d+)\\s*\\)\\s*$");
	protected static final Pattern wideTypePattern = Pattern.compile("^\\s*(\\w*CHAR|\\w*BINARY|\\w*TEXT|CLOB|BLOB)\\s*$", Pattern.CASE_INSENSITIVE);
	
	public DatabaseField(DatabaseAccess access, String name, String type, String attributes, boolean required, KeyType keyType) {
		super();
		
		if (attributes==null) attributes = "";
		
		if (required) attributes += " NOT NULL";
		else {
			if (attributes.toUpperCase().indexOf("DEFAULT")<0) attributes += " DEFAULT NULL";
		}
		
		this.name = name;
		this.type = type;
		this.attributes = attributes;
		this.keyType = keyType;
		this.required = required || (keyType!=null && keyType.isUnique() && attributes.toUpperCase().indexOf("AUTO")<0);
		
		StringBuilder s = new StringBuilder();
		s.append(access.quoteName(name));
		s.append(' ');
		s.append( access.mangleType(type) );
		s.append(' ');
		s.append(attributes);
		sql = s.toString();
		
		Matcher m; ;
		if ((m = twoArgTypePattern.matcher(type)).matches()) {
			size = Integer.parseInt(m.group(1)) + Integer.parseInt(m.group(2)) +1; 
		}
		else if ((m = oneArgTypePattern.matcher(type)).matches()) {
			size = Integer.parseInt(m.group(1)); 
		}
		else if ((m = wideTypePattern.matcher(type)).matches()) {
			size = 64; 
		}
		else {
			size = 8; 
		}
	}

	public String getAttributes() {
		return attributes;
	}

	public String getName() {
		return name;
	}

	public String getType() {
		return type;
	}
	
	public String getSQL() {
		return sql;
	}
	
	public KeyType getKeyType() {
		return keyType;
	}
	
	public boolean isRequired() {
		return required;
	}
	
	
	@Override
	public String toString() {
		String s = getSQL();
		if (keyType!=null) s += " [ "+keyType.getName()+" ]";
		return s;
	}

	public boolean isUnique() {
		return keyType.isUnique();
	}

	public int getSize() {
		return size;
	}

	public void setSize(int size) {
		this.size = size;
	}
	
}
