package de.brightbyte.db;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import de.brightbyte.data.cursor.CursorIterator;
import de.brightbyte.data.cursor.DataCursor;
import de.brightbyte.data.cursor.DataSet;
import de.brightbyte.util.PersistenceException;
import de.brightbyte.util.UncheckedPersistenceException;

public abstract class DatabaseDataSet<T> implements DataSet<T> {
	public interface Factory<T> {
		public T newInstance(ResultSet row) throws Exception;
	}
	
	public static class Cursor<T> implements DataCursor<T>{
		protected ResultSet resultSet;
		protected Factory<T> factory;
		protected boolean more = true;
		protected int counter = 0;
		
		public Cursor(ResultSet resultSet, Factory<T> factory) {
			this(factory);
			this.resultSet = resultSet;
		}

		protected Cursor(Factory<T> factory) {
			this.factory = factory;
		}

		@Override
		protected void finalize() {
			close();
		}

		public void close() {
			try {
				if (resultSet!=null) resultSet.close();
				more = false;
			} catch (SQLException e) {
				//ignore. who cares?
			}
		}

		protected boolean fetchNext() throws Exception {
			return resultSet.next();
		}
		
		public T next() throws PersistenceException {
			try {
				if (!more) return null;
				more = fetchNext();
				return more ? newInstance(resultSet) : null;
			} catch (PersistenceException e) {
				throw e;
			} catch (Exception e) {
				throw new PersistenceException(e);
			}
		}
		
		
		public int getItemCount() {
			return counter ;
		}
		
		protected T newInstance(ResultSet row) throws Exception {
			T item = factory.newInstance(row);
			counter ++;
			return item;
		}
		
	}
	
	protected DatabaseAccess access;
	protected Factory<T> factory;

	public DatabaseDataSet(DatabaseAccess access, Factory<T> factory) {
		this.access = access;
		this.factory = factory;
	}

	public Iterator<T> iterator() throws UncheckedPersistenceException {
		try {
			return new CursorIterator<T>(cursor());
		} catch (PersistenceException e) {
			throw new UncheckedPersistenceException(e); //XXX: fugly hack :(
		}
	}
	
	protected abstract ResultSet execute() throws SQLException;

	public DataCursor<T> cursor() throws PersistenceException {
		try {
			ResultSet rs = execute();
			return new Cursor<T>(rs, factory);
		} catch (SQLException e) {
			throw new PersistenceException(e); 
		}
	}

	public List<T> load() {
		ArrayList<T> r = new ArrayList<T>();
		for (T x: this) r.add(x);
		return r;
	}

	
}