package de.brightbyte.db;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.sql.DataSource;

import de.brightbyte.abstraction.Disposer;
import de.brightbyte.abstraction.Instantiator;
import de.brightbyte.util.SystemUtils;

public class DatabaseConnectionInfo implements DataSource {
	
	public static class DriverInfo {
		protected String className;
		protected ClassLoader loader;

		public DriverInfo(String className, ClassLoader loader) {
			this(className, loader, null);
		}

		public DriverInfo(String className, ClassLoader loader, URL jar) {
			if (className==null) throw new NullPointerException();
			
			this.className = className;
			
			if (jar==null) this.loader = loader;
			else this.loader = new URLClassLoader( new URL[] { jar }, loader == null ? ClassLoader.getSystemClassLoader() : loader );
		}

		public String getClassName() {
			return className;
		}
		
		public ClassLoader getClassLoader() {
			return loader;
		}
		
		public void load() throws ClassNotFoundException, InstantiationException, IllegalAccessException {
			if (loader!=null) loader.loadClass(className);
			else Class.forName(className);
		}
	}
	
	protected static class ConnectionInstantiator implements Instantiator<Connection>, Disposer<Connection> {

		protected DataSource info;
		
		public ConnectionInstantiator(DataSource info) {
			this.info = info;
		}

		public Connection newInstance() throws InstantiationException {
			try {
				return info.getConnection();
			} catch (SQLException e) {
				throw (InstantiationException)new InstantiationException("failed to open connection").initCause(e);
			}
		}

		public void dispose(Connection conn) {
			try {
				conn.close();
			} catch (SQLException e) {
				//TODO: log?!
				e.printStackTrace();
			}
		}
	}
	
	protected static Map<String, DriverInfo> drivers = new HashMap<String, DriverInfo>();
	
	protected DriverInfo driver;
	protected String type;
	protected String url;
	protected String user;
	protected String password;

	public DatabaseConnectionInfo(File info) throws IOException {
		this( SystemUtils.loadProperties(info, null) );
	}
	
	public DatabaseConnectionInfo(URL info) throws IOException {
		this( SystemUtils.loadProperties(info, null) );
	}

	
	public DatabaseConnectionInfo(Properties info) {
		this( info.getProperty("driver"),
				info.getProperty("type"),
				info.getProperty("url"),
				info.getProperty("user"),
				info.getProperty("password")
			);
	}
	
	public DatabaseConnectionInfo(String url, String user, String password) {
		this(null, url, user, password);
	}
	
	public DatabaseConnectionInfo(String type, String url, String user, String password) {
		this((DriverInfo)null, type, url, user, password );
	}
	
	public DatabaseConnectionInfo(String driver, String type, String url, String user, String password) {
		this(driver == null ? null : new DriverInfo(driver, null), type, url, user, password );
	}

	public DatabaseConnectionInfo(DriverInfo driver, String type, String url, String user, String password) {
		if (url==null) throw new NullPointerException("database url required");
		if (!url.matches("^jdbc:([-a-zA-Z0-1._]+):.*")) throw new IllegalArgumentException("bad database url: "+url);
		
		this.type = type == null ? url.replaceAll("^jdbc:([-a-zA-Z0-1._]+):.*$", "$1") : type;
		this.driver = driver == null ? getDriverFor(this.type) : driver;
		this.url = url;
		this.user = user;
		this.password = password;
	}

	public String getPassword() {
		return password;
	}

	public String getType() {
		return type;
	}

	public String getUrl() {
		return url;
	}

	public String getUser() {
		return user;
	}

	public String getDatabaseType() {
		return type;
	}
	
	public DriverInfo getDriver() {
		return driver;
	}
	
	public Instantiator instantiator() {
		return new ConnectionInstantiator(this);
	}
	
	public Connection getConnection() throws SQLException {
		return getConnection(user, password);
	}
	
	public Connection getConnection(String username, String password) throws SQLException {
		if (driver!=null) {
			try {
				driver.load();
			} catch (ClassNotFoundException e) {
				throw (SQLException)new SQLException("driver class not found: "+driver).initCause(e);
			} catch (InstantiationException e) {
				throw (SQLException)new SQLException("failed to instantiate driver class: "+driver).initCause(e);
			} catch (IllegalAccessException e) {
				throw (SQLException)new SQLException("failed to instantiate driver class: "+driver).initCause(e);
			}
		}
		
		Connection c = DriverManager.getConnection(url, user, password);
		return c;
	}
	
	public static void registerDrivers(Properties drivers, ClassLoader loader, URL jarBase) throws MalformedURLException {
		Enumeration en = drivers.propertyNames();
		while (en.hasMoreElements()) {
			String type = (String)en.nextElement();
			String driver = drivers.getProperty(type);
			
			String j = null;
			int idx = driver.indexOf('@');
			if (idx>=0) {
				j = driver.substring(idx+1);
				driver = driver.substring(0, idx);
			}
			
			URL jar = j==null ? null : jarBase == null ? new URL(j) : new URL(jarBase, j);
			registerDriverFor(type, driver, loader, jar); 
		}
	}
	
	public static void registerDriverFor(String type, String className, ClassLoader loader, URL jar) {
		registerDriverFor(type, new DriverInfo(className, loader, jar));
	}
	
	public static void registerDriverFor(String type, DriverInfo driver) {
		drivers.put(type, driver);
	}
	
	public static DriverInfo getDriverFor(String type) {
		return drivers.get(type.toLowerCase().trim());
	}

	public PrintWriter getLogWriter() throws SQLException {
		return null;
	}

	public void setLogWriter(PrintWriter out) throws SQLException {
		throw new UnsupportedOperationException();
	}

	public int getLoginTimeout() throws SQLException {
		return 0;
	}

	public void setLoginTimeout(int seconds) throws SQLException {
		throw new UnsupportedOperationException();
	}

}
