package de.brightbyte.db;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import de.brightbyte.application.Agenda;
import de.brightbyte.util.PersistenceException;

public class DatabaseAgendaPersistor implements Agenda.Persistor {
	
	private static final SimpleDateFormat timestampFormatter = new SimpleDateFormat( "yyyyMMddHHmmss" );
	
	static {
		timestampFormatter.setTimeZone(TimeZone.getTimeZone("GMT"));
	}
	
	protected DatabaseAccess access;
	protected DatabaseTable table;
	protected PreparedStatement insert;
	protected PreparedStatement update;
	
	protected int currentId = 0;
	
	public DatabaseAgendaPersistor(DatabaseTable table) throws SQLException {
		this.access = table.getAccess();
		this.table = table;
		
		insert = access.prepareStatement("insert into "+table.getSQLName()+" set "
				+ "level = ?, "
				+ "context = ?, "
				+ "task = ?, "
				+ "state = ?, "
				+ "timestamp = ?, "
				+ "complex = ?, "
				+ "parameters = ? "
				);
		
		update = access.prepareStatement("update "+table.getSQLName()+" set "
				+ "end = ?, "
				+ "state = ?, "
				+ "duration = ?, "
				+ "result = ? "
				+ "where start = ?"
				);
	}
	
	public static EntityTable makeTableSpec(DatabaseSchema access, String name) {
		EntityTable logTable;
		
		logTable = new EntityTable(access, name, access.getDefaultTableAttributes());
		logTable.addField( new DatabaseField(access, "level", "INT(3)", null, true, KeyType.INDEX ) );
		logTable.addField( new DatabaseField(access, "start", "INT(10)", "AUTO_INCREMENT", true, KeyType.PRIMARY ) );
		logTable.addField( new DatabaseField(access, "end", "INT(10)", "DEFAULT NULL", false, null ) );
		logTable.addField( new DatabaseField(access, "context", "VARCHAR(64)", null, false, KeyType.INDEX ) ); 
		logTable.addField( new DatabaseField(access, "task", "VARCHAR(64)", null, true, KeyType.INDEX ) ); 
		logTable.addField( new DatabaseField(access, "state", "INT(4)", null, true, null ) ); 
		logTable.addField( new DatabaseField(access, "timestamp", "CHAR(14)", null, true, KeyType.INDEX ) ); //TODO: which type in which db?
		logTable.addField( new DatabaseField(access, "duration", "INT(8)", "DEFAULT NULL", false, null ) ); 
		logTable.addField( new DatabaseField(access, "complex", "TINYINT(1)", null, true, null ) );  //TODO: which type in which db?
		logTable.addField( new DatabaseField(access, "parameters", "VARBINARY(255)", "DEFAULT NULL", false, null ) );
		logTable.addField( new DatabaseField(access, "result", "VARCHAR(255)", "DEFAULT NULL", false, null ) );
		logTable.setInserterFactory(StatementBasedInserter.factory); //NOTE: no buffering!
		access.addTable(logTable);

		return logTable;
	}

	protected static boolean asBoolean(Object n) {
		if (n instanceof Boolean) return ((Boolean)n).booleanValue();
		if (n==null) return false;
		return ((Number)n).doubleValue() != 0;
	}
	
	protected static int asInteger(Number n) {
		if (n==null) return -1;
		return n.intValue();
	}
	
	protected static long asLong(Number n) {
		if (n==null) return -1;
		return n.longValue();
	}
	
	public Agenda.Record getLastRootRecord() throws PersistenceException {
		return getLastRecord("level = 0");
	}
	
	protected Agenda.Record getLastRecord(String where) throws PersistenceException {
		try {
			String w = "";
			if (where!=null) w = " WHERE "+where;
			
			String sql = "SELECT * FROM "+table.getSQLName()+" "+w+" ORDER BY start DESC LIMIT 1";
			Map<String, Object> r = access.executeSingleRowQuery("last agenda record", sql);
			if (r==null) return null;
			
			return createRecord(r);
		} catch (SQLException e) {
			throw new PersistenceException(e);
		} 
	}
	
	public static Agenda.Record getRecord(ResultSet r) throws PersistenceException {
		try {
			Map<String, Object> m = DatabaseSchema.rowMap(r);
			return createRecord(m);
		} catch (SQLException e) {
			throw new PersistenceException(e);
		} 
	}
	
	public static Agenda.Record createRecord(Map<String, Object> r) throws PersistenceException {
		try {
			Agenda.Record rec = new Agenda.Record(
					asInteger((Number)r.get("level")), 
					asInteger((Number)r.get("start")), 
					asInteger((Number)r.get("end")), 
					timestampFormatter.parse((String)r.get("timestamp")).getTime(), 
					asLong((Number)r.get("duration")), 
					(String)r.get("context"), 
					(String)r.get("task"), 
					Agenda.State.valueOf((Integer)r.get("state")), 
					asBoolean(r.get("complex")), 
					Agenda.decodeParameters((String)r.get("properties"), false), 
					(String)r.get("result") 
			);
			
			return rec;
		} catch (ParseException e) {
			throw new PersistenceException(e);
		}
	}
	
	public Agenda.Record logStart(int level, String context, String task, Map<String, Object> parameters, boolean complex) throws PersistenceException {
		try {
/*
				+ "level = ?, "
				+ "task = ?, "
				+ "state = ?, "
				+ "timestamp = ?, "
				+ "complex = ?, "
				+ "parameters = ? "
 */			
			Date now = new Date();
			Agenda.State state = Agenda.State.STARTED;
			
			int idx = 1;
			insert.setInt(idx++, level);
			insert.setObject(idx++, context);
			insert.setObject(idx++, task);
			insert.setInt(idx++, state.ordinal());
			insert.setObject(idx++, timestampFormatter.format(now));
			insert.setInt(idx++, complex ? 1 : 0);
			insert.setObject(idx++, Agenda.encodeParameters(parameters));
			
			insert.executeUpdate();
			ResultSet keys = insert.getGeneratedKeys();
			
			int id;
			
			if (keys.next()) {
				id = keys.getInt(1);
			}
			else {
				throw new PersistenceException("failed to fetch automatic key value!");
			}
			
			return new Agenda.Record(level, id, id, now.getTime(), 0, context, task, state, complex, parameters, null);
		} catch (SQLException e) {
			throw new PersistenceException(e);
		}
	}

	public void logTerminated(int start, int end, long duration, Agenda.State state, String result) throws PersistenceException {
/*
		+ "end = ?, "
		+ "state = ?, "
		+ "duration = ? "
		+ "where start = ?"
 */		
		
		try {
			update.setInt(1, end);
			update.setInt(2, state.ordinal());
			update.setLong(3, duration);
			update.setString(4, result);
			update.setInt(5, start);
			
			update.executeUpdate();
		} catch (SQLException e) {
			throw new PersistenceException(e);
		}
	}

	public List<Agenda.Record> getRecordLog() throws PersistenceException {
		List<Agenda.Record> log = new ArrayList<Agenda.Record>();
		
		Agenda.Record root = getLastRecord("level = 0");
		if (root.state == Agenda.State.COMPLETE 
				|| root.state == Agenda.State.SKIPPED) {
			return log; //terminated
		}
		
		String sql = "select * from "+table.getSQLName()+" where start >= "+root.start+" order by start asc";
		
		try {
			int skip = 0;
			ResultSet rs = access.executeQuery("getRecordLog", sql);
			while (rs.next()) {
				int start = rs.getInt("start");				
				if (start<=skip) continue;
				
				int state = rs.getInt("state");
				if (state != Agenda.State.STARTED.ordinal()) {
					skip = rs.getInt("end");
				}
				
				Agenda.Record rec = new Agenda.Record(
						asInteger((Number)rs.getObject("level")), 
						asInteger((Number)rs.getObject("start")), 
						asInteger((Number)rs.getObject("end")), 
						timestampFormatter.parse((String)rs.getString("timestamp")).getTime(), 
						asLong((Number)rs.getObject("duration")), 
						(String)rs.getString("context"), 
						(String)rs.getString("task"), 
						Agenda.State.valueOf(rs.getInt("state")), 
						asBoolean(rs.getObject("complex")), 
						Agenda.decodeParameters(rs.getString("parameters"), false),
						(String)rs.getString("result") 
						); 
				
				log.add(rec);
			}
			
			return log;
		} catch (SQLException e) {
			throw new PersistenceException(e);
		} catch (ParseException e) {
			throw new PersistenceException(e);
		}
	}

	public void prepare() throws PersistenceException {
		try {
			if (!access.tableExists(table.getName())) {
				access.createTable(table.getName(), true);
			}
		} catch (SQLException e) {
			throw new PersistenceException(e);
		}
	}


}
