package de.brightbyte.db;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.Future;

import de.brightbyte.application.Agenda;
import de.brightbyte.util.PersistenceException;
import de.brightbyte.util.Processor;

public interface DatabaseAccess {

	public String getTablePrefix();
	public String quoteString(String s);
	public String encodeValue(Object v);
	public String encodeSet(Object[] v);
	public String encodeSet(Collection<?> v);
	public String quoteName(String s);
	public String quoteQualifiedName(String name);
	
	public Object fetchValue(String table, String field, Object value, String vfield) throws SQLException;
	public Map<String, Object> fetchRecord(String table, String field, Object value) throws SQLException;
	
	public ResultSet executeBigQuery(String name, String sql) throws SQLException;
	public ResultSet executeQuery(String name, String sql) throws SQLException;
	public Object executeSingleValueQuery(String name, String sql) throws SQLException;
	public Map<String, Object> executeSingleRowQuery(String name, String string) throws SQLException;

	public void warn(String msg);
	public void info(String msg);
	public void debug(String msg);
	public void trace(String msg);
	public void error(String string, Throwable e);

	public Future<Object> executeLater(DatabaseTask task) throws SQLException;
	public Object executeSynced(DatabaseTask task) throws SQLException;

	public int getBufferScale();
	public Connection getConnection() throws SQLException;
	
	public String getSQLTableName(String name);
	public String getSQLTableName(String name, boolean unsafe); //XXX: use with care! deprecate?!
	public DatabaseTable getTable(String name);
	
	public PreparedStatement prepareStatement(String sql) throws SQLException;
	public void flush() throws SQLException, InterruptedException;
	public void enableKeys() throws SQLException;
	public void disableKeys() throws SQLException;
	public void finish() throws SQLException, InterruptedException;
	public void truncateTable(String name, boolean opt) throws SQLException;
	public void optimizeTables()  throws SQLException;
	public void optimizeIndexes()  throws SQLException;
	public int executeUpdate(String name, String sql) throws SQLException;
	public int getTableSize(String name) throws SQLException;
	public void createTable(String t, boolean opt) throws SQLException;;
	public void dropTable(String t, boolean opt) throws SQLException;
	public void close() throws SQLException;
	public void open() throws SQLException;
	public boolean tableExists(String name) throws SQLException;
	
	public String mangleType(String type);
	public int getMaxStatementSize() throws SQLException; //bytes!
		
	public static class ChunkingInfo {
		public final int minId;
		public final int maxId;
		public final int chunkStep;
		
		public ChunkingInfo(final int minId, final int maxId, final int chunkStep) {
			this.minId = minId;
			this.maxId = maxId;
			this.chunkStep = chunkStep;
		}
	}
	
	public ChunkingInfo getChunkingInfo(ChunkedQuery query, int chunkSize) throws SQLException;
	
	public int executeChunkedUpdate(ChunkedQuery query, int chunk, Agenda agenda) throws SQLException, PersistenceException;
	public int executeChunkedQuery(ChunkedQuery query, int chunk, Agenda agenda, Processor<ResultSet> processor) throws SQLException, PersistenceException;
	
	public interface ChunkedQuery {
		public ResultSet executeQuery(int chunk, long first, long end) throws Exception;
		public int executeUpdate(int chunk, long first, long end) throws Exception;

		public String getContext();
		public String getName();
		
		public DatabaseTable getChunkTable();
		public String getChunkField();
	}
	
	public static abstract class AbstractChunkedQuery implements ChunkedQuery {
		protected DatabaseAccess database; 
		protected String context;
		protected String name;
		protected DatabaseTable table;
		protected String field;
		
		public AbstractChunkedQuery(DatabaseAccess database, String context, String name, DatabaseTable table, String field) {
			this.database = database;
			this.context = context;
			this.name = name;
			this.table = table;
			this.field = field;
		}

		public String getContext() {
			return context;
		}

		public String getName() {
			return name;
		}

		public String getChunkField() {
			return field;
		}

		public DatabaseTable getChunkTable() {
			return table;
		}

		public int executeUpdate(int chunk, long first, long end) throws Exception {
			String q = getSQL(first, end);
			
			return database.executeUpdate(context+"::"+name+"#chunk"+chunk, q);
		}

		public ResultSet executeQuery(int chunk, long first, long end) throws Exception {
			String q = getSQL(first, end);
			
			return database.executeQuery(context+"::"+name+"#chunk"+chunk, q);
		}

		protected abstract String getSQL(long first, long end);
		
	}

	public static class SimpleChunkedQuery extends AbstractChunkedQuery {
		
		protected String sql;
		protected String where;
		protected String rest;
		
		public SimpleChunkedQuery(DatabaseAccess database, String context, String name, String sql, String where, String rest, DatabaseTable table, String field) {
			super(database, context, name, table, field);
			this.sql = sql;
			this.where = where;
			this.rest = rest;
		}
		
		@Override
		public String getSQL(long first, long end) {
			String chunkSql = sql;
			
			chunkSql += " WHERE ";
			chunkSql += " ( "+field+" >= "+first+" AND "+field+" < "+end+" ) ";
			
			if (where!=null) chunkSql += " AND ( " + where + " ) ";
			if (rest!=null) chunkSql += " " + rest;
			
			return chunkSql;
		}
	}
	
}
